<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Http\Controllers\AdminDashboardController;
use App\Http\Controllers\TutorDashboardController;
use App\Http\Controllers\UserDashboardController;
use App\Http\Controllers\PageController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "web" middleware group. Make something great!
|
*/

// Public Pages
Route::get('/', [PageController::class, 'home'])->name('home');
Route::get('/about', [PageController::class, 'about'])->name('about');
Route::get('/courses', [PageController::class, 'courses'])->name('courses');
Route::get('/contact', [PageController::class, 'contact'])->name('contact');
Route::post('/contact', [PageController::class, 'submitContact'])->name('contact.submit');
Route::get('/terms', [PageController::class, 'terms'])->name('terms');
Route::get('/privacy', [PageController::class, 'privacy'])->name('privacy');

// Public API - Site Settings
Route::get('/api/settings', [App\Http\Controllers\Admin\SiteSettingsController::class, 'getPublic'])->name('api.settings.public');

// Public API - Active Languages
Route::get('/api/languages/active', [App\Http\Controllers\Admin\LanguageController::class, 'getActive'])->name('api.languages.active');

// Test route for access code status
Route::get('/test-access-code-api', function () {
    $controller = new App\Http\Controllers\Api\SettingsController();
    $response = $controller->getAccessCodeStatus();
    return response()->json([
        'test' => 'Direct controller call',
        'result' => $response->getData()
    ]);
});

// Course management routes
Route::middleware(['auth:sanctum', 'verified'])->group(function () {
    Route::resource('courses', App\Http\Controllers\CourseController::class)->except(['index', 'show']);
    Route::get('/courses/{course}', [App\Http\Controllers\CourseController::class, 'show'])->name('courses.show');

    // Lesson management routes
    Route::resource('lessons', App\Http\Controllers\LessonController::class)->except(['show']);
    Route::get('/lessons/{lesson}', [App\Http\Controllers\LessonController::class, 'show'])->name('lessons.show');

    // Topic management routes
    Route::resource('topics', App\Http\Controllers\TopicController::class)->except(['show']);
    Route::get('/topics/{topic}', [App\Http\Controllers\TopicController::class, 'show'])->name('topics.show');
    Route::post('/topics/{topic}/split', [App\Http\Controllers\TopicController::class, 'split'])->name('topics.split');
    Route::get('/topics/{topic}/preview-split', [App\Http\Controllers\TopicController::class, 'previewSplit'])->name('topics.preview-split');

    // Quiz question management routes
    Route::resource('quiz-questions', App\Http\Controllers\QuizQuestionController::class)->except(['show']);
    Route::get('/quiz-questions/{question}', [App\Http\Controllers\QuizQuestionController::class, 'show'])->name('quiz-questions.show');
});
Route::middleware(['auth'])->group(function () {
    // Main Dashboard - redirects based on role or shows student dashboard
    Route::get('/dashboard', [UserDashboardController::class, 'index'])
        ->name('dashboard');

    // Student Course Learning Routes
    Route::prefix('learn')->name('learn.')->group(function () {
        Route::get('/course/{course}', [App\Http\Controllers\StudentCourseController::class, 'show'])->name('course');
        Route::get('/course/{course}/lesson/{lesson}/topic/{topic}', [App\Http\Controllers\StudentCourseController::class, 'viewTopic'])->name('topic');
        Route::post('/topic/{topic}/complete', [App\Http\Controllers\StudentCourseController::class, 'markComplete'])->name('topic.complete');
        Route::post('/topic/{topic}/incomplete', [App\Http\Controllers\StudentCourseController::class, 'markIncomplete'])->name('topic.incomplete');
        Route::post('/topic/{topic}/quiz', [App\Http\Controllers\StudentCourseController::class, 'submitQuiz'])->name('topic.quiz');
        Route::post('/lesson/{lesson}/quiz', [App\Http\Controllers\StudentCourseController::class, 'submitLessonQuiz'])->name('lesson.quiz');
    });

    // API Routes for lesson completion check
    Route::get('/api/lesson/{lesson}/check-completion', [App\Http\Controllers\StudentCourseController::class, 'checkLessonCompletion'])->name('api.lesson.check-completion');

    // Admin survey API (exposed on /api/... so dashboard JS can call without "/admin" prefix)
    // These still require web session auth and will return JSON (controller enforces authorization)
    Route::middleware(['auth'])->group(function () {
        Route::get('/api/courses/{courseId}/survey', [\App\Http\Controllers\SurveyController::class, 'getCourseSurvey']);
        Route::post('/api/courses/{courseId}/survey/questions', [\App\Http\Controllers\SurveyController::class, 'manageQuestions']);
        Route::get('/api/courses/{courseId}/survey/stats', [\App\Http\Controllers\SurveyController::class, 'getStats']);
    });

    // Student Browse & Progress Routes
    Route::prefix('student')->name('student.')->group(function () {
        Route::get('/browse', [App\Http\Controllers\StudentBrowseController::class, 'browse'])->name('browse');
        Route::get('/schedule', [App\Http\Controllers\StudentBrowseController::class, 'schedule'])->name('schedule');
        Route::get('/progress', [App\Http\Controllers\StudentBrowseController::class, 'progress'])->name('progress');
        Route::get('/leaderboard', [App\Http\Controllers\StudentBrowseController::class, 'leaderboard'])->name('leaderboard');
        
        // Enrollment Routes
        Route::post('/enroll/{course}', [App\Http\Controllers\StudentEnrollmentController::class, 'enroll'])->name('enroll');
        Route::post('/drop/{course}', [App\Http\Controllers\StudentEnrollmentController::class, 'drop'])->name('drop');
        Route::get('/enrollment-status/{course}', [App\Http\Controllers\StudentEnrollmentController::class, 'status'])->name('enrollment.status');
        Route::post('/course/{course}/complete', [App\Http\Controllers\StudentEnrollmentController::class, 'markCompleted'])->name('course.complete');
        Route::get('/my-courses', [App\Http\Controllers\StudentEnrollmentController::class, 'myCourses'])->name('my-courses');
        
        // Certificate Routes
        Route::get('/certificates', [App\Http\Controllers\StudentCertificateController::class, 'index'])->name('certificates');
        Route::post('/certificate/generate/{course}', [App\Http\Controllers\StudentCertificateController::class, 'generate'])->name('certificate.generate');
    Route::post('/certificate/generate/lesson/{lesson}', [App\Http\Controllers\StudentCertificateController::class, 'generateLesson'])->name('certificate.generate.lesson');
        Route::get('/certificate/{certificate}/download', [App\Http\Controllers\StudentCertificateController::class, 'download'])->name('certificate.download');
        Route::get('/certificate/{certificate}/view', [App\Http\Controllers\StudentCertificateController::class, 'view'])->name('certificate.view');
        
        // Profile & Settings Routes
        Route::get('/profile', [App\Http\Controllers\StudentProfileController::class, 'index'])->name('profile');
        Route::post('/profile/update', [App\Http\Controllers\StudentProfileController::class, 'update'])->name('profile.update');
        Route::post('/profile/avatar', [App\Http\Controllers\StudentProfileController::class, 'updateAvatar'])->name('profile.avatar');
        Route::post('/profile/password', [App\Http\Controllers\StudentProfileController::class, 'changePassword'])->name('profile.password');
        Route::get('/settings', [App\Http\Controllers\StudentProfileController::class, 'settings'])->name('settings');
        Route::post('/settings/update', [App\Http\Controllers\StudentProfileController::class, 'updateSettings'])->name('settings.update');
        Route::post('/privacy/update', [App\Http\Controllers\StudentProfileController::class, 'updatePrivacy'])->name('privacy.update');
    });

    // Payment Routes
    Route::prefix('payment')->name('payment.')->group(function () {
        Route::get('/checkout/{course}', [App\Http\Controllers\PaymentController::class, 'checkout'])->name('checkout');
        Route::post('/process/stripe/{course}', [App\Http\Controllers\PaymentController::class, 'processStripe'])->name('process.stripe');
        Route::post('/process/paypal/{course}', [App\Http\Controllers\PaymentController::class, 'processPayPal'])->name('process.paypal');
        Route::post('/process/paystack/{course}', [App\Http\Controllers\PaymentController::class, 'processPaystack'])->name('process.paystack');
        Route::get('/success/{course}', [App\Http\Controllers\PaymentController::class, 'success'])->name('success');
        Route::get('/cancel/{course?}', [App\Http\Controllers\PaymentController::class, 'cancel'])->name('cancel');
        Route::get('/paypal/success/{course}', [App\Http\Controllers\PaymentController::class, 'paypalSuccess'])->name('paypal.success');
        Route::get('/paystack/callback', [App\Http\Controllers\PaymentController::class, 'paystackCallback'])->name('paystack.callback');
        Route::get('/history', [App\Http\Controllers\PaymentController::class, 'history'])->name('history');
    });

    Route::get('/dashboard/tutor', [TutorDashboardController::class, 'index'])
        ->middleware('role:tutor')
        ->name('tutor.dashboard');
    
    // Tutor API Testing Page
    Route::get('/dashboard/tutor/api-test', function() {
        return view('dashboard.tutor-api-test');
    })->middleware('role:tutor')->name('tutor.api.test');

    Route::get('/dashboard/user', [UserDashboardController::class, 'index'])
        ->middleware('role:user')
        ->name('user.dashboard');

    // Tutor Management Routes
    Route::prefix('tutor')->middleware(['auth', 'role:tutor'])->group(function () {
        // Course Management
        Route::get('/courses', [App\Http\Controllers\TutorController::class, 'getCourses'])->name('tutor.courses');
        Route::get('/courses/{courseId}', [App\Http\Controllers\TutorController::class, 'getCourse'])->name('tutor.courses.show');
        Route::post('/courses', [App\Http\Controllers\TutorController::class, 'storeCourse'])->name('tutor.courses.store');
        Route::put('/courses/{courseId}', [App\Http\Controllers\TutorController::class, 'updateCourse'])->name('tutor.courses.update');
        Route::delete('/courses/{courseId}', [App\Http\Controllers\TutorController::class, 'destroyCourse'])->name('tutor.courses.destroy');
        Route::get('/courses/{courseId}/stats', [App\Http\Controllers\TutorController::class, 'getCourseStats'])->name('tutor.courses.stats');
        Route::get('/courses/{courseId}/prerequisites', [App\Http\Controllers\TutorController::class, 'getCoursePrerequisites'])->name('tutor.courses.prerequisites');
        Route::get('/available-prerequisites/{excludeCourseId?}', [App\Http\Controllers\TutorController::class, 'getAvailablePrerequisites'])->name('tutor.courses.available-prerequisites');
        
        // Certificate Templates
        Route::get('/certificate-templates', [App\Http\Controllers\TutorController::class, 'getCertificateTemplates'])->name('tutor.certificate-templates');
        
        // Course Management Routes (Lessons & Topics)
        Route::get('/courses/{course}/lessons', [App\Http\Controllers\TutorController::class, 'getLessons'])->name('tutor.lessons.index');
        Route::get('/lessons/{lesson}', [App\Http\Controllers\TutorController::class, 'getLesson'])->name('tutor.lessons.show');
        Route::post('/lessons', [App\Http\Controllers\TutorController::class, 'storeLesson'])->name('tutor.lessons.store');
        Route::put('/lessons/{lesson}', [App\Http\Controllers\TutorController::class, 'updateLesson'])->name('tutor.lessons.update');
        Route::delete('/lessons/{lesson}', [App\Http\Controllers\TutorController::class, 'destroyLesson'])->name('tutor.lessons.destroy');
        Route::post('/lessons/reorder', [App\Http\Controllers\TutorController::class, 'reorderLessons'])->name('tutor.lessons.reorder');
        
        Route::get('/lessons/{lesson}/topics', [App\Http\Controllers\TutorController::class, 'getTopics'])->name('tutor.topics.index');
        Route::get('/topics/{topic}', [App\Http\Controllers\TutorController::class, 'getTopic'])->name('tutor.topics.show');
        Route::post('/topics', [App\Http\Controllers\TutorController::class, 'storeTopic'])->name('tutor.topics.store');
        Route::put('/topics/{topic}', [App\Http\Controllers\TutorController::class, 'updateTopic'])->name('tutor.topics.update');
        Route::delete('/topics/{topic}', [App\Http\Controllers\TutorController::class, 'destroyTopic'])->name('tutor.topics.destroy');
        Route::post('/topics/reorder', [App\Http\Controllers\TutorController::class, 'reorderTopics'])->name('tutor.topics.reorder');
        Route::post('/topics/{topic}/split', [App\Http\Controllers\TutorController::class, 'splitTopic'])->name('tutor.topics.split');
        Route::get('/topics/{topic}/preview-split', [App\Http\Controllers\TutorController::class, 'previewTopicSplit'])->name('tutor.topics.preview-split');
        
        // Course Enhancement Routes
        Route::get('/courses/{courseId}/students', [App\Http\Controllers\TutorController::class, 'getCourseStudents'])->name('tutor.courses.students');
        Route::post('/courses/{courseId}/duplicate', [App\Http\Controllers\TutorController::class, 'duplicateCourse'])->name('tutor.courses.duplicate');
        
        // Quiz Question Management Routes
        Route::get('/lessons/{lesson}/quiz-questions', [App\Http\Controllers\TutorController::class, 'getQuizQuestions'])->name('tutor.quiz-questions.index');
        Route::post('/quiz-questions', [App\Http\Controllers\TutorController::class, 'storeQuizQuestion'])->name('tutor.quiz-questions.store');
        Route::get('/quiz-questions/{quizQuestion}', [App\Http\Controllers\TutorController::class, 'getQuizQuestion'])->name('tutor.quiz-questions.show');
        Route::put('/quiz-questions/{quizQuestion}', [App\Http\Controllers\TutorController::class, 'updateQuizQuestion'])->name('tutor.quiz-questions.update');
        Route::delete('/quiz-questions/{quizQuestion}', [App\Http\Controllers\TutorController::class, 'destroyQuizQuestion'])->name('tutor.quiz-questions.destroy');
        
        // Student Management
        Route::get('/students', [App\Http\Controllers\TutorController::class, 'getStudents'])->name('tutor.students');
        Route::get('/students/{id}', [App\Http\Controllers\TutorController::class, 'getStudentDetails'])->name('tutor.students.details');
        
        // Certificate Management Routes (Tutors can manage certificates for their courses)
        Route::get('/certificates-builder', function () {
            return view('admin.certificates');
        })->name('tutor.certificates.builder');
        Route::get('/certificates', [App\Http\Controllers\CertificateTemplateController::class, 'index'])->name('tutor.certificates.index');
        Route::post('/certificates', [App\Http\Controllers\CertificateTemplateController::class, 'store'])->name('tutor.certificates.store');
        Route::get('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'show'])->name('tutor.certificates.show');
    Route::get('/certificates/{id}/preview', [App\Http\Controllers\CertificateTemplateController::class, 'preview'])->name('tutor.certificates.preview');
        Route::put('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'update'])->name('tutor.certificates.update');
        Route::delete('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'destroy'])->name('tutor.certificates.destroy');
        Route::post('/certificates/{id}/duplicate', [App\Http\Controllers\CertificateTemplateController::class, 'duplicate'])->name('tutor.certificates.duplicate');
        Route::post('/certificates/assign', [App\Http\Controllers\CertificateTemplateController::class, 'assign'])->name('tutor.certificates.assign');
        Route::delete('/certificates/assignments/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'unassign'])->name('tutor.certificates.unassign');
        Route::get('/certificates/assignments/get', [App\Http\Controllers\CertificateTemplateController::class, 'getAssignment'])->name('tutor.certificates.get-assignment');
        
        // Badge Management Routes (Tutors can manage badges for their students)
        Route::get('/badges/all', [App\Http\Controllers\AdminBadgeController::class, 'index'])->name('tutor.badges.all');
        Route::post('/badges', [App\Http\Controllers\AdminBadgeController::class, 'store'])->name('tutor.badges.store');
        Route::put('/badges/{id}', [App\Http\Controllers\AdminBadgeController::class, 'update'])->name('tutor.badges.update');
        Route::post('/badges/{id}/toggle', [App\Http\Controllers\AdminBadgeController::class, 'toggleStatus'])->name('tutor.badges.toggle');
        Route::delete('/badges/{id}', [App\Http\Controllers\AdminBadgeController::class, 'destroy'])->name('tutor.badges.destroy');
        Route::get('/badges/leaderboard', [App\Http\Controllers\AdminBadgeController::class, 'leaderboard'])->name('tutor.badges.leaderboard');
        Route::get('/badges/stats', [App\Http\Controllers\AdminBadgeController::class, 'stats'])->name('tutor.badges.stats');
        Route::post('/badges/award', [App\Http\Controllers\AdminBadgeController::class, 'awardBadge'])->name('tutor.badges.award');
        Route::post('/badges/remove', [App\Http\Controllers\AdminBadgeController::class, 'removeBadge'])->name('tutor.badges.remove');
        
        // Category Management Routes (Tutors can manage categories)
        Route::get('/categories', [App\Http\Controllers\CategoryController::class, 'index'])->name('tutor.categories.index');
        Route::post('/categories', [App\Http\Controllers\CategoryController::class, 'store'])->name('tutor.categories.store');
        Route::get('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'show'])->name('tutor.categories.show');
        Route::put('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'update'])->name('tutor.categories.update');
        Route::delete('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'destroy'])->name('tutor.categories.destroy');
    });

    // Admin Dashboard Routes
    Route::prefix('admin')->middleware('role:admin')->group(function () {
        Route::get('/dashboard', [App\Http\Controllers\AdminController::class, 'dashboard'])->name('admin.panel');
        Route::get('/dashboard/data', [App\Http\Controllers\AdminController::class, 'getDashboardData'])->name('admin.panel.data');
        
        // Category Management
        Route::get('/categories', function () {
            return Inertia::render('Admin/Categories');
        })->name('admin.categories');
        Route::get('/categories/list', [App\Http\Controllers\CategoryController::class, 'index'])->name('admin.categories.list');
        Route::post('/categories', [App\Http\Controllers\CategoryController::class, 'store'])->name('admin.categories.store');
        Route::get('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'show'])->name('admin.categories.show');
        Route::put('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'update'])->name('admin.categories.update');
        Route::delete('/categories/{category}', [App\Http\Controllers\CategoryController::class, 'destroy'])->name('admin.categories.destroy');
        
        Route::get('/users', [App\Http\Controllers\AdminController::class, 'getUsers'])->name('admin.panel.users');
        Route::get('/users/{userId}', [App\Http\Controllers\AdminController::class, 'getUserDetails'])->name('admin.users.details');
    Route::post('/make-admin', [App\Http\Controllers\AdminController::class, 'makeAdmin'])->name('admin.make-admin');
    Route::post('/remove-admin', [App\Http\Controllers\AdminController::class, 'removeAdmin'])->name('admin.remove-admin');
    Route::post('/change-role', [App\Http\Controllers\AdminController::class, 'changeRole'])->name('admin.change-role');
        
        // Course Management Routes
        Route::get('/courses', [App\Http\Controllers\AdminController::class, 'getCourses'])->name('admin.courses');

        // Provide JSON endpoints for survey management (use web session auth) to support dashboard fetches when Sanctum cookies are unavailable
        Route::get('/api/courses/{courseId}/survey', [\App\Http\Controllers\SurveyController::class, 'getCourseSurvey']);
        Route::post('/api/courses/{courseId}/survey/questions', [\App\Http\Controllers\SurveyController::class, 'manageQuestions']);
        Route::get('/api/courses/{courseId}/survey/stats', [\App\Http\Controllers\SurveyController::class, 'getStats']);
        // Note: student submission still handled via API routes (requires proper auth), web route for submit can be added if needed

        Route::get('/courses/{courseId}', [App\Http\Controllers\AdminController::class, 'getCourse'])->name('admin.courses.show');
        Route::post('/courses', [App\Http\Controllers\AdminController::class, 'storeCourse'])->name('admin.courses.store');
        Route::put('/courses/{courseId}', [App\Http\Controllers\AdminController::class, 'updateCourse'])->name('admin.courses.update');
        Route::delete('/courses/{courseId}', [App\Http\Controllers\AdminController::class, 'destroyCourse'])->name('admin.courses.destroy');
        Route::get('/courses/{courseId}/stats', [App\Http\Controllers\AdminController::class, 'getCourseStats'])->name('admin.courses.stats');
        Route::get('/courses/{courseId}/prerequisites', [App\Http\Controllers\AdminController::class, 'getCoursePrerequisites'])->name('admin.courses.prerequisites');
        Route::get('/available-prerequisites/{excludeCourseId?}', [App\Http\Controllers\AdminController::class, 'getAvailablePrerequisites'])->name('admin.courses.available-prerequisites');
        
        // Certificate Templates
        Route::get('/certificate-templates', [App\Http\Controllers\AdminController::class, 'getCertificateTemplates'])->name('admin.certificate-templates');
        
        // Course Management Routes (Lessons & Topics)
        Route::get('/courses/{course}/lessons', [App\Http\Controllers\AdminController::class, 'getLessons'])->name('admin.lessons.index');
        Route::get('/lessons/{lesson}', [App\Http\Controllers\AdminController::class, 'getLesson'])->name('admin.lessons.show');
        Route::post('/lessons', [App\Http\Controllers\AdminController::class, 'storeLesson'])->name('admin.lessons.store');
        Route::put('/lessons/{lesson}', [App\Http\Controllers\AdminController::class, 'updateLesson'])->name('admin.lessons.update');
        Route::delete('/lessons/{lesson}', [App\Http\Controllers\AdminController::class, 'destroyLesson'])->name('admin.lessons.destroy');
        Route::post('/lessons/reorder', [App\Http\Controllers\AdminController::class, 'reorderLessons'])->name('admin.lessons.reorder');
        
        Route::get('/lessons/{lesson}/topics', [App\Http\Controllers\AdminController::class, 'getTopics'])->name('admin.topics.index');
        Route::get('/topics/{topic}', [App\Http\Controllers\AdminController::class, 'getTopic'])->name('admin.topics.show');
        Route::post('/topics', [App\Http\Controllers\AdminController::class, 'storeTopic'])->name('admin.topics.store');
        Route::put('/topics/{topic}', [App\Http\Controllers\AdminController::class, 'updateTopic'])->name('admin.topics.update');
        Route::delete('/topics/{topic}', [App\Http\Controllers\AdminController::class, 'destroyTopic'])->name('admin.topics.destroy');
        Route::post('/topics/reorder', [App\Http\Controllers\AdminController::class, 'reorderTopics'])->name('admin.topics.reorder');
        Route::post('/topics/{topic}/split', [App\Http\Controllers\AdminController::class, 'splitTopic'])->name('admin.topics.split');
        Route::get('/topics/{topic}/preview-split', [App\Http\Controllers\AdminController::class, 'previewTopicSplit'])->name('admin.topics.preview-split');
        
        // Quiz Question Management Routes
        Route::get('/lessons/{lesson}/quiz-questions', [App\Http\Controllers\QuizQuestionController::class, 'index'])->name('admin.quiz-questions.index');
        Route::post('/quiz-questions', [App\Http\Controllers\QuizQuestionController::class, 'store'])->name('admin.quiz-questions.store');
        Route::get('/quiz-questions/{quizQuestion}', [App\Http\Controllers\QuizQuestionController::class, 'show'])->name('admin.quiz-questions.show');
        Route::put('/quiz-questions/{quizQuestion}', [App\Http\Controllers\QuizQuestionController::class, 'update'])->name('admin.quiz-questions.update');
        Route::delete('/quiz-questions/{quizQuestion}', [App\Http\Controllers\QuizQuestionController::class, 'destroy'])->name('admin.quiz-questions.destroy');
        
        // Access Code Management Routes
        Route::get('/settings/access-code-status', [App\Http\Controllers\Api\SettingsController::class, 'getAccessCodeStatus'])->name('admin.access-code.status');
        Route::post('/settings/access-code-toggle', [App\Http\Controllers\Api\SettingsController::class, 'toggleAccessCode'])->name('admin.access-code.toggle');
        Route::post('/settings/generate-access-codes', [App\Http\Controllers\Api\SettingsController::class, 'generateAccessCodes'])->name('admin.access-code.generate');
        Route::get('/settings/access-codes', [App\Http\Controllers\Api\SettingsController::class, 'listAccessCodes'])->name('admin.access-code.list');
        Route::post('/settings/clear-expired-codes', [App\Http\Controllers\Api\SettingsController::class, 'clearExpiredAccessCodes'])->name('admin.access-code.clear-expired');
        Route::delete('/settings/access-codes/{id}', [App\Http\Controllers\Api\SettingsController::class, 'deleteAccessCode'])->name('admin.access-code.delete');
        
    // Roles & Permissions API (basic CRUD for admin UI)
    Route::get('/roles', [App\Http\Controllers\Api\RoleController::class, 'index'])->name('admin.roles.index');
    Route::post('/roles', [App\Http\Controllers\Api\RoleController::class, 'store'])->name('admin.roles.store');
    Route::get('/roles/{id}', [App\Http\Controllers\Api\RoleController::class, 'show'])->name('admin.roles.show');
    Route::put('/roles/{id}', [App\Http\Controllers\Api\RoleController::class, 'update'])->name('admin.roles.update');
    Route::delete('/roles/{id}', [App\Http\Controllers\Api\RoleController::class, 'destroy'])->name('admin.roles.destroy');

    Route::get('/permissions', [App\Http\Controllers\Api\PermissionController::class, 'index'])->name('admin.permissions.index');
    Route::post('/permissions', [App\Http\Controllers\Api\PermissionController::class, 'store'])->name('admin.permissions.store');
    Route::get('/permissions/{id}', [App\Http\Controllers\Api\PermissionController::class, 'show'])->name('admin.permissions.show');
    Route::put('/permissions/{id}', [App\Http\Controllers\Api\PermissionController::class, 'update'])->name('admin.permissions.update');
    Route::delete('/permissions/{id}', [App\Http\Controllers\Api\PermissionController::class, 'destroy'])->name('admin.permissions.destroy');
        
        // Custom Pages Management Routes
        Route::get('/pages', [App\Http\Controllers\Admin\CustomPageController::class, 'index'])->name('admin.pages.index');
        Route::post('/pages', [App\Http\Controllers\Admin\CustomPageController::class, 'store'])->name('admin.pages.store');
        Route::get('/pages/{id}', [App\Http\Controllers\Admin\CustomPageController::class, 'show'])->name('admin.pages.show');
        Route::put('/pages/{id}', [App\Http\Controllers\Admin\CustomPageController::class, 'update'])->name('admin.pages.update');
        Route::delete('/pages/{id}', [App\Http\Controllers\Admin\CustomPageController::class, 'destroy'])->name('admin.pages.destroy');
        Route::post('/pages/{id}/toggle-publish', [App\Http\Controllers\Admin\CustomPageController::class, 'togglePublish'])->name('admin.pages.toggle-publish');
        Route::post('/pages/{id}/duplicate', [App\Http\Controllers\Admin\CustomPageController::class, 'duplicate'])->name('admin.pages.duplicate');
        
        // Site Settings Management Routes
        Route::get('/settings', [App\Http\Controllers\Admin\SiteSettingsController::class, 'index'])->name('admin.settings.index');
        Route::put('/settings/{key}', [App\Http\Controllers\Admin\SiteSettingsController::class, 'update'])
            ->where('key', '^(?!notifications$).+')
            ->name('admin.settings.update');
        Route::post('/settings/bulk', [App\Http\Controllers\Admin\SiteSettingsController::class, 'updateBulk'])->name('admin.settings.bulk');
        Route::post('/settings/upload-image', [App\Http\Controllers\Admin\SiteSettingsController::class, 'uploadImage'])->name('admin.settings.upload-image');
        Route::post('/settings/reset', [App\Http\Controllers\Admin\SiteSettingsController::class, 'reset'])->name('admin.settings.reset');
        
        // Notification Settings Management Routes
        Route::get('/settings/notifications', [App\Http\Controllers\Admin\NotificationSettingsController::class, 'index'])->name('admin.settings.notifications');
        Route::put('/settings/notifications', [App\Http\Controllers\Admin\NotificationSettingsController::class, 'update'])->name('admin.settings.notifications.update');
        Route::post('/settings/notifications/test/email', [App\Http\Controllers\Admin\NotificationSettingsController::class, 'testEmail'])->name('admin.settings.notifications.test.email');
        Route::post('/settings/notifications/test/push', [App\Http\Controllers\Admin\NotificationSettingsController::class, 'testPush'])->name('admin.settings.notifications.test.push');
        Route::post('/settings/notifications/test/sms', [App\Http\Controllers\Admin\NotificationSettingsController::class, 'testSms'])->name('admin.settings.notifications.test.sms');
        
        // Payment Settings Management Routes
        Route::get('/payment-settings', [App\Http\Controllers\Admin\PaymentSettingsController::class, 'index'])->name('admin.payment-settings');
        Route::get('/payment-settings/{gateway}', [App\Http\Controllers\Admin\PaymentSettingsController::class, 'show'])->name('admin.payment-settings.show');
        Route::put('/payment-settings/{gateway}', [App\Http\Controllers\Admin\PaymentSettingsController::class, 'update'])->name('admin.payment-settings.update');
        Route::post('/payment-settings/{gateway}/toggle', [App\Http\Controllers\Admin\PaymentSettingsController::class, 'toggle'])->name('admin.payment-settings.toggle');
        
        // Certificate Template Management Routes
        Route::get('/certificates-builder', function () {
            return view('admin.certificates');
        })->name('admin.certificates.builder');
        Route::get('/certificates', [App\Http\Controllers\CertificateTemplateController::class, 'index'])->name('admin.certificates.index');
        Route::post('/certificates', [App\Http\Controllers\CertificateTemplateController::class, 'store'])->name('admin.certificates.store');
        Route::get('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'show'])->name('admin.certificates.show');
    Route::get('/certificates/{id}/preview', [App\Http\Controllers\CertificateTemplateController::class, 'preview'])->name('admin.certificates.preview');
        Route::put('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'update'])->name('admin.certificates.update');
        Route::delete('/certificates/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'destroy'])->name('admin.certificates.destroy');
        Route::post('/certificates/{id}/duplicate', [App\Http\Controllers\CertificateTemplateController::class, 'duplicate'])->name('admin.certificates.duplicate');
        Route::post('/certificates/assign', [App\Http\Controllers\CertificateTemplateController::class, 'assign'])->name('admin.certificates.assign');
        Route::delete('/certificates/assignments/{id}', [App\Http\Controllers\CertificateTemplateController::class, 'unassign'])->name('admin.certificates.unassign');
        Route::get('/certificates/assignments/get', [App\Http\Controllers\CertificateTemplateController::class, 'getAssignment'])->name('admin.certificates.get-assignment');
        
        // Badge Management Routes (Admin)
        Route::get('/badges/all', [App\Http\Controllers\AdminBadgeController::class, 'index'])->name('admin.badges.all');
        Route::post('/badges', [App\Http\Controllers\AdminBadgeController::class, 'store'])->name('admin.badges.store');
        Route::put('/badges/{id}', [App\Http\Controllers\AdminBadgeController::class, 'update'])->name('admin.badges.update');
        Route::post('/badges/{id}/toggle', [App\Http\Controllers\AdminBadgeController::class, 'toggleStatus'])->name('admin.badges.toggle');
        Route::delete('/badges/{id}', [App\Http\Controllers\AdminBadgeController::class, 'destroy'])->name('admin.badges.destroy');
        Route::get('/badges/leaderboard', [App\Http\Controllers\AdminBadgeController::class, 'leaderboard'])->name('admin.badges.leaderboard');
        Route::get('/badges/stats', [App\Http\Controllers\AdminBadgeController::class, 'stats'])->name('admin.badges.stats');
        Route::post('/badges/award', [App\Http\Controllers\AdminBadgeController::class, 'awardBadge'])->name('admin.badges.award');
        Route::post('/badges/remove', [App\Http\Controllers\AdminBadgeController::class, 'removeBadge'])->name('admin.badges.remove');
        
        // Penalty Management Routes (Admin)
        Route::get('/penalties/rules', [App\Http\Controllers\AdminPenaltyController::class, 'getRules'])->name('admin.penalties.rules');
        Route::post('/penalties/rules', [App\Http\Controllers\AdminPenaltyController::class, 'store'])->name('admin.penalties.rules.store');
        Route::put('/penalties/rules/{id}', [App\Http\Controllers\AdminPenaltyController::class, 'update'])->name('admin.penalties.rules.update');
        Route::post('/penalties/rules/{id}/toggle', [App\Http\Controllers\AdminPenaltyController::class, 'toggleRule'])->name('admin.penalties.rules.toggle');
        Route::delete('/penalties/rules/{id}', [App\Http\Controllers\AdminPenaltyController::class, 'destroy'])->name('admin.penalties.rules.destroy');
        Route::get('/penalties/cards', [App\Http\Controllers\AdminPenaltyController::class, 'getCards'])->name('admin.penalties.cards');
        Route::get('/penalties/cards/{id}', [App\Http\Controllers\AdminPenaltyController::class, 'getCardDetails'])->name('admin.penalties.cards.details');
        Route::post('/penalties/cards/issue', [App\Http\Controllers\AdminPenaltyController::class, 'issueCard'])->name('admin.penalties.cards.issue');
        Route::post('/penalties/cards/{id}/revoke', [App\Http\Controllers\AdminPenaltyController::class, 'revokeCard'])->name('admin.penalties.cards.revoke');
        Route::get('/penalties/stats', [App\Http\Controllers\AdminPenaltyController::class, 'getStats'])->name('admin.penalties.stats');
        Route::post('/penalties/check', [App\Http\Controllers\AdminPenaltyController::class, 'runCheck'])->name('admin.penalties.check');
        
        // Language Management Routes
        Route::get('/languages', [App\Http\Controllers\Admin\LanguageController::class, 'index'])->name('admin.languages.index');
        Route::post('/languages', [App\Http\Controllers\Admin\LanguageController::class, 'store'])->name('admin.languages.store');
        Route::get('/languages/{id}', [App\Http\Controllers\Admin\LanguageController::class, 'show'])->name('admin.languages.show');
        Route::put('/languages/{id}', [App\Http\Controllers\Admin\LanguageController::class, 'update'])->name('admin.languages.update');
        Route::delete('/languages/{id}', [App\Http\Controllers\Admin\LanguageController::class, 'destroy'])->name('admin.languages.destroy');
        Route::post('/languages/{id}/toggle', [App\Http\Controllers\Admin\LanguageController::class, 'toggle'])->name('admin.languages.toggle');
        Route::post('/languages/reorder', [App\Http\Controllers\Admin\LanguageController::class, 'reorder'])->name('admin.languages.reorder');
    });
});

// Public certificate verification routes
Route::get('/verify-certificate/{code}', [App\Http\Controllers\CertificateVerificationController::class, 'verify'])->name('certificate.verify');
Route::get('/api/verify-certificate/{code}', [App\Http\Controllers\CertificateVerificationController::class, 'verifyApi'])->name('certificate.verify.api');

// Test certificate generation routes (for development)
Route::get('/test/generate-certificate', [App\Http\Controllers\TestCertificateController::class, 'generateTestCertificate'])->name('test.generate-certificate');
Route::get('/test/view-certificate/{id}', [App\Http\Controllers\TestCertificateController::class, 'viewCertificate'])->name('test.view-certificate');

// Badge routes (requires authentication)
Route::middleware(['auth'])->group(function () {
    Route::get('/my-badges', function () {
        return view('badges.index');
    })->name('badges.page');
    Route::get('/badges', [App\Http\Controllers\BadgeController::class, 'index'])->name('badges.index');
    Route::get('/badges/earned', [App\Http\Controllers\BadgeController::class, 'earned'])->name('badges.earned');
    Route::get('/badges/stats', [App\Http\Controllers\BadgeController::class, 'stats'])->name('badges.stats');
    
    // Debug route for progress
    Route::get('/debug/progress', function () {
        $user = Auth::user();
        $html = "<h1>Progress Debug for User: {$user->name} (ID: {$user->id})</h1>";
        
        $enrollments = \App\Models\CourseEnrollment::where('user_id', $user->id)
            ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
            ->with('course')
            ->get();
        
        $html .= "<h2>Active Enrollments: " . $enrollments->count() . "</h2>";
        
        foreach ($enrollments as $enrollment) {
            $html .= "<div style='border: 1px solid #ccc; padding: 15px; margin: 10px 0;'>";
            $html .= "<h3>{$enrollment->course->name}</h3>";
            $html .= "<p><strong>Course ID:</strong> {$enrollment->course_id}</p>";
            $html .= "<p><strong>Status:</strong> {$enrollment->status}</p>";
            $html .= "<p><strong>Stored Progress:</strong> {$enrollment->progress}%</p>";
            
            // Calculate actual progress
            $course = \App\Models\Course::with(['lessons.topics'])->find($enrollment->course_id);
            $totalTopics = 0;
            $completedTopics = 0;
            
            foreach ($course->lessons as $lesson) {
                foreach ($lesson->topics as $topic) {
                    if ($topic->is_published) {
                        $totalTopics++;
                        $progress = \App\Models\UserProgress::where('user_id', $user->id)
                            ->where('topic_id', $topic->id)
                            ->first();
                        
                        if ($progress && $progress->completed) {
                            $completedTopics++;
                            $html .= "<p style='margin-left: 20px; color: green;'>✓ Topic: {$topic->title}</p>";
                        } else {
                            $html .= "<p style='margin-left: 20px; color: #999;'>○ Topic: {$topic->title}</p>";
                        }
                    }
                }
            }
            
            $actualProgress = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
            $html .= "<p><strong>Calculated Progress:</strong> {$actualProgress}% ({$completedTopics} / {$totalTopics} topics)</p>";
            $html .= "<p><a href='/courses/{$enrollment->course_id}' target='_blank' style='background: #4f46e5; color: white; padding: 8px 16px; text-decoration: none; border-radius: 4px; display: inline-block;'>View Course</a></p>";
            $html .= "</div>";
        }
        
        return $html;
    })->name('debug.progress');
});

// Auth routes must be loaded before catch-all route
require __DIR__.'/auth.php';

// Custom page route (must be last to avoid conflicts)
Route::get('/{slug}', [PageController::class, 'show'])->name('page.show');