<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=no">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>{{ $topic->title }} - {{ $course->name }}</title>
    
    @php
        $brand = \App\Support\BrandSettings::data();
        $faviconUrl = $brand['favicon_url'] ?? null;
    @endphp
    
    @if($faviconUrl)
        <link rel="icon" type="image/png" href="{{ $faviconUrl }}">
        <link rel="shortcut icon" type="image/png" href="{{ $faviconUrl }}">
    @endif
    
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800;900&display=swap" rel="stylesheet">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            -webkit-tap-highlight-color: transparent;
        }

        :root {
            --primary: #667eea;
            --primary-dark: #5568d3;
            --secondary: #764ba2;
            --success: #10b981;
            --danger: #ef4444;
            --text-dark: #1f2937;
            --text-gray: #6b7280;
            --bg-light: #f9fafb;
            --card-shadow: 0 10px 30px rgba(0,0,0,0.08);
            --card-shadow-hover: 0 20px 40px rgba(0,0,0,0.12);
        }

        body {
            font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            overflow-x: hidden;
            -webkit-font-smoothing: antialiased;
        }

        /* Navbar */
        .navbar {
            background: rgba(255,255,255,0.98);
            backdrop-filter: blur(20px);
            box-shadow: 0 2px 20px rgba(0,0,0,0.08);
            position: sticky;
            top: 0;
            z-index: 100;
        }

        .navbar-content {
            max-width: 900px;
            margin: 0 auto;
            padding: 16px 20px;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .btn-back {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 10px 18px;
            background: rgba(102, 126, 234, 0.1);
            color: var(--primary);
            text-decoration: none;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.9em;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }

        .btn-back:hover {
            background: var(--primary);
            color: white;
            transform: translateX(-4px);
        }

        .btn-back:active {
            transform: translateX(-4px) scale(0.95);
        }

        /* Progress Badge */
        .progress-badge {
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
            padding: 8px 16px;
            border-radius: 12px;
            font-weight: 700;
            font-size: 0.85em;
            display: flex;
            align-items: center;
            gap: 6px;
        }

        /* Study Time Widget */
        .study-time-widget {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 8px 16px;
            border-radius: 12px;
            font-weight: 600;
            font-size: 0.85em;
            display: flex;
            align-items: center;
            gap: 8px;
        }

        .study-time-widget.warning {
            background: linear-gradient(135deg, #f59e0b, #d97706);
            animation: pulse 2s ease-in-out infinite;
        }

        .study-time-widget.danger {
            background: linear-gradient(135deg, #ef4444, #dc2626);
            animation: pulse 1s ease-in-out infinite;
        }

        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.8; }
        }

        .study-time-widget .timer-icon {
            font-size: 1.1em;
        }

        /* Container */
        .container {
            max-width: 900px;
            margin: 0 auto;
            padding: 24px 20px;
            height: calc(100vh - 70px); /* Viewport height minus navbar */
            display: flex;
            flex-direction: column;
        }

        /* Breadcrumb */
        .breadcrumb {
            background: rgba(255,255,255,0.15);
            backdrop-filter: blur(10px);
            padding: 12px 20px;
            border-radius: 12px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            flex-wrap: wrap;
            font-size: 0.85em;
            flex-shrink: 0;
        }

        .breadcrumb a {
            color: white;
            text-decoration: none;
            font-weight: 600;
            transition: opacity 0.2s;
        }

        .breadcrumb a:hover {
            opacity: 0.8;
        }

        .breadcrumb span {
            color: rgba(255,255,255,0.7);
        }

        /* Progress Bar */
        .progress-bar-container {
            padding: 20px 28px 24px;
            background: white;
            border-top: 1px solid #f3f4f6;
            flex-shrink: 0;
        }

        .progress-bar {
            height: 6px;
            background: rgba(102, 126, 234, 0.1);
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 8px;
        }

        .progress-fill {
            height: 100%;
            background: linear-gradient(90deg, var(--primary), var(--secondary));
            border-radius: 10px;
            transition: width 0.6s cubic-bezier(0.4, 0, 0.2, 1);
            position: relative;
            overflow: hidden;
        }

        .progress-fill::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            bottom: 0;
            right: 0;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }

        .progress-text {
            font-size: 0.85em;
            color: var(--text-gray);
            font-weight: 600;
        }

        /* Content Card - The Main Star */
        .content-card {
            background: white;
            border-radius: 24px;
            box-shadow: var(--card-shadow);
            overflow: hidden;
            transition: all 0.4s cubic-bezier(0.4, 0, 0.2, 1);
            animation: slideUp 0.5s ease-out;
            display: flex;
            flex-direction: column;
            flex: 1;
            min-height: 0; /* Important for flex scrolling */
        }

        @keyframes slideUp {
            from {
                opacity: 0;
                transform: translateY(30px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }

        /* Stage Indicator */
        .stage-indicator {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 16px 24px;
            display: flex;
            align-items: center;
            justify-content: space-between;
            flex-shrink: 0;
        }

        .stage-name {
            font-weight: 700;
            font-size: 0.9em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .stage-icon {
            font-size: 1.4em;
        }

        .stage-counter {
            background: rgba(255,255,255,0.2);
            padding: 4px 12px;
            border-radius: 8px;
            font-weight: 700;
            font-size: 0.85em;
        }

        /* Content Body */
        .content-body {
            padding: 32px 28px;
            overflow-y: auto;
            overflow-x: hidden;
            flex: 1;
            min-height: 0;
            -webkit-overflow-scrolling: touch; /* Smooth scrolling on iOS */
        }

        /* Custom scrollbar */
        .content-body::-webkit-scrollbar {
            width: 8px;
        }

        .content-body::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }

        .content-body::-webkit-scrollbar-thumb {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            border-radius: 10px;
        }

        .content-body::-webkit-scrollbar-thumb:hover {
            background: var(--primary-dark);
        }

        /* Text Content */
        .text-content {
            color: var(--text-dark);
            font-size: 1.05em;
            line-height: 1.8;
            font-weight: 400;
        }

        .text-content h1, .text-content h2, .text-content h3 {
            margin: 24px 0 16px;
            font-weight: 700;
            color: var(--text-dark);
        }

        .text-content p {
            margin-bottom: 16px;
        }

        .text-content ul, .text-content ol {
            margin-left: 24px;
            margin-bottom: 16px;
        }

        .text-content li {
            margin-bottom: 8px;
        }

        .text-content strong {
            font-weight: 700;
            color: var(--text-dark);
        }

        .text-content a {
            color: var(--primary);
            text-decoration: underline;
        }

        /* Image Display */
        .image-display {
            text-align: center;
        }

        .image-display img {
            max-width: 100%;
            height: auto;
            border-radius: 16px;
            box-shadow: 0 8px 24px rgba(0,0,0,0.1);
            transition: transform 0.3s ease;
            cursor: pointer;
        }

        .image-display img:hover {
            transform: scale(1.02);
        }

        .image-caption {
            margin-top: 16px;
            font-size: 0.9em;
            color: var(--text-gray);
            font-style: italic;
        }

        /* Video Display */
        .video-display {
            text-align: center;
        }

        .video-wrapper {
            position: relative;
            padding-bottom: 56.25%; /* 16:9 ratio */
            height: 0;
            border-radius: 16px;
            overflow: hidden;
            box-shadow: 0 8px 24px rgba(0,0,0,0.15);
        }

        .video-wrapper iframe,
        .video-wrapper video {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            border-radius: 16px;
        }

        .video-controls {
            margin-top: 20px;
            display: flex;
            justify-content: center;
        }

        /* Navigation Buttons */
        .navigation-buttons {
            display: flex;
            gap: 12px;
            padding: 24px 28px;
            border-top: 1px solid #f3f4f6;
            background: #fafafa;
            flex-shrink: 0;
        }

        .nav-btn {
            flex: 1;
            padding: 16px 24px;
            border: none;
            border-radius: 14px;
            font-weight: 700;
            font-size: 0.95em;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            text-decoration: none;
            position: relative;
            overflow: hidden;
        }

        .nav-btn::before {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 0;
            height: 0;
            border-radius: 50%;
            background: rgba(255,255,255,0.3);
            transform: translate(-50%, -50%);
            transition: width 0.6s, height 0.6s;
        }

        .nav-btn:active::before {
            width: 300px;
            height: 300px;
        }

        .nav-btn.btn-prev {
            background: #e5e7eb;
            color: var(--text-dark);
        }

        .nav-btn.btn-prev:hover {
            background: #d1d5db;
            transform: translateX(-4px);
        }

        .nav-btn.btn-prev:disabled {
            opacity: 0.4;
            cursor: not-allowed;
            transform: none;
        }

        .nav-btn.btn-next {
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
        }

        .nav-btn.btn-next:hover {
            box-shadow: 0 8px 24px rgba(102, 126, 234, 0.4);
            transform: translateY(-2px);
        }

        .nav-btn.btn-next:active {
            transform: translateY(0);
        }

        /* Loading State */
        .loading {
            text-align: center;
            padding: 60px 20px;
        }

        .loading i {
            font-size: 2.5em;
            color: var(--primary);
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }

        .loading p {
            margin-top: 16px;
            color: var(--text-gray);
            font-weight: 600;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--text-gray);
        }

        .empty-state i {
            font-size: 3em;
            margin-bottom: 16px;
            opacity: 0.5;
        }

        /* Quiz Styles */
        .quiz-container {
            max-width: 700px;
            margin: 0 auto;
        }

        .quiz-header {
            text-align: center;
            margin-bottom: 32px;
        }

        .quiz-header h3 {
            font-size: 1.8em;
            color: var(--text-dark);
            margin-bottom: 8px;
        }

        .quiz-header p {
            color: var(--text-gray);
            font-size: 1em;
            margin-bottom: 12px;
        }
        
        .quiz-progress {
            display: inline-block;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 8px 20px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.9em;
        }

        .quiz-question-single {
            background: white;
            border-radius: 16px;
            padding: 32px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            animation: slideUp 0.4s ease;
        }
        
        .quiz-question {
            background: white;
            border-radius: 16px;
            padding: 24px;
            margin-bottom: 24px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }

        .question-number {
            color: var(--primary);
            font-weight: 700;
            font-size: 0.85em;
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-bottom: 12px;
        }

        .question-text {
            font-size: 1.3em;
            font-weight: 600;
            color: var(--text-dark);
            margin-bottom: 24px;
            line-height: 1.6;
        }

        .quiz-options {
            display: flex;
            flex-direction: column;
            gap: 12px;
        }

        .quiz-option {
            position: relative;
            display: flex;
            align-items: center;
            padding: 16px 20px;
            background: var(--bg-light);
            border: 2px solid transparent;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .quiz-option:hover {
            background: #f3f4f6;
            border-color: var(--primary);
        }

        .quiz-option input[type="radio"] {
            position: absolute;
            opacity: 0;
        }

        .quiz-option input[type="radio"]:checked ~ .option-indicator {
            background: var(--primary);
            border-color: var(--primary);
        }

        .quiz-option input[type="radio"]:checked ~ .option-indicator:after {
            display: block;
        }

        .option-indicator {
            width: 24px;
            height: 24px;
            border: 2px solid #d1d5db;
            border-radius: 50%;
            margin-right: 12px;
            flex-shrink: 0;
            position: relative;
            transition: all 0.3s ease;
        }

        .option-indicator:after {
            content: '';
            position: absolute;
            display: none;
            left: 50%;
            top: 50%;
            transform: translate(-50%, -50%);
            width: 10px;
            height: 10px;
            border-radius: 50%;
            background: white;
        }

        .option-text {
            flex: 1;
            font-size: 1em;
            color: var(--text-dark);
        }

        .quiz-option.correct {
            background: #d1fae5;
            border-color: var(--success);
        }

        .quiz-option.correct .option-indicator {
            background: var(--success);
            border-color: var(--success);
        }

        .quiz-option.incorrect {
            background: #fee2e2;
            border-color: var(--danger);
        }

        .quiz-option.incorrect .option-indicator {
            background: var(--danger);
            border-color: var(--danger);
        }

        .quiz-explanation {
            margin-top: 16px;
            padding: 16px;
            background: #dbeafe;
            border-left: 4px solid #3b82f6;
            border-radius: 8px;
            font-size: 0.95em;
            color: #1e3a8a;
        }

        .btn-submit-quiz {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1em;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 24px;
        }

        .btn-submit-quiz:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }

        .btn-submit-quiz:active {
            transform: translateY(0);
        }
        
        .btn-submit-quiz:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Answer Feedback */
        .answer-feedback {
            margin-top: 24px;
            padding: 24px;
            background: white;
            border-radius: 12px;
            text-align: center;
            animation: slideUp 0.4s ease;
        }
        
        .feedback-icon {
            font-size: 3em;
            margin-bottom: 12px;
        }
        
        .feedback-text {
            font-size: 1.3em;
            font-weight: 700;
            margin-bottom: 16px;
            color: var(--text-dark);
        }
        
        .btn-next-question {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1em;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 16px;
        }
        
        .btn-next-question:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(102, 126, 234, 0.4);
        }
        
        .auto-progress-message {
            text-align: center;
            margin-top: 20px;
            padding: 12px;
            background: rgba(102, 126, 234, 0.1);
            border-radius: 8px;
            font-size: 1em;
            color: var(--text-dark);
            font-weight: 600;
        }
        
        .auto-progress-message span {
            color: var(--primary);
            font-size: 1.3em;
            font-weight: 800;
        }
        
        .btn-continue {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1em;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 16px;
        }
        
        .btn-continue:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(16, 185, 129, 0.4);
        }
        
        .btn-retry {
            width: 100%;
            padding: 16px;
            background: linear-gradient(135deg, #f59e0b, #d97706);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.1em;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 16px;
        }
        
        .btn-retry:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(245, 158, 11, 0.4);
        }

        .quiz-result {
            text-align: center;
            padding: 32px;
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            margin-top: 24px;
            animation: slideUp 0.4s ease;
        }

        .quiz-result .result-icon {
            font-size: 4em;
            margin-bottom: 16px;
        }

        .quiz-result h3 {
            font-size: 1.8em;
            margin-bottom: 12px;
            color: var(--text-dark);
        }

        .quiz-result p {
            font-size: 1.2em;
            color: var(--text-gray);
            margin-bottom: 8px;
        }

        .quiz-result.passed {
            background: linear-gradient(135deg, #d1fae5, #a7f3d0);
        }

        .quiz-result.failed {
            background: linear-gradient(135deg, #fef3c7, #fde68a);
        }

        .result-message {
            font-weight: 600;
            margin-top: 16px;
        }

        /* Quiz Failure Modal */
        .quiz-failure-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.85);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            animation: fadeIn 0.3s ease;
        }

        .quiz-failure-content {
            background: white;
            border-radius: 24px;
            padding: 48px 40px;
            max-width: 500px;
            width: 90%;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            animation: scaleIn 0.3s ease;
            border: 4px solid #ef4444;
        }

        .quiz-failure-icon {
            font-size: 5em;
            margin-bottom: 20px;
            animation: shake 0.5s ease;
        }

        .quiz-failure-title {
            font-size: 2.2em;
            font-weight: 800;
            color: #dc2626;
            margin-bottom: 16px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .quiz-failure-message {
            font-size: 1.3em;
            color: #374151;
            margin-bottom: 20px;
            line-height: 1.6;
            font-weight: 600;
        }

        .quiz-failure-countdown {
            font-size: 3em;
            font-weight: 900;
            color: #ef4444;
            margin: 20px 0;
            animation: pulse 1s ease infinite;
        }

        @keyframes fadeIn {
            from { opacity: 0; }
            to { opacity: 1; }
        }

        @keyframes scaleIn {
            from {
                transform: scale(0.8);
                opacity: 0;
            }
            to {
                transform: scale(1);
                opacity: 1;
            }
        }

        @keyframes shake {
            0%, 100% { transform: translateX(0); }
            10%, 30%, 50%, 70%, 90% { transform: translateX(-10px); }
            20%, 40%, 60%, 80% { transform: translateX(10px); }
        }

        @keyframes pulse {
            0%, 100% { transform: scale(1); opacity: 1; }
            50% { transform: scale(1.1); opacity: 0.8; }
        }

        /* Answer Feedback Modal */
        .answer-feedback-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.85);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            animation: fadeIn 0.3s ease;
        }

        .answer-feedback-modal-content {
            background: white;
            border-radius: 24px;
            padding: 40px 35px;
            max-width: 450px;
            width: 90%;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            animation: scaleIn 0.3s ease;
        }

        .answer-feedback-modal-content.correct-answer {
            border: 4px solid #10b981;
        }

        .answer-feedback-modal-content.incorrect-answer {
            border: 4px solid #ef4444;
        }

        .feedback-modal-icon {
            font-size: 4em;
            margin-bottom: 15px;
            animation: bounce 0.6s ease;
        }

        .feedback-modal-title {
            font-size: 2em;
            font-weight: 800;
            margin-bottom: 15px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .correct-answer .feedback-modal-title {
            color: #059669;
        }

        .incorrect-answer .feedback-modal-title {
            color: #dc2626;
        }

        .feedback-modal-explanation {
            font-size: 1.1em;
            color: #374151;
            margin-bottom: 20px;
            line-height: 1.6;
            padding: 15px;
            background: #dbeafe;
            border-radius: 12px;
            border-left: 4px solid #3b82f6;
        }

        .feedback-modal-countdown {
            font-size: 1.2em;
            color: #6b7280;
            font-weight: 600;
            margin-top: 20px;
        }

        .feedback-modal-countdown span {
            font-size: 1.8em;
            font-weight: 900;
            color: var(--primary);
        }

        /* Lesson Intro */
        .lesson-intro {
            background: white;
            border-radius: 16px;
            padding: 28px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
        }

        .lesson-intro-header h2 {
            font-size: 1.6em;
            margin-bottom: 12px;
            color: var(--text-dark);
        }

        .lesson-badge {
            display: inline-block;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            padding: 6px 12px;
            border-radius: 999px;
            font-weight: 700;
            font-size: 0.85em;
            margin-bottom: 10px;
        }

        .lesson-description {
            font-size: 1.05em;
            color: var(--text-gray);
            line-height: 1.6;
            margin-bottom: 16px;
        }

        .lesson-intro-cta {
            margin-top: 12px;
            text-align: right;
        }

        .btn-start-lesson {
            padding: 12px 18px;
            background: linear-gradient(135deg, var(--primary), var(--secondary));
            color: white;
            border: none;
            border-radius: 10px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .btn-start-lesson:hover {
            transform: translateY(-1px);
            box-shadow: 0 6px 18px rgba(102,126,234,0.3);
        }

        /* Quiz Success Modal */
        .quiz-success-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.85);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            animation: fadeIn 0.3s ease;
        }

        .quiz-success-content {
            background: white;
            border-radius: 24px;
            padding: 48px 40px;
            max-width: 500px;
            width: 90%;
            text-align: center;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
            animation: scaleIn 0.3s ease;
            border: 4px solid #10b981;
        }

        .quiz-success-icon {
            font-size: 5em;
            margin-bottom: 20px;
            animation: bounce 0.6s ease;
        }

        .quiz-success-title {
            font-size: 2.2em;
            font-weight: 800;
            color: #059669;
            margin-bottom: 16px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .quiz-success-message {
            font-size: 1.3em;
            color: #374151;
            margin-bottom: 20px;
            line-height: 1.6;
            font-weight: 600;
        }

        .quiz-success-score {
            font-size: 2.5em;
            font-weight: 900;
            color: #10b981;
            margin: 20px 0;
        }

        .quiz-success-button {
            width: 100%;
            padding: 18px;
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
            border: none;
            border-radius: 12px;
            font-size: 1.2em;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 20px;
            text-transform: uppercase;
            letter-spacing: 1px;
        }

        .quiz-success-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px rgba(16, 185, 129, 0.4);
        }

        @keyframes bounce {
            0%, 20%, 50%, 80%, 100% { transform: translateY(0); }
            40% { transform: translateY(-30px); }
            60% { transform: translateY(-15px); }
        }

        /* Toast Notification */
        .toast {
            position: fixed;
            top: 24px;
            right: 24px;
            background: linear-gradient(135deg, var(--success), #059669);
            color: white;
            padding: 16px 24px;
            border-radius: 14px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            z-index: 9999;
            font-weight: 600;
            animation: slideInRight 0.4s ease;
            max-width: 90%;
        }

        .toast.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
        }

        @keyframes slideInRight {
            from {
                transform: translateX(400px);
                opacity: 0;
            }
            to {
                transform: translateX(0);
                opacity: 1;
            }
        }

        @keyframes slideOutRight {
            from {
                transform: translateX(0);
                opacity: 1;
            }
            to {
                transform: translateX(400px);
                opacity: 0;
            }
        }

        /* Mobile Responsive */
        @media (max-width: 768px) {
            .navbar-content {
                padding: 14px 16px;
            }

            .container {
                padding: 20px 16px;
                height: calc(100vh - 60px);
            }

            .content-body {
                padding: 24px 20px;
            }

            .text-content {
                font-size: 1em;
            }

            .navigation-buttons {
                padding: 16px;
                gap: 8px;
            }

            .nav-btn {
                padding: 12px 16px;
                font-size: 0.85em;
            }

            .nav-btn span {
                display: none;
            }

            .nav-btn i {
                margin: 0;
            }

            .stage-indicator {
                padding: 14px 20px;
            }

            .btn-back {
                font-size: 0.85em;
                padding: 8px 14px;
            }

            .progress-badge {
                font-size: 0.8em;
                padding: 6px 12px;
            }

            .progress-bar-container {
                padding: 16px 20px 20px;
            }

            .breadcrumb {
                margin-bottom: 16px;
            }
        }

        @media (max-width: 480px) {
            .container {
                height: calc(100vh - 56px);
            }

            .breadcrumb {
                font-size: 0.8em;
                padding: 10px 16px;
            }

            .content-body {
                padding: 20px 16px;
            }

            .text-content {
                font-size: 0.95em;
                line-height: 1.7;
            }

            .toast {
                right: 16px;
                top: 16px;
                left: 16px;
                max-width: none;
            }
        }

        /* Focus Mode - Hide distractions on scroll */
        .focus-mode .navbar {
            transform: translateY(-100%);
            transition: transform 0.3s ease;
        }

        /* Smooth transitions everywhere */
        * {
            transition: background-color 0.2s ease, color 0.2s ease, border-color 0.2s ease;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="navbar-content">
            <a href="{{ route('learn.course', $course->id) }}" class="btn-back">
                <i class="fas fa-arrow-left"></i>
                <span>Back to Course</span>
            </a>
            <div style="display: flex; gap: 12px; align-items: center;">
                <div id="study-time-widget" class="study-time-widget" style="display: none;">
                    <i class="fas fa-clock timer-icon"></i>
                    <span id="study-timer">--:--</span>
                </div>
                <div class="progress-badge">
                    <i class="fas fa-chart-line"></i>
                    <span id="overall-progress">{{ $overallProgress }}%</span>
                </div>
            </div>
        </div>
    </nav>

    <div class="container">
        <!-- Breadcrumb -->
        <div class="breadcrumb">
            <a href="{{ route('dashboard') }}">Dashboard</a>
            <span>/</span>
            <a href="{{ route('learn.course', $course->id) }}">{{ $course->name }}</a>
            <span>/</span>
            <span>{{ $lesson->title }}</span>
        </div>

        <!-- Main Content Card -->
        <div class="content-card">
            <!-- Stage Indicator -->
            <div class="stage-indicator">
                <div class="stage-name">
                    <span class="stage-icon" id="stage-icon">📝</span>
                    <span id="stage-name">Text Content</span>
                </div>
                <div class="stage-counter" id="stage-counter">1 of 3</div>
            </div>

            <!-- Content Body -->
            <div class="content-body" id="content-body">
                <div class="loading">
                    <i class="fas fa-spinner"></i>
                    <p>Loading content...</p>
                </div>
            </div>

            <!-- Navigation Buttons -->
            <div class="navigation-buttons">
                <button class="nav-btn btn-prev" id="btn-prev" disabled>
                    <i class="fas fa-arrow-left"></i>
                    <span>Previous</span>
                </button>
                <button class="nav-btn btn-next" id="btn-next">
                    <span>Next</span>
                    <i class="fas fa-arrow-right"></i>
                </button>
            </div>

            <!-- Progress Bar Section -->
            <div class="progress-bar-container">
                <div class="progress-bar">
                    <div class="progress-fill" id="stage-progress" style="width: 0%"></div>
                </div>
                <div class="progress-text" id="progress-text">Loading...</div>
            </div>
        </div>
    </div>

    <script>
        const topicId = {{ $topic->id }};
        const courseId = {{ $course->id }};
        const lessonId = {{ $lesson->id }};
        const csrfToken = document.querySelector('meta[name="csrf-token"]').content;

        // Topic data
        const topicData = {
            id: {{ $topic->id }},
            title: "{{ $topic->title }}",
            body: `{!! addslashes($topic->body ?? '') !!}`,
            image_url: "{{ $topic->image_url ?? '' }}",
            video_url: "{{ $topic->video_url ?? '' }}",
            hasQuiz: {{ $quizQuestions->count() > 0 ? 'true' : 'false' }}
        };

        // Quiz questions data
        const quizQuestionsData = {!! json_encode($quizQuestions->map(function($q) {
            return [
                'id' => $q->id,
                'question' => $q->question,
                'options' => json_decode($q->options_json ?? '[]', true) ?: json_decode($q->options ?? '[]', true) ?: [],
                'correct_answer' => $q->correct_index ?? $q->correct_answer ?? 0,
                'explanation' => $q->explanation
            ];
        })->values()) !!};
        
        // Lesson quiz questions data (shown after last topic)
        const lessonQuizQuestionsData = {!! json_encode($lessonQuizQuestions->map(function($q) {
            return [
                'id' => $q->id,
                'question' => $q->question,
                'options' => json_decode($q->options_json ?? '[]', true) ?: json_decode($q->options ?? '[]', true) ?: [],
                'correct_answer' => $q->correct_index ?? $q->correct_answer ?? 0,
                'explanation' => $q->explanation
            ];
        })->values()) !!};
        
        const hasLessonQuiz = lessonQuizQuestionsData.length > 0;

        // Course settings
        const courseChunkWords = {{ $course->chunk_words ?? 100 }};

        const nextTopicUrl = "{{ $nextTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $nextTopic->lesson_id, 'topic' => $nextTopic->id]) : '' }}";
        const prevTopicUrl = "{{ $previousTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $previousTopic->lesson_id, 'topic' => $previousTopic->id]) : '' }}";
        const courseUrl = "{{ route('learn.course', $course->id) }}";
        const firstTopicUrl = "{{ isset($firstTopic) && $firstTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $lesson->id, 'topic' => $firstTopic->id]) : '' }}";
        const lessonData = {
            title: "{{ addslashes($lesson->name) }}",
            description: `{!! addslashes($lesson->intro ?? $lesson->description ?? '') !!}`
        };
        const isFirstTopicInLesson = {{ $isFirstTopicInLesson ? 'true' : 'false' }};
        
        const isLastTopicInLesson = {{ $isLastTopicInLesson ? 'true' : 'false' }};
        const hasNextLesson = {{ isset($nextLesson) && $nextLesson ? 'true' : 'false' }};
        
        @if(isset($nextLesson) && $nextLesson)
        const nextLessonData = {
            id: {{ $nextLesson->id }},
            title: "{{ addslashes($nextLesson->name) }}",
            description: `{!! addslashes($nextLesson->intro ?? $nextLesson->description ?? '') !!}`,
            firstTopicUrl: "{{ $nextLessonFirstTopic ? route('learn.topic', ['course' => $course->id, 'lesson' => $nextLesson->id, 'topic' => $nextLessonFirstTopic->id]) : '' }}"
        };
        @endif

        // App State
        const state = {
            stages: [],
            currentStageIndex: 0,
            textChunks: [],
            currentChunkIndex: 0,
            completed: {{ $userProgress && $userProgress->completed ? 'true' : 'false' }},
            quizPassed: false,
            currentStageCompleted: false,
            passedQuizStages: [] // Track which quiz stages have been passed
        };

        // Study Time Tracking
        const studyTimeState = {
            sessionStart: null,
            timerInterval: null,
            totalMinutesToday: 0,
            maxMinutes: null,
            minMinutes: null,
            enabled: false
        };

        // Initialize
        document.addEventListener('DOMContentLoaded', () => {
            initializeStages();
            renderCurrentStage();
            updateProgress();
            initializeStudyTimeTracking();
        });

        async function initializeStudyTimeTracking() {
            try {
                const response = await fetch(`/api/study-time/${courseId}/status`, {
                    credentials: 'include',
                    headers: {
                        'Accept': 'application/json'
                    }
                });
                
                const ct = response.headers.get('content-type') || '';
                if (!response.ok || !ct.includes('application/json')) {
                    // Not available or not authenticated; quietly disable widget
                    return;
                }
                
                const data = await response.json();
                const status = data.status;
                
                if (!status.limited && status.max_minutes) {
                    studyTimeState.enabled = true;
                    studyTimeState.maxMinutes = status.max_minutes;
                    studyTimeState.minMinutes = status.min_minutes || 0;
                    studyTimeState.totalMinutesToday = status.minutes_studied || 0;
                    
                    // Show widget
                    document.getElementById('study-time-widget').style.display = 'flex';
                    
                    // Start session
                    startStudySession();
                    
                    // Check if already at limit
                    if (!status.can_study) {
                        showTimeLimitReached();
                    }
                }
            } catch (error) {
                console.error('Error initializing study time:', error);
            }
        }

        async function startStudySession() {
            studyTimeState.sessionStart = new Date();
            
            // Start timer display
            studyTimeState.timerInterval = setInterval(updateStudyTimer, 1000);
            
            // Auto-save every 2 minutes
            setInterval(saveStudyProgress, 120000);
        }

        function updateStudyTimer() {
            if (!studyTimeState.sessionStart || !studyTimeState.enabled) return;
            
            const now = new Date();
            const sessionMinutes = Math.floor((now - studyTimeState.sessionStart) / 60000);
            const totalMinutes = studyTimeState.totalMinutesToday + sessionMinutes;
            const remaining = studyTimeState.maxMinutes - totalMinutes;
            
            // Update display
            const timerEl = document.getElementById('study-timer');
            const widgetEl = document.getElementById('study-time-widget');
            
            if (remaining <= 0) {
                timerEl.textContent = '0:00 left';
                widgetEl.className = 'study-time-widget danger';
                showTimeLimitReached();
                return;
            }
            
            const hours = Math.floor(remaining / 60);
            const mins = remaining % 60;
            timerEl.textContent = hours > 0 ? `${hours}h ${mins}m left` : `${mins}m left`;
            
            // Warning states
            if (remaining <= 5) {
                widgetEl.className = 'study-time-widget danger';
            } else if (remaining <= 10) {
                widgetEl.className = 'study-time-widget warning';
            } else {
                widgetEl.className = 'study-time-widget';
            }
        }

        async function saveStudyProgress() {
            if (!studyTimeState.sessionStart) return;
            
            const now = new Date();
            const minutes = Math.floor((now - studyTimeState.sessionStart) / 60000);
            
            if (minutes < 1) return;
            
            try {
                const response = await fetch(`/api/study-time/${courseId}/add`, {
                    method: 'POST',
                    credentials: 'include',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    },
                    body: JSON.stringify({ minutes: minutes })
                });
                
                const ct = response.headers.get('content-type') || '';
                if (!response.ok || !ct.includes('application/json')) return;
                const data = await response.json();
                studyTimeState.totalMinutesToday = data.total_today;
                studyTimeState.sessionStart = new Date(); // Reset session start
            } catch (error) {
                console.error('Error saving study progress:', error);
            }
        }

        function showTimeLimitReached() {
            clearInterval(studyTimeState.timerInterval);
            
            alert(`⏱️ Daily Study Limit Reached!\n\nYou've reached your maximum study time of ${studyTimeState.maxMinutes} minutes for today.\n\nGreat work! Come back tomorrow to continue learning. Consistent daily practice is the key to mastery! 🎯`);
            
            // Redirect to course page
            setTimeout(() => {
                window.location.href = courseUrl;
            }, 3000);
        }

        // Save progress before leaving page
        window.addEventListener('beforeunload', () => {
            if (studyTimeState.sessionStart) {
                saveStudyProgress();
            }
        });

        function initializeStages() {
            // Build stages based on available content
            state.stages = [];

            // Lesson intro stage (only on first topic of the lesson)
            if (isFirstTopicInLesson) {
                state.stages.push({
                    type: 'lessonIntro',
                    title: lessonData.title,
                    description: lessonData.description || 'Get ready for this lesson.'
                });
            }

            // Text content (body field)
            if (topicData.body && topicData.body.trim()) {
                const chunks = chunkText(topicData.body, courseChunkWords || 300);
                chunks.forEach((chunk, index) => {
                    state.stages.push({
                        type: 'text',
                        content: chunk,
                        chunkIndex: index,
                        totalChunks: chunks.length
                    });
                });
            }

            // Image
            if (topicData.image_url && topicData.image_url.trim()) {
                state.stages.push({
                    type: 'image',
                    content: topicData.image_url
                });
            }

            // Video
            if (topicData.video_url && topicData.video_url.trim()) {
                state.stages.push({
                    type: 'video',
                    content: topicData.video_url
                });
            }

            // Quiz (if available)
            if (topicData.hasQuiz) {
                state.stages.push({
                    type: 'quiz',
                    content: 'quiz',
                    quizType: 'topic'
                });
            }
            
            // Lesson Quiz (if this is last topic and lesson has quiz)
            if (isLastTopicInLesson && hasLessonQuiz) {
                state.stages.push({
                    type: 'quiz',
                    content: 'lesson_quiz',
                    quizType: 'lesson'
                });
            }

            // If no content, show empty state
            if (state.stages.length === 0) {
                state.stages.push({
                    type: 'empty',
                    content: 'No content available for this topic.'
                });
            }
        }

        function chunkText(text, maxWords) {
            // Simple HTML-aware chunking
            const hasHTML = /<[a-z][\s\S]*>/i.test(text);
            
            if (hasHTML) {
                // Don't chunk HTML content
                return [text];
            }

            // Split by words
            const words = text.split(/\s+/);
            const chunks = [];
            
            for (let i = 0; i < words.length; i += maxWords) {
                chunks.push(words.slice(i, i + maxWords).join(' '));
            }
            
            return chunks.length > 0 ? chunks : [text];
        }

        function renderCurrentStage() {
            const stage = state.stages[state.currentStageIndex];
            const contentBody = document.getElementById('content-body');
            const stageIcon = document.getElementById('stage-icon');
            const stageName = document.getElementById('stage-name');
            const stageCounter = document.getElementById('stage-counter');
            
            // Update stage indicator
            stageCounter.textContent = `${state.currentStageIndex + 1} of ${state.stages.length}`;
            
            // Render based on type
            switch (stage.type) {
                case 'text':
                    stageIcon.textContent = '📝';
                    stageName.textContent = stage.totalChunks > 1 
                        ? `Text Content (Part ${stage.chunkIndex + 1}/${stage.totalChunks})` 
                        : 'Text Content';
                    contentBody.innerHTML = `<div class="text-content">${stage.content}</div>`;
                    break;

                case 'lessonIntro':
                    stageIcon.textContent = '📘';
                    stageName.textContent = 'Lesson Overview';
                    contentBody.innerHTML = `
                        <div class="lesson-intro">
                            <div class="lesson-intro-header">
                                <div class="lesson-badge">${lessonData.title || 'Lesson'}</div>
                                <h2>${stage.title}</h2>
                            </div>
                            <p class="lesson-description">${lessonData.description || stage.description || 'Get ready to start this lesson.'}</p>
                            <div class="lesson-intro-cta">
                                <button class="btn-start-lesson" onclick="handleNext()">Start Lesson</button>
                            </div>
                        </div>
                    `;
                    break;
                    
                case 'image':
                    stageIcon.textContent = '🖼️';
                    stageName.textContent = 'Image';
                    contentBody.innerHTML = `
                        <div class="image-display">
                            <img src="${stage.content}" alt="Topic Image" onload="this.style.opacity=1" style="opacity:0; transition: opacity 0.3s;">
                        </div>
                    `;
                    break;
                    
                case 'video':
                    stageIcon.textContent = '🎥';
                    stageName.textContent = 'Video';
                    contentBody.innerHTML = `
                        <div class="video-display">
                            <div class="video-wrapper">
                                ${getVideoEmbed(stage.content)}
                            </div>
                        </div>
                    `;
                    break;
                    
                case 'quiz':
                    stageIcon.textContent = '📝';
                    stageName.textContent = stage.quizType === 'lesson' ? 'Lesson Quiz' : 'Quiz';
                    contentBody.innerHTML = renderQuiz(stage);
                    initializeQuiz(stage);
                    break;
                    
                case 'empty':
                    stageIcon.textContent = '📭';
                    stageName.textContent = 'No Content';
                    contentBody.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-inbox"></i>
                            <p>${stage.content}</p>
                        </div>
                    `;
                    break;
            }
            
            // Update button states
            updateButtonStates();
            
            // Trigger animation
            contentBody.style.animation = 'none';
            setTimeout(() => {
                contentBody.style.animation = 'slideUp 0.5s ease-out';
            }, 10);
        }

        function getVideoEmbed(url) {
            // YouTube
            if (/youtu\.be|youtube\.com/.test(url)) {
                const id = (url.match(/(?:v=|\/)([A-Za-z0-9_-]{11})/)||[])[1] || '';
                return `<iframe src="https://www.youtube.com/embed/${id}?rel=0&modestbranding=1" 
                        frameborder="0" allowfullscreen></iframe>`;
            }
            // Regular video
            return `<video controls style="width:100%;border-radius:16px;">
                    <source src="${url}" type="video/mp4">
                    Your browser does not support the video tag.
                    </video>`;
        }

        function renderQuiz(stage) {
            const quizData = stage.quizType === 'lesson' ? lessonQuizQuestionsData : quizQuestionsData;
            const quizTitle = stage.quizType === 'lesson' ? 'Lesson Quiz! 🎓' : 'Quiz Time! 📝';
            const quizDesc = stage.quizType === 'lesson' ? 'Complete this quiz to finish the lesson' : 'Test your knowledge before moving on';
            
            let html = '<div class="quiz-container">';
            html += '<div class="quiz-header">';
            html += `<h3>${quizTitle}</h3>`;
            html += `<p>${quizDesc}</p>`;
            html += `<div class="quiz-progress">Question <span id="current-question">1</span> of <span id="total-questions">${quizData.length}</span></div>`;
            html += '</div>';
            
            html += `<div id="quiz-content" data-quiz-type="${stage.quizType}">`;
            // Questions will be rendered one at a time by JavaScript
            html += '</div>';
            
            html += '<div id="quiz-result" style="display:none;"></div>';
            html += '</div>';
            
            return html;
        }

        function initializeQuiz(stage) {
            const quizContent = document.getElementById('quiz-content');
            if (!quizContent) return;
            
            const quizType = stage.quizType; // Use stage parameter instead of dataset
            const quizData = quizType === 'lesson' ? lessonQuizQuestionsData : quizQuestionsData;
            
            let currentQuestionIndex = 0;
            let userAnswers = {};
            let correctCount = 0;
            
            function renderCurrentQuestion() {
                const q = quizData[currentQuestionIndex];
                const questionNum = currentQuestionIndex + 1;
                
                document.getElementById('current-question').textContent = questionNum;
                
                let html = `<div class="quiz-question-single" data-question-id="${q.id}">`;
                html += `<div class="question-text">${q.question}</div>`;
                html += '<div class="quiz-options">';
                
                q.options.forEach((option, oIndex) => {
                    html += `
                        <label class="quiz-option" data-option-index="${oIndex}">
                            <input type="radio" name="current_question" value="${oIndex}" required>
                            <span class="option-text">${option}</span>
                            <span class="option-indicator"></span>
                        </label>
                    `;
                });
                
                html += '</div>';
                html += '</div>';
                
                quizContent.innerHTML = html;
                
                // Attach auto-submit handler to radio buttons
                document.querySelectorAll('input[name="current_question"]').forEach(radio => {
                    radio.addEventListener('change', function() {
                        // Small delay to show selection animation
                        setTimeout(() => {
                            submitCurrentAnswer();
                        }, 300);
                    });
                });
            }
            
            function submitCurrentAnswer() {
                const q = quizData[currentQuestionIndex];
                const selected = document.querySelector('input[name="current_question"]:checked');
                
                if (!selected) {
                    showToast('Please select an answer', 'error');
                    return;
                }
                
                const userAnswer = parseInt(selected.value);
                const correctAnswer = q.correct_answer;
                const isCorrect = userAnswer === correctAnswer;
                
                userAnswers[q.id] = userAnswer;
                if (isCorrect) correctCount++;
                
                // Disable all options and remove event listeners
                document.querySelectorAll('input[name="current_question"]').forEach(input => {
                    input.disabled = true;
                    input.parentElement.style.cursor = 'not-allowed';
                    input.parentElement.style.opacity = '0.7';
                });
                
                // Show feedback as modal popup
                showAnswerFeedbackModal(isCorrect, q.explanation);
            }
            
            function showAnswerFeedbackModal(isCorrect, explanation) {
                const isLastQuestion = currentQuestionIndex >= quizData.length - 1;
                
                // Create modal HTML
                const modalHtml = `
                    <div class="answer-feedback-modal" id="answer-feedback-modal">
                        <div class="answer-feedback-modal-content ${isCorrect ? 'correct-answer' : 'incorrect-answer'}">
                            <div class="feedback-modal-icon">${isCorrect ? '✅' : '❌'}</div>
                            <div class="feedback-modal-title">${isCorrect ? 'Correct!' : 'Incorrect!'}</div>
                            ${isCorrect && explanation ? `<div class="feedback-modal-explanation">${explanation}</div>` : ''}
                            <div class="feedback-modal-countdown">
                                ${isLastQuestion ? 'Finishing quiz in' : 'Next question in'} 
                                <span id="feedback-countdown">3</span>...
                            </div>
                        </div>
                    </div>
                `;
                
                // Add modal to body
                document.body.insertAdjacentHTML('beforeend', modalHtml);
                
                // Auto-progress countdown
                let countdown = 3;
                const countdownEl = document.getElementById('feedback-countdown');
                
                const autoProgressInterval = setInterval(() => {
                    countdown--;
                    if (countdownEl) {
                        countdownEl.textContent = countdown;
                    }
                    
                    if (countdown <= 0) {
                        clearInterval(autoProgressInterval);
                        
                        // Remove modal
                        const modal = document.getElementById('answer-feedback-modal');
                        if (modal) {
                            modal.remove();
                        }
                        
                        if (isLastQuestion) {
                            // Finish quiz
                            submitQuiz();
                        } else {
                            // Move to next question
                            currentQuestionIndex++;
                            renderCurrentQuestion();
                        }
                    }
                }, 1000);
            }
            
            async function submitQuiz() {
                const totalQuestions = quizData.length;
                const score = Math.round((correctCount / totalQuestions) * 100);
                const passed = score >= 70;
                
                // If failed, show modal and restart from beginning
                if (!passed) {
                    showQuizFailureModal(score, totalQuestions);
                    return;
                }
                
                // Only submit to server if passed
                try {
                    const url = quizType === 'lesson' 
                        ? `/learn/lesson/${lessonId}/quiz`
                        : `/learn/topic/${topicId}/quiz`;
                    
                    const response = await fetch(url, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': csrfToken
                        },
                        body: JSON.stringify({ answers: userAnswers })
                    });
                    
                    if (!response.ok) {
                        throw new Error(`HTTP error! status: ${response.status}`);
                    }
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        showFinalQuizResults(data, passed, quizType);
                    } else {
                        console.error('Server response:', data);
                        showToast('Error submitting quiz: ' + (data.message || 'Unknown error'), 'error');
                    }
                } catch (error) {
                    console.error('Quiz submission error:', error);
                    showToast('Error submitting quiz: ' + error.message, 'error');
                }
            }
            
            // Start with first question
            renderCurrentQuestion();
        }

        function showFinalQuizResults(data, passed, quizType) {
            // Set state
            state.quizPassed = true;
            state.completed = true;
            
            // Mark this quiz stage as passed
            if (!state.passedQuizStages.includes(state.currentStageIndex)) {
                state.passedQuizStages.push(state.currentStageIndex);
            }
            
            // Create success modal
            const modalHtml = `
                <div class="quiz-success-modal" id="quiz-success-modal">
                    <div class="quiz-success-content">
                        <div class="quiz-success-icon">🎉</div>
                        <div class="quiz-success-title">Lesson Completed!</div>
                        <div class="quiz-success-message">
                            Great job! You've finished this lesson.
                        </div>
                        <div class="quiz-success-score">${data.score}%</div>
                        <div class="quiz-success-message" style="font-size: 1.1em;">
                            You got ${data.correct} out of ${data.total} questions correct
                        </div>
                        ${hasNextLesson && typeof nextLessonData !== 'undefined' ? `
                            <div class="quiz-success-message" style="margin-top:16px;font-weight:700;">Next Lesson:</div>
                            <div class="quiz-success-message" style="font-size:1.05em;">${nextLessonData.title}</div>
                            ${nextLessonData.description && nextLessonData.description.trim() ? `
                                <div class="quiz-success-message" style="font-size:0.95em;color:#6b7280;margin-top:6px;">
                                    ${nextLessonData.description}
                                </div>
                            ` : ''}
                        ` : `
                            <div class="quiz-success-message" style="margin-top:16px;font-weight:700;">Course Progress</div>
                            <div class="quiz-success-message" style="font-size:0.95em;color:#6b7280;margin-top:6px;">
                                Return to course overview to pick your next lesson.
                            </div>
                        `}
                        <button type="button" class="quiz-success-button" id="btn-success-continue">
                            ${hasNextLesson ? 'Go to Next Lesson' : 'Return to Course'}
                        </button>
                    </div>
                </div>
            `;
            
            // Add modal to body
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            
            // Attach continue button handler
            document.getElementById('btn-success-continue')?.addEventListener('click', () => {
                // Remove modal
                const modal = document.getElementById('quiz-success-modal');
                if (modal) {
                    modal.remove();
                }
                
                // Mark stage as completed and proceed
                if (hasNextLesson && typeof nextLessonData !== 'undefined') {
                    // If there's a next lesson, navigate to its first topic if available
                    if (nextTopicUrl) {
                        window.location.href = nextTopicUrl;
                    } else {
                        handleComplete();
                    }
                } else {
                    if (state.currentStageIndex < state.stages.length - 1) {
                        handleNext();
                    } else {
                        handleComplete();
                    }
                }
            });
        }

        function showQuizFailureModal(score, totalQuestions) {
            // Create modal HTML
            const modalHtml = `
                <div class="quiz-failure-modal" id="quiz-failure-modal">
                    <div class="quiz-failure-content">
                        <div class="quiz-failure-icon">😞</div>
                        <div class="quiz-failure-title">Quiz Failed!</div>
                        <div class="quiz-failure-message">
                            You scored ${score}%<br>
                            You need at least 70% to pass.<br><br>
                            <strong>You will restart from the beginning of this lesson.</strong>
                        </div>
                        <div class="quiz-failure-countdown" id="failure-countdown">5</div>
                    </div>
                </div>
            `;
            
            // Add modal to body
            document.body.insertAdjacentHTML('beforeend', modalHtml);
            
            // Countdown from 5 to 1
            let countdown = 5;
            const countdownEl = document.getElementById('failure-countdown');
            
            const countdownInterval = setInterval(() => {
                countdown--;
                if (countdownEl) {
                    countdownEl.textContent = countdown;
                }
                
                if (countdown <= 0) {
                    clearInterval(countdownInterval);
                    
                    // Remove modal
                    const modal = document.getElementById('quiz-failure-modal');
                    if (modal) {
                        modal.remove();
                    }
                    
                    // Restart from beginning - clear all quiz progress
                    state.currentStageIndex = 0;
                    state.quizPassed = false;
                    state.passedQuizStages = []; // Clear all quiz completions

                    // Redirect to the first topic of this lesson to ensure full restart
                    if (firstTopicUrl && firstTopicUrl.length > 0) {
                        window.location.href = firstTopicUrl;
                        return;
                    }

                    // Fallback: Rebuild stages and start from first non-quiz content
                    initializeStages();
                    let firstContentStage = state.stages.findIndex(s => s && s.type !== 'quiz');
                    if (firstContentStage === -1) {
                        firstContentStage = 0; // Fallback if only quiz exists
                    }
                    state.currentStageIndex = firstContentStage;
                    state.currentStageCompleted = false;

                    renderCurrentStage();
                    updateProgress();
                }
            }, 1000);
        }

        function updateButtonStates() {
            const btnPrev = document.getElementById('btn-prev');
            const btnNext = document.getElementById('btn-next');
            const currentStage = state.stages[state.currentStageIndex];
            
            // Previous button
            if (state.currentStageIndex === 0) {
                btnPrev.disabled = true;
                if (prevTopicUrl) {
                    btnPrev.innerHTML = '<i class="fas fa-arrow-left"></i><span>Previous Topic</span>';
                    btnPrev.onclick = () => window.location.href = prevTopicUrl;
                    btnPrev.disabled = false;
                }
            } else {
                btnPrev.disabled = false;
                btnPrev.innerHTML = '<i class="fas fa-arrow-left"></i><span>Previous</span>';
                btnPrev.onclick = handlePrevious;
            }
            
            // Next button - disable on quiz stages until passed
            if (currentStage && currentStage.type === 'quiz') {
                // Check if this specific quiz stage has been passed
                if (!state.passedQuizStages.includes(state.currentStageIndex)) {
                    btnNext.style.display = 'none'; // Hide next button during quiz
                    return;
                }
            }
            
            btnNext.style.display = 'flex'; // Show next button for non-quiz stages
            
            if (state.currentStageIndex === state.stages.length - 1) {
                btnNext.innerHTML = '<span>Complete & Next</span><i class="fas fa-check"></i>';
                btnNext.onclick = handleComplete;
            } else {
                btnNext.innerHTML = '<span>Next</span><i class="fas fa-arrow-right"></i>';
                btnNext.onclick = handleNext;
            }
        }

        function handlePrevious() {
            if (state.currentStageIndex > 0) {
                state.currentStageIndex--;
                renderCurrentStage();
                updateProgress();
            }
        }

        function handleNext() {
            const currentStage = state.stages[state.currentStageIndex];
            
            // Check if current stage is quiz and not passed
            if (currentStage && currentStage.type === 'quiz') {
                // Check if this specific quiz stage has been passed
                if (!state.passedQuizStages.includes(state.currentStageIndex)) {
                    showToast('You must pass the quiz to proceed', 'error');
                    return;
                }
            }
            
            // Move to next stage
            if (state.currentStageIndex < state.stages.length - 1) {
                state.currentStageIndex++;
                renderCurrentStage();
                updateProgress();
            }
        }

        async function handleComplete() {
            const btnNext = document.getElementById('btn-next');
            btnNext.disabled = true;
            btnNext.innerHTML = '<i class="fas fa-spinner fa-spin"></i><span>Completing...</span>';
            
            try {
                const response = await fetch(`/learn/topic/${topicId}/complete`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': csrfToken
                    }
                });
                
                const data = await response.json();
                
                if (data.success) {
                    state.completed = true;
                    showToast('Topic completed!', 'success');
                    
                    // Check if we need to show congratulations modal
                    if (isLastTopicInLesson && hasNextLesson) {
                        setTimeout(async () => {
                            const lessonCheck = await fetch(`/api/lesson/${lessonId}/check-completion`, {
                                headers: { 'X-CSRF-TOKEN': csrfToken }
                            });
                            const lessonData = await lessonCheck.json();
                            
                            if (lessonData.completed) {
                                // Show congratulations modal (implement this)
                                showCongratsModal();
                                return;
                            }
                            
                            // Navigate to next topic or course
                            if (nextTopicUrl) {
                                window.location.href = nextTopicUrl;
                            } else {
                                window.location.href = courseUrl;
                            }
                        }, 500);
                    } else {
                        // Navigate to next topic or course
                        setTimeout(() => {
                            if (nextTopicUrl) {
                                window.location.href = nextTopicUrl;
                            } else {
                                window.location.href = courseUrl;
                            }
                        }, 1000);
                    }
                } else {
                    throw new Error(data.message || 'Failed to complete topic');
                }
            } catch (error) {
                showToast(error.message, 'error');
                btnNext.disabled = false;
                updateButtonStates();
            }
        }

        function updateProgress() {
            const progress = ((state.currentStageIndex + 1) / state.stages.length) * 100;
            document.getElementById('stage-progress').style.width = progress + '%';
            document.getElementById('progress-text').textContent = 
                `Stage ${state.currentStageIndex + 1} of ${state.stages.length}`;
        }

        function showToast(message, type = 'success') {
            const toast = document.createElement('div');
            toast.className = `toast ${type}`;
            toast.textContent = message;
            document.body.appendChild(toast);
            
            setTimeout(() => {
                toast.style.animation = 'slideOutRight 0.4s ease';
                setTimeout(() => toast.remove(), 400);
            }, 4000);
        }

        function showCongratsModal() {
            // Create a beautiful modal for lesson completion
            const modal = document.createElement('div');
            modal.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0,0,0,0.8);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 9999;
                animation: fadeIn 0.3s ease;
            `;
            
            modal.innerHTML = `
                <div style="
                    background: white;
                    border-radius: 20px;
                    padding: 40px;
                    max-width: 500px;
                    width: 90%;
                    text-align: center;
                    box-shadow: 0 25px 50px rgba(0,0,0,0.3);
                    animation: slideUp 0.4s ease;
                ">
                    <div style="font-size: 4em; margin-bottom: 20px;">🎉</div>
                    <h2 style="
                        font-size: 2em;
                        font-weight: 700;
                        color: #1f2937;
                        margin-bottom: 10px;
                    ">Lesson Completed!</h2>
                    <p style="
                        color: #6b7280;
                        font-size: 1.1em;
                        margin-bottom: 30px;
                    ">Great job! You've finished this lesson.</p>
                    
                    ${hasNextLesson && typeof nextLessonData !== 'undefined' ? `
                        <div style="
                            background: #f3f4f6;
                            border-radius: 15px;
                            padding: 25px;
                            margin-bottom: 30px;
                            text-align: left;
                        ">
                            <h3 style="
                                font-size: 1.3em;
                                font-weight: 600;
                                color: #1f2937;
                                margin-bottom: 10px;
                            ">Next Lesson: ${nextLessonData.title}</h3>
                            <p style="
                                color: #6b7280;
                                line-height: 1.6;
                                margin-bottom: 20px;
                            ">${nextLessonData.description && nextLessonData.description.trim() ? nextLessonData.description : ''}</p>
                            <button onclick="proceedToNextLesson()" style="
                                background: linear-gradient(135deg, #667eea, #764ba2);
                                color: white;
                                border: none;
                                padding: 12px 30px;
                                border-radius: 25px;
                                font-weight: 600;
                                font-size: 1em;
                                cursor: pointer;
                                transition: transform 0.2s ease;
                                width: 100%;
                            " onmouseover="this.style.transform='translateY(-2px)'" onmouseout="this.style.transform='translateY(0)'">
                                Proceed to Topics
                            </button>
                        </div>
                    ` : ''}
                    
                    <button onclick="backToCourse()" style="
                        background: #f3f4f6;
                        color: #6b7280;
                        border: none;
                        padding: 12px 30px;
                        border-radius: 25px;
                        font-weight: 600;
                        cursor: pointer;
                        transition: background 0.2s ease;
                    " onmouseover="this.style.background='#e5e7eb'" onmouseout="this.style.background='#f3f4f6'">
                        Back to Course
                    </button>
                </div>
            `;
            
            // Add animation styles
            const style = document.createElement('style');
            style.textContent = `
                @keyframes fadeIn {
                    from { opacity: 0; }
                    to { opacity: 1; }
                }
                @keyframes slideUp {
                    from { transform: translateY(30px); opacity: 0; }
                    to { transform: translateY(0); opacity: 1; }
                }
            `;
            document.head.appendChild(style);
            
            document.body.appendChild(modal);
            
            // Close modal functions
            window.proceedToNextLesson = function() {
                modal.remove();
                window.location.href = nextTopicUrl || courseUrl;
            };
            
            window.backToCourse = function() {
                modal.remove();
                window.location.href = courseUrl;
            };
            
            // Close on escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    modal.remove();
                }
            });
        }

        // Focus mode - hide navbar on scroll down
        let lastScroll = 0;
        window.addEventListener('scroll', () => {
            const currentScroll = window.pageYOffset;
            if (currentScroll > lastScroll && currentScroll > 100) {
                document.body.classList.add('focus-mode');
            } else {
                document.body.classList.remove('focus-mode');
            }
            lastScroll = currentScroll;
        });

        // Update overall progress badge (robust to non-JSON responses)
        fetch(`/api/course/${courseId}/progress`, {
            credentials: 'include',
            headers: {
                'Accept': 'application/json'
            }
        })
            .then(r => {
                const ct = r.headers.get('content-type') || '';
                if (!r.ok || !ct.includes('application/json')) {
                    return null; // Likely redirected HTML; keep current display
                }
                return r.json();
            })
            .then(d => {
                if (d && typeof d.progress !== 'undefined') {
                    const el = document.getElementById('overall-progress');
                    if (el) el.textContent = d.progress + '%';
                }
            })
            .catch(() => {});
    </script>
</body>
</html>
