<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    @php
        $brand = \App\Support\BrandSettings::data();
        $siteName = $brand['name'] ?? 'EE Micro LMS';
        $siteTagline = $brand['tagline'] ?? 'Entrepreneurship Training Hub';
        $logoUrl = $brand['logo_url'] ?? null;
        $faviconUrl = $brand['favicon_url'] ?? null;
        $primaryColor = $brand['colors']['primary'] ?? '#667eea';
        $secondaryColor = $brand['colors']['secondary'] ?? '#764ba2';
    @endphp
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>🎓 My Learning Journey - {{ $siteName }}</title>
    @if (!empty($faviconUrl))
        <link rel="icon" type="image/x-icon" href="{{ $faviconUrl }}">
    @endif
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;500;600;700;800&display=swap" rel="stylesheet">
    
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        :root {
            --primary: {{ $primaryColor }};
            --primary-light: {{ $primaryColor }};
            --primary-dark: {{ $primaryColor }};
            --secondary: {{ $secondaryColor }};
            --accent: #f093fb;
            --success: #10b981;
            --warning: #f59e0b;
            --danger: #ef4444;
            --info: #3b82f6;
            --purple-gradient: linear-gradient(135deg, var(--primary) 0%, var(--secondary) 100%);
            --card-shadow: 0 10px 30px rgba(0,0,0,0.1);
            --hover-shadow: 0 15px 40px rgba(102, 126, 234, 0.3);
        }

        body {
            font-family: 'Poppins', sans-serif;
            background: var(--purple-gradient);
            min-height: 100vh;
            position: relative;
            overflow-x: hidden;
        }

        /* Animated Background */
        body::before {
            content: '';
            position: fixed;
            top: -50%;
            left: -50%;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.1) 1px, transparent 1px);
            background-size: 50px 50px;
            animation: backgroundMove 20s linear infinite;
            pointer-events: none;
            z-index: 0;
        }

        @keyframes backgroundMove {
            0% { transform: translate(0, 0); }
            100% { transform: translate(50px, 50px); }
        }

        /* Navbar */
        .navbar {
            background: rgba(255,255,255,0.98);
            padding: 15px 0;
            backdrop-filter: blur(20px);
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            z-index: 1000;
        }

        .navbar-content {
            max-width: 100%;
            margin: 0 auto;
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0 30px;
        }

        .logo {
            display: flex;
            align-items: center;
            gap: 12px;
            max-height: 50px;
        }
        
        .logo img {
            display: block;
            max-height: 45px;
            max-width: 180px;
            width: auto;
            height: auto;
            object-fit: contain;
        }

        .logo-icon {
            color: var(--primary);
            animation: bounce 2s ease-in-out infinite;
            font-size: 1.8em;
            flex-shrink: 0;
        }

        .logo-text {
            display: flex;
            flex-direction: column;
            line-height: 1.1;
        }

        .logo-text span {
            font-size: 1.25em;
            font-weight: 800;
            color: #1f2937;
        }

        .logo-text small {
            font-size: 0.75em;
            font-weight: 600;
            color: var(--secondary);
        }

        @keyframes bounce {
            0%, 100% { transform: translateY(0); }
            50% { transform: translateY(-5px); }
        }

        .user-info {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .user-avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: var(--purple-gradient);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            font-weight: 700;
            font-size: 1.1em;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.4);
            transition: all 0.3s ease;
        }

        .user-avatar:hover {
            transform: scale(1.1) rotate(5deg);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.6);
        }

        .user-name {
            font-weight: 600;
            color: #333;
        }

        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 25px;
            text-decoration: none;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }

        .btn-primary {
            background: var(--purple-gradient);
            color: white;
            box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
        }

        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(102, 126, 234, 0.5);
        }

        .btn-logout {
            background: rgba(239, 68, 68, 0.1);
            color: var(--danger);
        }

        .btn-logout:hover {
            background: var(--danger);
            color: white;
        }

        /* Main Layout */
        .dashboard-container {
            display: flex;
            margin-top: 75px;
            min-height: calc(100vh - 75px);
            position: relative;
            z-index: 1;
        }

        /* Sidebar */
        .sidebar {
            width: 280px;
            background: rgba(255,255,255,0.15);
            backdrop-filter: blur(20px);
            padding: 25px;
            position: fixed;
            height: calc(100vh - 75px);
            overflow-y: auto;
            border-right: 1px solid rgba(255,255,255,0.3);
            transition: all 0.3s ease;
        }

        .sidebar::-webkit-scrollbar {
            width: 6px;
        }

        .sidebar::-webkit-scrollbar-track {
            background: rgba(255,255,255,0.1);
            border-radius: 10px;
        }

        .sidebar::-webkit-scrollbar-thumb {
            background: rgba(255,255,255,0.3);
            border-radius: 10px;
        }

        .sidebar::-webkit-scrollbar-thumb:hover {
            background: rgba(255,255,255,0.5);
        }

        .sidebar-section {
            margin-bottom: 30px;
        }

        .sidebar-section h3 {
            color: white;
            font-size: 0.85em;
            font-weight: 700;
            text-transform: uppercase;
            letter-spacing: 1.5px;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid rgba(255,255,255,0.3);
        }

        .sidebar-menu {
            list-style: none;
        }

        .sidebar-menu li {
            margin-bottom: 8px;
        }

        .sidebar-nav {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 14px 18px;
            color: rgba(255,255,255,0.9);
            text-decoration: none;
            border-radius: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
            font-weight: 500;
            position: relative;
            overflow: hidden;
        }

        .sidebar-nav::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            background: white;
            transform: scaleY(0);
            transition: transform 0.3s ease;
        }

        .sidebar-nav:hover {
            background: rgba(255,255,255,0.25);
            color: white;
            transform: translateX(5px);
        }

        .sidebar-nav.active {
            background: rgba(255,255,255,0.3);
            color: white;
            font-weight: 700;
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .sidebar-nav.active::before {
            transform: scaleY(1);
        }

        .sidebar-nav i {
            width: 24px;
            text-align: center;
            font-size: 1.2em;
        }

        .sidebar-nav .badge {
            margin-left: auto;
            background: rgba(239, 68, 68, 0.9);
            color: white;
            padding: 4px 10px;
            border-radius: 12px;
            font-size: 0.75em;
            font-weight: 700;
        }

        /* Content Area */
        .content-area {
            flex: 1;
            margin-left: 280px;
            padding: 30px;
            position: relative;
        }

        .welcome-header {
            background: rgba(255,255,255,0.2);
            backdrop-filter: blur(20px);
            padding: 35px;
            border-radius: 25px;
            margin-bottom: 30px;
            border: 1px solid rgba(255,255,255,0.3);
            position: relative;
            overflow: hidden;
        }

        .welcome-header::before {
            content: '🎉';
            position: absolute;
            top: 20px;
            right: 30px;
            font-size: 4em;
            opacity: 0.3;
            animation: celebration 3s ease-in-out infinite;
        }

        @keyframes celebration {
            0%, 100% { transform: rotate(-10deg); }
            50% { transform: rotate(10deg); }
        }

        .welcome-header h1 {
            color: white;
            font-size: 2.5em;
            margin-bottom: 10px;
            font-weight: 800;
        }

        .welcome-header p {
            color: rgba(255,255,255,0.95);
            font-size: 1.1em;
            font-weight: 400;
        }

        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(240px, 1fr));
            gap: 25px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: rgba(255,255,255,0.98);
            padding: 30px;
            border-radius: 20px;
            box-shadow: var(--card-shadow);
            transition: all 0.4s ease;
            cursor: pointer;
            position: relative;
            overflow: hidden;
        }

        .stat-card::before {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            height: 5px;
            background: var(--purple-gradient);
            transform: scaleX(0);
            transition: transform 0.4s ease;
        }

        .stat-card:hover {
            transform: translateY(-10px);
            box-shadow: var(--hover-shadow);
        }

        .stat-card:hover::before {
            transform: scaleX(1);
        }

        .stat-icon {
            width: 65px;
            height: 65px;
            border-radius: 18px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.8em;
            margin-bottom: 20px;
            transition: all 0.3s ease;
        }

        .stat-card:hover .stat-icon {
            transform: scale(1.1) rotate(5deg);
        }

    .stat-card:nth-child(1) .stat-icon { background: var(--purple-gradient); color: white; }
        .stat-card:nth-child(2) .stat-icon { background: linear-gradient(135deg, #10b981 0%, #059669 100%); color: white; }
        .stat-card:nth-child(3) .stat-icon { background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); color: white; }
        .stat-card:nth-child(4) .stat-icon { background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); color: white; }

        .stat-value {
            font-size: 2.5em;
            font-weight: 800;
            color: #333;
            margin-bottom: 5px;
        }

        .stat-label {
            color: #6b7280;
            font-weight: 600;
            font-size: 0.95em;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        /* Section Cards */
        .section {
            margin-bottom: 30px;
        }

        .section-title {
            color: white;
            font-size: 1.8em;
            font-weight: 700;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .section-title i {
            background: rgba(255,255,255,0.2);
            padding: 12px;
            border-radius: 12px;
        }

        .card {
            background: rgba(255,255,255,0.98);
            border-radius: 20px;
            padding: 30px;
            box-shadow: var(--card-shadow);
            margin-bottom: 25px;
            transition: all 0.3s ease;
        }

        .card:hover {
            box-shadow: var(--hover-shadow);
        }

        /* Course Cards */
        .courses-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(320px, 1fr));
            gap: 25px;
        }

        .course-card {
            background: white;
            border-radius: 20px;
            overflow: hidden;
            box-shadow: var(--card-shadow);
            transition: all 0.4s ease;
            cursor: pointer;
            position: relative;
        }

        .course-card::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: var(--purple-gradient);
            opacity: 0;
            transition: opacity 0.3s ease;
            pointer-events: none;
        }

        .course-card:hover {
            transform: translateY(-10px) scale(1.02);
            box-shadow: 0 20px 50px rgba(102, 126, 234, 0.4);
        }

        .course-thumbnail {
            height: 180px;
            background: var(--purple-gradient);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 4em;
            position: relative;
            overflow: hidden;
        }

        .course-thumbnail::before {
            content: '';
            position: absolute;
            width: 200%;
            height: 200%;
            background: radial-gradient(circle, rgba(255,255,255,0.2) 0%, transparent 70%);
            animation: ripple 3s ease-in-out infinite;
        }

        @keyframes ripple {
            0%, 100% { transform: scale(0.8); opacity: 0.5; }
            50% { transform: scale(1.2); opacity: 0.8; }
        }

        .course-content {
            padding: 25px;
        }

        .course-title {
            font-size: 1.3em;
            font-weight: 700;
            color: #333;
            margin-bottom: 12px;
            line-height: 1.3;
        }

        .course-meta {
            display: flex;
            gap: 20px;
            margin-bottom: 20px;
            color: #6b7280;
            font-size: 0.9em;
        }

        .course-meta span {
            display: flex;
            align-items: center;
            gap: 6px;
        }

        .progress-bar-container {
            background: #f3f4f6;
            height: 10px;
            border-radius: 10px;
            overflow: hidden;
            margin-bottom: 10px;
            position: relative;
        }

        .progress-bar {
            height: 100%;
            background: var(--purple-gradient);
            border-radius: 10px;
            transition: width 0.6s ease;
            position: relative;
            overflow: hidden;
        }

        .progress-bar::after {
            content: '';
            position: absolute;
            top: 0;
            left: 0;
            bottom: 0;
            right: 0;
            background: linear-gradient(90deg, transparent, rgba(255,255,255,0.3), transparent);
            animation: shimmer 2s infinite;
        }

        @keyframes shimmer {
            0% { transform: translateX(-100%); }
            100% { transform: translateX(100%); }
        }

        .progress-text {
            font-size: 0.85em;
            color: #6b7280;
            font-weight: 600;
        }

        .course-actions {
            display: flex;
            gap: 10px;
            margin-top: 20px;
        }

        .btn-course {
            flex: 1;
            padding: 12px;
            border-radius: 12px;
            border: none;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .btn-continue {
            background: var(--purple-gradient);
            color: white;
        }

        .btn-continue:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        .btn-view {
            background: rgba(102, 126, 234, 0.1);
            color: var(--primary);
        }

        .btn-view:hover {
            background: rgba(102, 126, 234, 0.2);
        }

        /* Achievements */
        .achievements-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(140px, 1fr));
            gap: 20px;
        }

        .achievement-badge {
            background: white;
            padding: 25px;
            border-radius: 18px;
            text-align: center;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            transition: all 0.3s ease;
            cursor: pointer;
            position: relative;
        }

        .achievement-badge:hover {
            transform: translateY(-8px) rotate(2deg);
            box-shadow: 0 15px 30px rgba(102, 126, 234, 0.3);
        }

        .badge-icon {
            font-size: 3em;
            margin-bottom: 12px;
            display: block;
            filter: grayscale(100%);
            opacity: 0.5;
            transition: all 0.3s ease;
        }

        .achievement-badge.earned .badge-icon {
            filter: grayscale(0%);
            opacity: 1;
            animation: badgePop 0.5s ease;
        }

        @keyframes badgePop {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.2); }
        }

        .achievement-badge.earned::after {
            content: '✓';
            position: absolute;
            top: 10px;
            right: 10px;
            background: var(--success);
            color: white;
            width: 24px;
            height: 24px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 0.8em;
            font-weight: bold;
        }

        .badge-name {
            font-size: 0.85em;
            font-weight: 600;
            color: #333;
        }

        /* Activity Feed */
        .activity-item {
            display: flex;
            gap: 18px;
            padding: 20px;
            background: #f9fafb;
            border-radius: 15px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
        }

        .activity-item::before {
            content: '';
            position: absolute;
            left: 0;
            top: 0;
            bottom: 0;
            width: 4px;
            background: var(--purple-gradient);
        }

        .activity-item:hover {
            background: white;
            box-shadow: 0 5px 15px rgba(0,0,0,0.08);
            transform: translateX(5px);
        }

        .activity-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.3em;
            flex-shrink: 0;
        }

        .activity-icon.completed { background: linear-gradient(135deg, #10b981, #059669); color: white; }
        .activity-icon.in-progress { background: linear-gradient(135deg, #f59e0b, #d97706); color: white; }
        .activity-icon.started { background: linear-gradient(135deg, #3b82f6, #2563eb); color: white; }

        .activity-details {
            flex: 1;
        }

        .activity-title {
            font-weight: 700;
            color: #333;
            margin-bottom: 5px;
        }

        .activity-course {
            color: #6b7280;
            font-size: 0.9em;
            margin-bottom: 5px;
        }

        .activity-time {
            color: #9ca3af;
            font-size: 0.85em;
        }

        /* Tabs */
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
            background: rgba(255,255,255,0.2);
            padding: 6px;
            border-radius: 15px;
            backdrop-filter: blur(10px);
        }

        .tab {
            flex: 1;
            padding: 12px 20px;
            background: transparent;
            border: none;
            border-radius: 12px;
            color: rgba(255,255,255,0.8);
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .tab:hover {
            color: white;
            background: rgba(255,255,255,0.15);
        }

        .tab.active {
            background: white;
            color: var(--primary);
            box-shadow: 0 4px 15px rgba(0,0,0,0.1);
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
            animation: fadeIn 0.4s ease;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Chart Container */
        .chart-container {
            position: relative;
            height: 300px;
            margin-top: 20px;
        }

        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
        }

        .empty-state i {
            font-size: 5em;
            color: #d1d5db;
            margin-bottom: 20px;
        }

        .empty-state h3 {
            color: #6b7280;
            font-size: 1.3em;
            margin-bottom: 10px;
        }

        .empty-state p {
            color: #9ca3af;
            margin-bottom: 25px;
        }

        /* Responsive */
        @media (max-width: 1024px) {
            .sidebar {
                width: 240px;
            }
            .content-area {
                margin-left: 240px;
            }
        }

        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                z-index: 999;
                background: rgba(102, 126, 234, 0.98);
                backdrop-filter: blur(30px);
            }

            .sidebar.mobile-open {
                transform: translateX(0);
            }

            .content-area {
                margin-left: 0;
            }
            
            .navbar {
                padding: 8px 0;
            }
            
            .navbar-content {
                padding: 0 15px;
            }
            
            .logo {
                font-size: 1.1em;
                max-height: 40px;
                gap: 8px;
            }
            
            .logo img {
                max-height: 35px;
                max-width: 140px;
            }
            
            .logo i {
                font-size: 1.4em;
            }
            
            .user-info {
                gap: 10px;
            }
            
            .user-name {
                display: none;
            }
            
            .user-avatar {
                width: 38px;
                height: 38px;
                font-size: 0.95em;
            }

            .welcome-header h1 {
                font-size: 1.8em;
            }

            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }

            .courses-grid {
                grid-template-columns: 1fr;
            }
        }

        /* Mobile Toggle */
        .mobile-toggle {
            display: none;
            position: fixed;
            bottom: 20px;
            right: 20px;
            width: 60px;
            height: 60px;
            background: var(--purple-gradient);
            border-radius: 50%;
            border: none;
            color: white;
            font-size: 1.5em;
            cursor: pointer;
            box-shadow: 0 8px 20px rgba(102, 126, 234, 0.5);
            z-index: 1001;
            transition: all 0.3s ease;
        }

        .mobile-toggle:hover {
            transform: scale(1.1);
        }

        @media (max-width: 768px) {
            .mobile-toggle {
                display: flex;
                align-items: center;
                justify-content: center;
            }
        }

        /* Loading Animation */
        .loading {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }

        @keyframes spin {
            to { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar">
        <div class="navbar-content">
            <div class="logo">
                @if (!empty($logoUrl))
                    <img src="{{ $logoUrl }}" alt="{{ $siteName }}">
                @else
                    <i class="fas fa-lightbulb logo-icon"></i>
                @endif
                <div class="logo-text">
                    <span>{{ $siteName }}</span>
                    @if (!empty($siteTagline))
                        <small>{{ $siteTagline }}</small>
                    @endif
                </div>
            </div>
            <div class="user-info">
                <span class="user-name">{{ Auth::user()->name }}</span>
                <div class="user-avatar">{{ strtoupper(substr(Auth::user()->name, 0, 1)) }}</div>
                
                @if(Auth::user()->hasRole('admin'))
                    <a href="{{ route('admin.panel') }}" class="btn" style="background: var(--purple-gradient); color: white; padding: 8px 16px; border-radius: 8px; text-decoration: none; margin-right: 10px; display: inline-flex; align-items: center; gap: 6px;">
                        <i class="fas fa-user-shield"></i>
                        Admin Panel
                    </a>
                @endif
                
                @if(Auth::user()->hasRole('tutor'))
                    <a href="{{ route('tutor.dashboard') }}" class="btn" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; padding: 8px 16px; border-radius: 8px; text-decoration: none; margin-right: 10px; display: inline-flex; align-items: center; gap: 6px;">
                        <i class="fas fa-chalkboard-teacher"></i>
                        Tutor Panel
                    </a>
                @endif
                
                <form action="{{ route('logout') }}" method="POST" style="display: inline;">
                    @csrf
                    <button type="submit" class="btn btn-logout">
                        <i class="fas fa-sign-out-alt"></i>
                        Logout
                    </button>
                </form>
            </div>
        </div>
    </nav>

    <div class="dashboard-container">
        <!-- Sidebar -->
        <aside class="sidebar" id="sidebar">
            <div class="sidebar-section">
                <h3>📚 Learning</h3>
                <ul class="sidebar-menu">
                    <li>
                        <a href="#" class="sidebar-nav active" data-section="dashboard">
                            <i class="fas fa-home"></i>
                            <span>Dashboard</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="my-courses">
                            <i class="fas fa-book-open"></i>
                            <span>My Courses</span>
                            <span class="badge">{{ count($enrolledCourses) }}</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="browse">
                            <i class="fas fa-compass"></i>
                            <span>Browse Courses</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="calendar">
                            <i class="fas fa-calendar-alt"></i>
                            <span>My Schedule</span>
                        </a>
                    </li>
                </ul>
            </div>

            <div class="sidebar-section">
                <h3>🏆 Progress</h3>
                <ul class="sidebar-menu">
                    <li>
                        <a href="#" class="sidebar-nav" data-section="achievements">
                            <i class="fas fa-trophy"></i>
                            <span>Achievements</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="certificates">
                            <i class="fas fa-certificate"></i>
                            <span>Certificates</span>
                        </a>
                    </li>
                    <li>
                        <a href="{{ route('payment.history') }}" class="sidebar-nav">
                            <i class="fas fa-credit-card"></i>
                            <span>Payment History</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="leaderboard">
                            <i class="fas fa-chart-line"></i>
                            <span>Leaderboard</span>
                        </a>
                    </li>
                </ul>
            </div>

            <div class="sidebar-section">
                <h3>👤 Account</h3>
                <ul class="sidebar-menu">
                    <li>
                        <a href="#" class="sidebar-nav" data-section="profile">
                            <i class="fas fa-user-circle"></i>
                            <span>My Profile</span>
                        </a>
                    </li>
                    <li>
                        <a href="#" class="sidebar-nav" data-section="settings">
                            <i class="fas fa-cog"></i>
                            <span>Settings</span>
                        </a>
                    </li>
                </ul>
            </div>
        </aside>

        <!-- Main Content -->
        <main class="content-area">
            <!-- Welcome Header -->
            <div class="welcome-header">
                <h1>Welcome back, {{ explode(' ', Auth::user()->name)[0] }}! 🚀</h1>
                <p>Ready to continue your learning journey? You're doing amazing!</p>
            </div>

            <!-- Stats Grid -->
            <div class="stats-grid">
                <div class="stat-card" onclick="showSection('my-courses')">
                    <div class="stat-icon">
                        <i class="fas fa-book"></i>
                    </div>
                    <div class="stat-value">{{ count($enrolledCourses) }}</div>
                    <div class="stat-label">Enrolled Courses</div>
                </div>

                <div class="stat-card" onclick="showSection('achievements')">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-value">
                        @php
                            $completedCount = 0;
                            foreach($enrolledCourses as $course) {
                                $totalTopics = 0;
                                $completedTopics = 0;
                                foreach($course->lessons as $lesson) {
                                    foreach($lesson->topics as $topic) {
                                        $totalTopics++;
                                        $topicProgress = $topic->userProgress()->where('user_id', Auth::id())->first();
                                        if ($topicProgress && $topicProgress->completed) {
                                            $completedTopics++;
                                        }
                                    }
                                }
                                if ($totalTopics > 0 && $completedTopics == $totalTopics) {
                                    $completedCount++;
                                }
                            }
                            echo $completedCount;
                        @endphp
                    </div>
                    <div class="stat-label">Completed</div>
                </div>

                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-spinner"></i>
                    </div>
                    <div class="stat-value">{{ count($enrolledCourses) - $completedCount }}</div>
                    <div class="stat-label">In Progress</div>
                </div>

                <div class="stat-card">
                    <div class="stat-icon">
                        <i class="fas fa-fire"></i>
                    </div>
                    <div class="stat-value">{{ $streak }}</div>
                    <div class="stat-label">Day Streak</div>
                </div>
            </div>

            <!-- Dashboard Section -->
            <section id="section-dashboard">
                <!-- Recent Activity -->
                <div class="section">
                    <h2 class="section-title">
                        <i class="fas fa-history"></i>
                        Recent Activity
                    </h2>
                    <div class="card">
                        @forelse($recentActivity as $activity)
                            <div class="activity-item">
                                <div class="activity-icon {{ $activity->completed ? 'completed' : 'in-progress' }}">
                                    <i class="fas fa-{{ $activity->completed ? 'check' : 'clock' }}"></i>
                                </div>
                                <div class="activity-details">
                                    <div class="activity-title">{{ $activity->topic->title }}</div>
                                    <div class="activity-course">{{ $activity->topic->lesson->course->title }}</div>
                                    <div class="activity-time">
                                        <i class="fas fa-clock"></i> {{ $activity->updated_at->diffForHumans() }}
                                    </div>
                                </div>
                            </div>
                        @empty
                            <div class="empty-state">
                                <i class="fas fa-inbox"></i>
                                <h3>No activity yet</h3>
                                <p>Start learning to see your progress here</p>
                            </div>
                        @endforelse
                    </div>
                </div>

                <!-- Performance Chart -->
                <div class="section">
                    <h2 class="section-title">
                        <i class="fas fa-chart-bar"></i>
                        Learning Progress
                    </h2>
                    <div class="card">
                        <div class="chart-container">
                            <canvas id="progressChart"></canvas>
                        </div>
                    </div>
                </div>

                <!-- Continue Learning -->
                <div class="section">
                    <h2 class="section-title">
                        <i class="fas fa-play-circle"></i>
                        Continue Learning
                    </h2>
                    <div class="courses-grid">
                        @php
                            $inProgressCourses = [];
                            foreach($enrolledCourses as $course) {
                                $totalTopics = 0;
                                $completedTopics = 0;
                                foreach($course->lessons as $lesson) {
                                    foreach($lesson->topics as $topic) {
                                        $totalTopics++;
                                        $topicProgress = $topic->userProgress()->where('user_id', Auth::id())->first();
                                        if ($topicProgress && $topicProgress->completed) {
                                            $completedTopics++;
                                        }
                                    }
                                }
                                if ($totalTopics > 0 && $completedTopics > 0 && $completedTopics < $totalTopics) {
                                    $course->progress_percentage = round(($completedTopics / $totalTopics) * 100);
                                    $course->completed_topics = $completedTopics;
                                    $course->total_topics = $totalTopics;
                                    $inProgressCourses[] = $course;
                                }
                            }
                        @endphp

                        @forelse($inProgressCourses as $course)
                            <div class="course-card">
                                <div class="course-thumbnail">
                                    <i class="fas fa-laptop-code"></i>
                                </div>
                                <div class="course-content">
                                    <h3 class="course-title">{{ $course->title }}</h3>
                                    <div class="course-meta">
                                        <span><i class="fas fa-book"></i> {{ $course->lessons->count() }} Lessons</span>
                                        <span><i class="fas fa-tasks"></i> {{ $course->total_topics }} Topics</span>
                                    </div>
                                    <div class="progress-bar-container">
                                        <div class="progress-bar" style="width: {{ $course->progress_percentage }}%"></div>
                                    </div>
                                    <div class="progress-text">{{ $course->progress_percentage }}% Complete</div>
                                    <div class="course-actions">
                                        <a href="{{ route('learn.course', $course->id) }}" class="btn-course btn-continue">
                                            <i class="fas fa-play"></i>
                                            Continue
                                        </a>
                                        <a href="{{ route('learn.course', $course->id) }}" class="btn-course btn-view">
                                            <i class="fas fa-info-circle"></i>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        @empty
                            <div class="empty-state">
                                <i class="fas fa-rocket"></i>
                                <h3>No courses in progress yet!</h3>
                                <p>Start learning by enrolling in a course</p>
                                <button class="btn btn-primary" onclick="showSection('browse')">
                                    <i class="fas fa-search"></i>
                                    Browse Courses
                                </button>
                            </div>
                        @endforelse
                    </div>
                </div>
            </section>

            <!-- My Courses Section -->
            <section id="section-my-courses" style="display: none;">
                <h2 class="section-title">
                    <i class="fas fa-book-open"></i>
                    My Courses ({{ count($enrolledCourses) }})
                </h2>

                @if(count($enrolledCourses) > 0)
                    <!-- Course Tabs -->
                    <div class="tabs">
                        <button class="tab active" onclick="showCourseTab('all')">All Courses</button>
                        <button class="tab" onclick="showCourseTab('in-progress')">In Progress</button>
                        <button class="tab" onclick="showCourseTab('completed')">Completed</button>
                    </div>

                    <!-- All Courses Tab -->
                    <div id="tab-all" class="tab-content active">
                        <div class="courses-grid">
                            @foreach($enrolledCourses as $course)
                                @php
                                    $totalTopics = 0;
                                    $completedTopics = 0;
                                    foreach($course->lessons as $lesson) {
                                        foreach($lesson->topics as $topic) {
                                            $totalTopics++;
                                            $topicProgress = $topic->userProgress()->where('user_id', Auth::id())->first();
                                            if ($topicProgress && $topicProgress->completed) {
                                                $completedTopics++;
                                            }
                                        }
                                    }
                                    $progressPercentage = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
                                @endphp
                                <div class="course-card">
                                    <div class="course-thumbnail">
                                        <i class="fas fa-code"></i>
                                    </div>
                                    <div class="course-content">
                                        <h3 class="course-title">{{ $course->title }}</h3>
                                        <div class="course-meta">
                                            <span><i class="fas fa-book"></i> {{ $course->lessons->count() }} Lessons</span>
                                            <span><i class="fas fa-tasks"></i> {{ $totalTopics }} Topics</span>
                                        </div>
                                        <div class="progress-bar-container">
                                            <div class="progress-bar" style="width: {{ $progressPercentage }}%"></div>
                                        </div>
                                        <div class="progress-text">{{ $progressPercentage }}% Complete</div>
                                        <div class="course-actions">
                                            <a href="{{ route('learn.course', $course->id) }}" class="btn-course btn-continue" style="text-decoration: none;">
                                                <i class="fas fa-play"></i>
                                                {{ $progressPercentage > 0 ? 'Continue' : 'Start' }}
                                            </a>
                                            <a href="{{ route('learn.course', $course->id) }}" class="btn-course btn-view" style="text-decoration: none;">
                                                <i class="fas fa-info-circle"></i>
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    </div>

                    <!-- In Progress Tab -->
                    <div id="tab-in-progress" class="tab-content">
                        <div class="courses-grid">
                            @php
                                $inProgressList = [];
                                foreach($enrolledCourses as $course) {
                                    $totalTopics = 0;
                                    $completedTopics = 0;
                                    foreach($course->lessons as $lesson) {
                                        foreach($lesson->topics as $topic) {
                                            $totalTopics++;
                                            $topicProgress = $topic->userProgress()->where('user_id', Auth::id())->first();
                                            if ($topicProgress && $topicProgress->completed) {
                                                $completedTopics++;
                                            }
                                        }
                                    }
                                    if ($totalTopics > 0 && $completedTopics > 0 && $completedTopics < $totalTopics) {
                                        $course->progress_percentage = round(($completedTopics / $totalTopics) * 100);
                                        $course->completed_topics = $completedTopics;
                                        $course->total_topics = $totalTopics;
                                        $inProgressList[] = $course;
                                    }
                                }
                            @endphp
                            @forelse($inProgressList as $course)
                                <div class="course-card">
                                    <div class="course-thumbnail">
                                        <i class="fas fa-laptop-code"></i>
                                    </div>
                                    <div class="course-content">
                                        <h3 class="course-title">{{ $course->title }}</h3>
                                        <div class="course-meta">
                                            <span><i class="fas fa-book"></i> {{ $course->lessons->count() }} Lessons</span>
                                            <span><i class="fas fa-tasks"></i> {{ $course->total_topics }} Topics</span>
                                        </div>
                                        <div class="progress-bar-container">
                                            <div class="progress-bar" style="width: {{ $course->progress_percentage }}%"></div>
                                        </div>
                                        <div class="progress-text">{{ $course->progress_percentage }}% Complete</div>
                                        <div class="course-actions">
                                            <a href="{{ route('learn.course', $course->id) }}" class="btn-course btn-continue">
                                                <i class="fas fa-play"></i>
                                                Continue
                                            </a>
                                        </div>
                                    </div>
                                </div>
                            @empty
                                <div class="empty-state">
                                    <i class="fas fa-hourglass-start"></i>
                                    <h3>No courses in progress</h3>
                                    <p>Start learning to see courses here!</p>
                                </div>
                            @endforelse
                        </div>
                    </div>

                    <!-- Completed Tab -->
                    <div id="tab-completed" class="tab-content">
                        <div class="empty-state">
                            <i class="fas fa-medal"></i>
                            <h3>No completed courses yet</h3>
                            <p>Keep learning to complete your first course!</p>
                        </div>
                    </div>
                @else
                    <div class="card">
                        <div class="empty-state">
                            <i class="fas fa-book-open"></i>
                            <h3>No Courses Enrolled</h3>
                            <p>You haven't enrolled in any courses yet. Start your learning journey!</p>
                            <button class="btn btn-primary" onclick="showSection('browse')">
                                <i class="fas fa-search"></i>
                                Browse Courses
                            </button>
                        </div>
                    </div>
                @endif
            </section>

            <!-- Achievements Section -->
            <section id="section-achievements" style="display: none;">
                <h2 class="section-title">
                    <i class="fas fa-trophy"></i>
                    Achievements & Badges
                </h2>
                <div class="card">
                    <div class="achievements-grid">
                        <div class="achievement-badge earned">
                            <span class="badge-icon">🎯</span>
                            <div class="badge-name">First Steps</div>
                        </div>
                        <div class="achievement-badge earned">
                            <span class="badge-icon">📚</span>
                            <div class="badge-name">Bookworm</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">⭐</span>
                            <div class="badge-name">Rising Star</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">🔥</span>
                            <div class="badge-name">Hot Streak</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">🏆</span>
                            <div class="badge-name">Champion</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">💎</span>
                            <div class="badge-name">Diamond</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">🚀</span>
                            <div class="badge-name">Rocket</div>
                        </div>
                        <div class="achievement-badge">
                            <span class="badge-icon">👑</span>
                            <div class="badge-name">Master</div>
                        </div>
                    </div>
                </div>
            </section>

            <!-- Browse Courses Section -->
            <section id="section-browse" style="display: none;">
                <h2 class="section-title"><i class="fas fa-compass"></i> Browse Courses</h2>
                
                <!-- Show preview of available courses -->
                <div class="courses-grid" style="margin-bottom: 30px;">
                    @php
                        $allCourses = \App\Models\Course::where('is_active', true)
                            ->with(['lessons.topics'])
                            ->limit(4)
                            ->get();
                    @endphp
                    
                    @foreach($allCourses as $course)
                        @php
                            $totalTopics = $course->lessons->sum(function($lesson) {
                                return $lesson->topics->count();
                            });
                            $enrollment = \App\Models\CourseEnrollment::where('user_id', Auth::id())
                                ->where('course_id', $course->id)
                                ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
                                ->first();
                            $isEnrolled = $enrollment !== null;
                            $canEnroll = true;
                            $prerequisiteMessage = '';
                            
                            // Check prerequisite
                            if ($course->has_prerequisite && $course->prerequisite_course_id) {
                                $prerequisiteCourse = \App\Models\Course::find($course->prerequisite_course_id);
                                $prerequisiteEnrollment = \App\Models\CourseEnrollment::where('user_id', Auth::id())
                                    ->where('course_id', $course->prerequisite_course_id)
                                    ->where('status', 'completed')
                                    ->first();
                                
                                if (!$prerequisiteEnrollment) {
                                    $canEnroll = false;
                                    $prerequisiteMessage = "Requires: {$prerequisiteCourse->name}";
                                }
                            }
                        @endphp
                        <div class="course-card">
                            <div class="course-thumbnail">
                                <i class="fas fa-graduation-cap"></i>
                            </div>
                            <div class="course-content">
                                <h3 class="course-title">{{ $course->title }}</h3>
                                <p style="color: #6b7280; font-size: 0.9em; margin: 10px 0;">{{ Str::limit($course->description ?? 'Learn and master new skills', 80) }}</p>
                                <div class="course-meta">
                                    <span><i class="fas fa-book"></i> {{ $course->lessons->count() }} Lessons</span>
                                    <span><i class="fas fa-tasks"></i> {{ $totalTopics }} Topics</span>
                                </div>
                                <div style="margin-top: 15px; display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
                                    @if($isEnrolled)
                                        <span style="display: inline-block; padding: 5px 12px; background: #10b981; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-check-circle"></i> Enrolled
                                        </span>
                                        <a href="{{ route('learn.course', $course) }}" class="btn btn-primary" style="flex: 1; padding: 10px 15px; text-align: center; text-decoration: none;">
                                            <i class="fas fa-play-circle"></i> Continue Learning
                                        </a>
                                    @elseif(!$canEnroll)
                                        <span style="display: inline-block; padding: 5px 12px; background: #f59e0b; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-lock"></i> Locked
                                        </span>
                                        <span style="color: #6b7280; font-size: 0.85em; flex: 1;">{{ $prerequisiteMessage }}</span>
                                    @elseif(!$course->is_free)
                                        <span style="display: inline-block; padding: 5px 12px; background: #667eea; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-tag"></i> ${{ number_format($course->price, 2) }}
                                        </span>
                                        <a class="btn btn-primary" style="flex: 1; padding: 10px 15px; text-align: center;" href="{{ route('payment.checkout', $course->id) }}">
                                            <i class="fas fa-shopping-cart"></i> Purchase Course
                                        </a>
                                    @else
                                        <span style="display: inline-block; padding: 5px 12px; background: #10b981; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-gift"></i> Free
                                        </span>
                                        <form action="{{ route('student.enroll', $course) }}" method="POST" style="flex: 1;">
                                            @csrf
                                            <button type="submit" class="btn btn-primary" style="width: 100%; padding: 10px 15px;">
                                                <i class="fas fa-plus-circle"></i> Enroll Now
                                            </button>
                                        </form>
                                    @endif
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
                
                <div style="text-align: center; margin-top: 30px;">
                    <a href="#" data-section="browse-all" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: 10px; font-size: 1.1em; padding: 15px 30px;">
                        <i class="fas fa-th"></i>
                        Browse All Courses
                    </a>
                </div>
            </section>

            <section id="section-calendar" style="display: none;">
                <h2 class="section-title"><i class="fas fa-calendar-alt"></i> My Schedule</h2>
                
                <!-- Show upcoming topics -->
                @if(count($nextTopics) > 0)
                    <div class="card" style="margin-bottom: 30px;">
                        <h3 style="font-size: 1.3em; margin-bottom: 20px; color: #333;">
                            <i class="fas fa-list-check"></i> Up Next
                        </h3>
                        @foreach(array_slice($nextTopics, 0, 3) as $item)
                            <div style="padding: 15px; border-left: 4px solid #667eea; background: #f9fafb; margin-bottom: 15px; border-radius: 8px;">
                                <div style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                    {{ $item['topic']->title }}
                                </div>
                                <div style="color: #6b7280; font-size: 0.9em; margin-bottom: 5px;">
                                    <i class="fas fa-book"></i> {{ $item['course']->title }} • {{ $item['lesson']->title }}
                                </div>
                                <span style="display: inline-block; padding: 3px 10px; background: {{ $item['status'] == 'in-progress' ? '#f59e0b' : '#667eea' }}; color: white; border-radius: 12px; font-size: 0.8em;">
                                    {{ ucfirst(str_replace('-', ' ', $item['status'])) }}
                                </span>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="card" style="margin-bottom: 30px;">
                        <div class="empty-state">
                            <i class="fas fa-calendar-check"></i>
                            <h3>No upcoming topics</h3>
                            <p>Start a course to see your schedule</p>
                        </div>
                    </div>
                @endif
                
                <div style="text-align: center; margin-top: 30px;">
                    <a href="#" data-section="schedule-all" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: 10px; font-size: 1.1em; padding: 15px 30px;">
                        <i class="fas fa-calendar-alt"></i>
                        View Full Schedule
                    </a>
                </div>
            </section>

            <section id="section-certificates" style="display: none;">
                <h2 class="section-title"><i class="fas fa-certificate"></i> My Certificates</h2>
                
                <!-- Show certificates if any -->
                @if(count($certificates) > 0)
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px; margin-bottom: 30px;">
                        @foreach($certificates->take(4) as $cert)
                            <div class="card" style="padding: 25px; text-align: center; border: 2px solid #667eea;">
                                <div style="font-size: 3em; color: #667eea; margin-bottom: 15px;">
                                    <i class="fas fa-award"></i>
                                </div>
                                <h3 style="font-size: 1.2em; color: #333; margin-bottom: 10px;">{{ $cert->course->title }}</h3>
                                <p style="color: #6b7280; font-size: 0.9em; margin-bottom: 15px;">
                                    <i class="fas fa-calendar"></i> Issued {{ $cert->issued_at->format('M d, Y') }}
                                </p>
                                <a href="{{ route('student.certificate.view', $cert->id) }}" class="btn btn-primary" style="width: 100%;">
                                    <i class="fas fa-eye"></i> View
                                </a>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="card" style="margin-bottom: 30px;">
                        <div class="empty-state">
                            <i class="fas fa-certificate"></i>
                            <h3>No certificates yet</h3>
                            <p>Complete courses to earn certificates</p>
                        </div>
                    </div>
                @endif
                
                <div style="text-align: center; margin-top: 30px;">
                    <a href="#" data-section="certificates-all" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: 10px; font-size: 1.1em; padding: 15px 30px;">
                        <i class="fas fa-certificate"></i>
                        View All Certificates
                    </a>
                </div>
            </section>

            <section id="section-leaderboard" style="display: none;">
                <h2 class="section-title"><i class="fas fa-chart-line"></i> Leaderboard</h2>
                
                <!-- Show top learners -->
                @if(isset($leaderboardUsers) && count($leaderboardUsers) > 0)
                    <div class="card" style="margin-bottom: 30px;">
                        <h3 style="font-size: 1.3em; margin-bottom: 20px; color: #333;">
                            <i class="fas fa-trophy"></i> Top Learners
                        </h3>
                        @foreach($leaderboardUsers->take(5) as $user)
                            <div style="display: flex; align-items: center; padding: 15px; background: {{ $user->id == Auth::id() ? '#f0f9ff' : '#f9fafb' }}; margin-bottom: 10px; border-radius: 10px; border-left: 4px solid {{ $user->rank <= 3 ? '#f59e0b' : '#667eea' }};">
                                <div style="font-size: 1.5em; font-weight: 700; color: {{ $user->rank <= 3 ? '#f59e0b' : '#667eea' }}; width: 50px;">
                                    #{{ $user->rank }}
                                </div>
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; color: #333;">
                                        {{ $user->name }}
                                        @if($user->id == Auth::id())
                                            <span style="color: #667eea;">(You)</span>
                                        @endif
                                    </div>
                                    <div style="color: #6b7280; font-size: 0.9em;">
                                        {{ $user->total_points }} points • {{ $user->completed_topics }} topics completed
                                    </div>
                                </div>
                                @if($user->rank == 1)
                                    <div style="font-size: 2em;">🥇</div>
                                @elseif($user->rank == 2)
                                    <div style="font-size: 2em;">🥈</div>
                                @elseif($user->rank == 3)
                                    <div style="font-size: 2em;">🥉</div>
                                @endif
                            </div>
                        @endforeach
                    </div>
                    
                    @if($currentUserRank && $currentUserRank->rank > 5)
                        <div class="card" style="margin-bottom: 30px; background: #f0f9ff; border: 2px solid #667eea;">
                            <h4 style="margin-bottom: 15px; color: #667eea;">Your Rank</h4>
                            <div style="display: flex; align-items: center; padding: 15px; background: white; border-radius: 10px;">
                                <div style="font-size: 1.5em; font-weight: 700; color: #667eea; width: 50px;">
                                    #{{ $currentUserRank->rank }}
                                </div>
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; color: #333;">{{ $currentUserRank->name }}</div>
                                    <div style="color: #6b7280; font-size: 0.9em;">
                                        {{ $currentUserRank->total_points }} points • {{ $currentUserRank->completed_topics }} topics completed
                                    </div>
                                </div>
                            </div>
                        </div>
                    @endif
                @else
                    <div class="card" style="margin-bottom: 30px;">
                        <div class="empty-state">
                            <i class="fas fa-trophy"></i>
                            <h3>No rankings yet</h3>
                            <p>Start learning to compete with others</p>
                        </div>
                    </div>
                @endif
                
                <div style="text-align: center; margin-top: 30px;">
                    <a href="#" data-section="leaderboard-all" class="btn btn-primary" style="display: inline-flex; align-items: center; gap: 10px; font-size: 1.1em; padding: 15px 30px;">
                        <i class="fas fa-ranking-star"></i>
                        View Full Leaderboard
                    </a>
                </div>
            </section>

            <!-- Full Browse All Courses Section -->
            <section id="section-browse-all" style="display: none;">
                <h2 class="section-title"><i class="fas fa-th"></i> All Courses</h2>
                
                @php
                    $allCoursesComplete = \App\Models\Course::where('is_active', true)
                        ->with(['lessons.topics'])
                        ->get();
                @endphp
                
                <div class="courses-grid">
                    @foreach($allCoursesComplete as $course)
                        @php
                            $totalTopics = $course->lessons->sum(function($lesson) {
                                return $lesson->topics->count();
                            });
                            $enrollment = \App\Models\CourseEnrollment::where('user_id', Auth::id())
                                ->where('course_id', $course->id)
                                ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
                                ->first();
                            $isEnrolled = $enrollment !== null;
                            $canEnroll = true;
                            $prerequisiteMessage = '';
                            
                            // Check prerequisite
                            if ($course->has_prerequisite && $course->prerequisite_course_id) {
                                $prerequisiteCourse = \App\Models\Course::find($course->prerequisite_course_id);
                                $prerequisiteEnrollment = \App\Models\CourseEnrollment::where('user_id', Auth::id())
                                    ->where('course_id', $course->prerequisite_course_id)
                                    ->where('status', 'completed')
                                    ->first();
                                
                                if (!$prerequisiteEnrollment) {
                                    $canEnroll = false;
                                    $prerequisiteMessage = "Requires: {$prerequisiteCourse->name}";
                                }
                            }
                        @endphp
                        <div class="course-card">
                            <div class="course-thumbnail">
                                <i class="fas fa-graduation-cap"></i>
                            </div>
                            <div class="course-content">
                                <h3 class="course-title">{{ $course->title }}</h3>
                                <p style="color: #6b7280; font-size: 0.9em; margin: 10px 0;">{{ Str::limit($course->description ?? 'Learn and master new skills', 100) }}</p>
                                <div class="course-meta">
                                    <span><i class="fas fa-book"></i> {{ $course->lessons->count() }} Lessons</span>
                                    <span><i class="fas fa-tasks"></i> {{ $totalTopics }} Topics</span>
                                </div>
                                <div style="margin-top: 15px; display: flex; gap: 10px; align-items: center; flex-wrap: wrap;">
                                    @if($isEnrolled)
                                        <span style="display: inline-block; padding: 5px 12px; background: #10b981; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-check-circle"></i> Enrolled
                                        </span>
                                        <a href="{{ route('learn.course', $course) }}" class="btn btn-primary" style="flex: 1; padding: 10px 15px; text-align: center; text-decoration: none;">
                                            <i class="fas fa-play-circle"></i> Continue Learning
                                        </a>
                                    @elseif(!$canEnroll)
                                        <span style="display: inline-block; padding: 5px 12px; background: #f59e0b; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-lock"></i> Locked
                                        </span>
                                        <span style="color: #6b7280; font-size: 0.85em; flex: 1;">{{ $prerequisiteMessage }}</span>
                                    @elseif(!$course->is_free)
                                        <span style="display: inline-block; padding: 5px 12px; background: #667eea; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-tag"></i> ${{ number_format($course->price, 2) }}
                                        </span>
                                        <button class="btn btn-primary" style="flex: 1; padding: 10px 15px; background: #9ca3af; cursor: not-allowed;" disabled title="Payment system coming soon">
                                            <i class="fas fa-shopping-cart"></i> Purchase (Coming Soon)
                                        </button>
                                    @else
                                        <span style="display: inline-block; padding: 5px 12px; background: #10b981; color: white; border-radius: 20px; font-size: 0.85em;">
                                            <i class="fas fa-gift"></i> Free
                                        </span>
                                        <form action="{{ route('student.enroll', $course) }}" method="POST" style="flex: 1;">
                                            @csrf
                                            <button type="submit" class="btn btn-primary" style="width: 100%; padding: 10px 15px;">
                                                <i class="fas fa-plus-circle"></i> Enroll Now
                                            </button>
                                        </form>
                                    @endif
                                </div>
                            </div>
                        </div>
                    @endforeach
                </div>
            </section>

            <!-- Full Schedule Section -->
            <section id="section-schedule-all" style="display: none;">
                <h2 class="section-title"><i class="fas fa-calendar-check"></i> My Complete Schedule</h2>
                
                @if(count($nextTopics) > 0)
                    <div class="card" style="margin-bottom: 30px;">
                        <h3 style="font-size: 1.4em; margin-bottom: 20px; color: #333;">
                            <i class="fas fa-list-check"></i> Upcoming Topics
                        </h3>
                        @foreach($nextTopics as $item)
                            <div style="padding: 20px; border-left: 4px solid #667eea; background: #f9fafb; margin-bottom: 15px; border-radius: 8px;">
                                <div style="font-weight: 600; color: #333; font-size: 1.1em; margin-bottom: 8px;">
                                    {{ $item['topic']->title }}
                                </div>
                                <div style="color: #6b7280; font-size: 0.95em; margin-bottom: 10px;">
                                    <i class="fas fa-book"></i> {{ $item['course']->title }} • {{ $item['lesson']->title }}
                                </div>
                                <div style="display: flex; align-items: center; gap: 10px;">
                                    <span style="display: inline-block; padding: 5px 12px; background: {{ $item['status'] == 'in-progress' ? '#f59e0b' : '#667eea' }}; color: white; border-radius: 12px; font-size: 0.85em;">
                                        {{ ucfirst(str_replace('-', ' ', $item['status'])) }}
                                    </span>
                                    @if(isset($item['due_date']))
                                        <span style="color: #6b7280; font-size: 0.9em;">
                                            <i class="fas fa-clock"></i> Due: {{ $item['due_date'] }}
                                        </span>
                                    @endif
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="card">
                        <div class="empty-state">
                            <i class="fas fa-calendar-check"></i>
                            <h3>No upcoming topics</h3>
                            <p>Start a course to see your schedule</p>
                        </div>
                    </div>
                @endif
            </section>

            <!-- Full Certificates Section -->
            <section id="section-certificates-all" style="display: none;">
                <h2 class="section-title"><i class="fas fa-award"></i> All My Certificates</h2>
                
                @if(count($certificates) > 0)
                    <div style="display: grid; grid-template-columns: repeat(auto-fill, minmax(300px, 1fr)); gap: 20px;">
                        @foreach($certificates as $cert)
                            <div class="card" style="padding: 25px; text-align: center; border: 2px solid #667eea;">
                                <div style="font-size: 3em; color: #667eea; margin-bottom: 15px;">
                                    <i class="fas fa-award"></i>
                                </div>
                                <h3 style="font-size: 1.2em; color: #333; margin-bottom: 10px;">{{ $cert->course->title }}</h3>
                                <p style="color: #6b7280; font-size: 0.9em; margin-bottom: 15px;">
                                    <i class="fas fa-calendar"></i> Issued {{ $cert->issued_at->format('M d, Y') }}
                                </p>
                                <div style="display: flex; gap: 10px; justify-content: center;">
                                    <a href="{{ route('student.certificate.view', $cert->id) }}" class="btn btn-primary" style="flex: 1;">
                                        <i class="fas fa-eye"></i> View
                                    </a>
                                    <a href="{{ route('student.certificate.download', $cert->id) }}" class="btn btn-primary" style="flex: 1; background: #10b981;">
                                        <i class="fas fa-download"></i> Download
                                    </a>
                                </div>
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="card">
                        <div class="empty-state">
                            <i class="fas fa-certificate"></i>
                            <h3>No certificates yet</h3>
                            <p>Complete courses to earn certificates</p>
                        </div>
                    </div>
                @endif
            </section>

            <!-- Full Leaderboard Section -->
            <section id="section-leaderboard-all" style="display: none;">
                <h2 class="section-title"><i class="fas fa-trophy"></i> Complete Leaderboard</h2>
                
                @if(isset($leaderboardUsers) && count($leaderboardUsers) > 0)
                    <!-- Top 3 Podium -->
                    <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px; margin-bottom: 30px;">
                        @foreach($leaderboardUsers->take(3) as $user)
                            <div class="card" style="text-align: center; padding: 30px; background: linear-gradient(135deg, {{ $user->rank == 1 ? '#fbbf24, #f59e0b' : ($user->rank == 2 ? '#d1d5db, #9ca3af' : '#c2410c, #ea580c') }});">
                                <div style="font-size: 4em; margin-bottom: 10px;">
                                    @if($user->rank == 1) 🥇
                                    @elseif($user->rank == 2) 🥈
                                    @else 🥉
                                    @endif
                                </div>
                                <div style="font-size: 2.5em; font-weight: 700; color: white; margin-bottom: 10px;">#{{ $user->rank }}</div>
                                <h3 style="color: white; font-size: 1.3em; margin-bottom: 5px;">{{ $user->name }}</h3>
                                <div style="color: rgba(255,255,255,0.9); font-weight: 600;">{{ $user->total_points }} points</div>
                                <div style="color: rgba(255,255,255,0.8); font-size: 0.9em;">{{ $user->completed_topics }} topics</div>
                            </div>
                        @endforeach
                    </div>

                    <!-- Rest of Rankings -->
                    <div class="card">
                        <h3 style="font-size: 1.3em; margin-bottom: 20px; color: #333;">
                            <i class="fas fa-list-ol"></i> All Rankings
                        </h3>
                        @foreach($leaderboardUsers as $user)
                            <div style="display: flex; align-items: center; padding: 15px; background: {{ $user->id == Auth::id() ? '#f0f9ff' : ($loop->index % 2 == 0 ? '#ffffff' : '#f9fafb') }}; margin-bottom: 5px; border-radius: 8px; border-left: 4px solid {{ $user->rank <= 3 ? '#f59e0b' : '#667eea' }};">
                                <div style="font-size: 1.3em; font-weight: 700; color: {{ $user->rank <= 3 ? '#f59e0b' : '#667eea' }}; width: 60px;">
                                    #{{ $user->rank }}
                                </div>
                                <div style="flex: 1;">
                                    <div style="font-weight: 600; color: #333; font-size: 1.05em;">
                                        {{ $user->name }}
                                        @if($user->id == Auth::id())
                                            <span style="color: #667eea; font-weight: 700;">(You)</span>
                                        @endif
                                    </div>
                                    <div style="color: #6b7280; font-size: 0.9em;">
                                        <i class="fas fa-star"></i> {{ $user->total_points }} points • 
                                        <i class="fas fa-check-circle"></i> {{ $user->completed_topics }} topics completed
                                    </div>
                                </div>
                                @if($user->rank == 1)
                                    <div style="font-size: 2em;">🥇</div>
                                @elseif($user->rank == 2)
                                    <div style="font-size: 2em;">🥈</div>
                                @elseif($user->rank == 3)
                                    <div style="font-size: 2em;">🥉</div>
                                @endif
                            </div>
                        @endforeach
                    </div>
                @else
                    <div class="card">
                        <div class="empty-state">
                            <i class="fas fa-trophy"></i>
                            <h3>No rankings yet</h3>
                            <p>Start learning to compete with others</p>
                        </div>
                    </div>
                @endif
            </section>

            <section id="section-profile" style="display: none;">
                <h2 class="section-title"><i class="fas fa-user-circle"></i> My Profile</h2>
                
                <!-- Profile Avatar -->
                <div class="card" style="margin-bottom: 30px;">
                    <div style="text-align: center; padding: 40px;">
                        <div style="width: 150px; height: 150px; border-radius: 50%; background: var(--purple-gradient); margin: 0 auto 20px; display: flex; align-items: center; justify-content: center; font-size: 4em; color: white; font-weight: 700; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
                            @if(Auth::user()->avatar)
                                <img src="{{ Storage::url(Auth::user()->avatar) }}" alt="Avatar" style="width: 100%; height: 100%; border-radius: 50%; object-fit: cover;">
                            @else
                                {{ strtoupper(substr(Auth::user()->name, 0, 1)) }}
                            @endif
                        </div>
                        <h3 style="font-size: 2em; color: #333; margin-bottom: 10px;">{{ Auth::user()->name }}</h3>
                        <p style="color: #6b7280; font-size: 1.1em; margin-bottom: 20px;">{{ Auth::user()->email }}</p>
                        <a href="#" data-section="edit-profile" class="btn btn-primary">
                            <i class="fas fa-user-edit"></i> Edit Full Profile
                        </a>
                    </div>
                </div>

                <!-- Quick Stats -->
                <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 20px; margin-bottom: 30px;">
                    <div class="card" style="text-align: center; padding: 30px;">
                        <div style="font-size: 3em; margin-bottom: 10px;">📚</div>
                        <div style="font-size: 2em; font-weight: 700; color: #667eea;">{{ count($enrolledCourses) }}</div>
                        <div style="color: #6b7280; font-weight: 600;">Enrolled Courses</div>
                    </div>
                    <div class="card" style="text-align: center; padding: 30px;">
                        <div style="font-size: 3em; margin-bottom: 10px;">🏆</div>
                        <div style="font-size: 2em; font-weight: 700; color: #667eea;">{{ Auth::user()->certificates()->count() }}</div>
                        <div style="color: #6b7280; font-weight: 600;">Certificates Earned</div>
                    </div>
                    <div class="card" style="text-align: center; padding: 30px;">
                        <div style="font-size: 3em; margin-bottom: 10px;">🔥</div>
                        <div style="font-size: 2em; font-weight: 700; color: #667eea;">{{ $streak }}</div>
                        <div style="color: #6b7280; font-weight: 600;">Day Streak</div>
                    </div>
                </div>

                <!-- Profile Information -->
                <div class="card">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-info-circle"></i> Personal Information
                    </h3>
                    <div style="display: grid; gap: 20px;">
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Full Name:</span>
                            <span style="color: #333; font-weight: 600;">{{ Auth::user()->name }}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Email:</span>
                            <span style="color: #333; font-weight: 600;">{{ Auth::user()->email }}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Phone:</span>
                            <span style="color: #333; font-weight: 600;">{{ Auth::user()->phone ?? 'Not set' }}</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Member Since:</span>
                            <span style="color: #333; font-weight: 600;">{{ Auth::user()->created_at->format('F j, Y') }}</span>
                        </div>
                    </div>
                    <div style="margin-top: 30px; text-align: center;">
                        <a href="#" data-section="edit-profile" class="btn btn-primary">
                            <i class="fas fa-edit"></i> Update Profile Information
                        </a>
                    </div>
                </div>
            </section>

            <!-- Edit Profile Section -->
            <section id="section-edit-profile" style="display: none;">
                <h2 class="section-title"><i class="fas fa-user-edit"></i> Edit Profile</h2>
                
                <!-- Profile Form -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-id-card"></i> Personal Information
                    </h3>
                    <form action="{{ route('student.profile.update') }}" method="POST" class="ajax-form">
                        @csrf
                        <div style="display: grid; gap: 20px;">
                            <!-- Full Name -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-user"></i> Full Name *
                                </label>
                                <input type="text" name="name" value="{{ Auth::user()->name }}" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            
                            <!-- Email -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-envelope"></i> Email Address *
                                </label>
                                <input type="email" name="email" value="{{ Auth::user()->email }}" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            
                            <!-- Phone -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-phone"></i> Phone Number
                                </label>
                                <input type="tel" name="phone" value="{{ Auth::user()->phone ?? '' }}" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;" placeholder="+1 (555) 000-0000">
                            </div>
                            
                            <!-- Location -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-map-marker-alt"></i> Location
                                </label>
                                <input type="text" name="location" value="{{ Auth::user()->location ?? '' }}" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;" placeholder="City, Country">
                            </div>
                            
                            <!-- Bio -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-pen"></i> Bio
                                </label>
                                <textarea name="bio" rows="4" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; resize: vertical; background: white; pointer-events: auto;" placeholder="Tell us about yourself...">{{ Auth::user()->bio ?? '' }}</textarea>
                                <small style="color: #6b7280;">Maximum 500 characters</small>
                            </div>
                        </div>
                        
                        <div style="margin-top: 30px; display: flex; gap: 15px; justify-content: center;">
                            <button type="submit" class="btn btn-primary" style="padding: 12px 30px;">
                                <i class="fas fa-save"></i> Save Changes
                            </button>
                            <a href="#" data-section="profile" class="btn btn-secondary" style="padding: 12px 30px;">
                                <i class="fas fa-times"></i> Cancel
                            </a>
                        </div>
                    </form>
                </div>
                
                <!-- Avatar Upload -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-camera"></i> Profile Picture
                    </h3>
                    <div style="text-align: center;">
                        <div style="width: 150px; height: 150px; border-radius: 50%; background: var(--purple-gradient); margin: 0 auto 20px; display: flex; align-items: center; justify-content: center; font-size: 4em; color: white; font-weight: 700; box-shadow: 0 10px 30px rgba(0,0,0,0.2);">
                            @if(Auth::user()->avatar)
                                <img src="{{ Storage::url(Auth::user()->avatar) }}" alt="Avatar" style="width: 100%; height: 100%; border-radius: 50%; object-fit: cover;">
                            @else
                                {{ strtoupper(substr(Auth::user()->name, 0, 1)) }}
                            @endif
                        </div>
                        
                        <form action="{{ route('student.profile.avatar') }}" method="POST" enctype="multipart/form-data" class="ajax-form">
                            @csrf
                            <input type="file" name="avatar" id="avatarInput" accept="image/*" style="display: none;" onchange="this.form.submit()">
                            <button type="button" class="btn btn-primary" onclick="document.getElementById('avatarInput').click()">
                                <i class="fas fa-upload"></i> Upload New Picture
                            </button>
                            <p style="color: #6b7280; margin-top: 10px; font-size: 0.9em;">JPG, PNG or GIF (max 2MB)</p>
                        </form>
                    </div>
                </div>
                
                <!-- Change Password -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-key"></i> Change Password
                    </h3>
                    <form action="{{ route('student.profile.password') }}" method="POST" class="ajax-form">
                        @csrf
                        <div style="display: grid; gap: 20px;">
                            <!-- Current Password -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-lock"></i> Current Password *
                                </label>
                                <input type="password" name="current_password" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            
                            <!-- New Password -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-key"></i> New Password *
                                </label>
                                <input type="password" name="new_password" required minlength="8" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                                <small style="color: #6b7280;">Minimum 8 characters</small>
                            </div>
                            
                            <!-- Confirm New Password -->
                            <div>
                                <label style="display: block; color: #333; font-weight: 600; margin-bottom: 8px;">
                                    <i class="fas fa-check-circle"></i> Confirm New Password *
                                </label>
                                <input type="password" name="new_password_confirmation" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                        </div>
                        
                        <div style="margin-top: 30px; text-align: center;">
                            <button type="submit" class="btn btn-primary" style="padding: 12px 30px;">
                                <i class="fas fa-save"></i> Update Password
                            </button>
                        </div>
                    </form>
                </div>
            </section>

            <section id="section-settings" style="display: none;">
                <h2 class="section-title"><i class="fas fa-cog"></i> Settings</h2>
                
                <!-- Notification Settings -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-bell"></i> Notification Preferences
                    </h3>
                    <form id="settingsForm" action="{{ route('student.settings.update') }}" method="POST">
                        @csrf
                        <div style="display: grid; gap: 25px;">
                            <!-- Email Notifications -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-envelope"></i> Email Notifications
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Receive email updates about your courses and activities</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="email_notifications" value="1" 
                                           {{ (Auth::user()->settings['email_notifications'] ?? true) ? 'checked' : '' }}>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>

                            <!-- Course Updates -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-book"></i> Course Updates
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Get notified when new content is added to your courses</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="course_updates" value="1"
                                           {{ (Auth::user()->settings['course_updates'] ?? true) ? 'checked' : '' }}>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>

                            <!-- Achievement Notifications -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-trophy"></i> Achievement Alerts
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Celebrate your milestones and earned badges</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="achievement_notifications" value="1"
                                           {{ (Auth::user()->settings['achievement_notifications'] ?? true) ? 'checked' : '' }}>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>

                            <!-- Weekly Digest -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-calendar-week"></i> Weekly Digest
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Get a weekly summary of your learning progress</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="weekly_digest" value="1"
                                           {{ (Auth::user()->settings['weekly_digest'] ?? false) ? 'checked' : '' }}>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>

                            <!-- Marketing Emails -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-bullhorn"></i> Marketing Communications
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Receive news about new courses and features</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="marketing_emails" value="1"
                                           {{ (Auth::user()->settings['marketing_emails'] ?? false) ? 'checked' : '' }}>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>
                        </div>

                        <div style="margin-top: 30px; text-align: center;">
                            <button type="submit" class="btn btn-primary" style="padding: 12px 40px;">
                                <i class="fas fa-save"></i> Save Settings
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Privacy & Security -->
                <div class="card">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-shield-alt"></i> Privacy & Security
                    </h3>
                    <div style="display: grid; gap: 20px;">
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Profile Visibility:</span>
                            <span style="color: #333; font-weight: 600;">Public</span>
                        </div>
                        <div style="display: flex; justify-content: space-between; padding: 15px; background: #f9fafb; border-radius: 10px;">
                            <span style="color: #6b7280; font-weight: 600;">Show on Leaderboard:</span>
                            <span style="color: #333; font-weight: 600;">Enabled</span>
                        </div>
                    </div>
                    <div style="margin-top: 30px; text-align: center;">
                        <a href="#" data-section="advanced-settings" class="btn btn-secondary">
                            <i class="fas fa-cog"></i> Advanced Settings
                        </a>
                    </div>
                </div>
            </section>

            <!-- Advanced Settings Section -->
            <section id="section-advanced-settings" style="display: none;">
                <h2 class="section-title"><i class="fas fa-cog"></i> Advanced Settings</h2>
                
                <!-- Account Security -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-shield-alt"></i> Security Settings
                    </h3>
                    
                    <!-- Change Password -->
                    <div style="padding: 20px; background: #f9fafb; border-radius: 10px; margin-bottom: 20px;">
                        <h4 style="font-weight: 600; color: #333; margin-bottom: 15px;">
                            <i class="fas fa-key"></i> Change Password
                        </h4>
                        <form action="{{ route('student.profile.password') }}" method="POST" class="ajax-form">
                            @csrf
                            <div style="margin-bottom: 15px;">
                                <label style="display: block; color: #6b7280; font-weight: 600; margin-bottom: 5px;">Current Password</label>
                                <input type="password" name="current_password" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            <div style="margin-bottom: 15px;">
                                <label style="display: block; color: #6b7280; font-weight: 600; margin-bottom: 5px;">New Password</label>
                                <input type="password" name="new_password" required minlength="8" style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            <div style="margin-bottom: 15px;">
                                <label style="display: block; color: #6b7280; font-weight: 600; margin-bottom: 5px;">Confirm New Password</label>
                                <input type="password" name="new_password_confirmation" required style="width: 100%; padding: 12px; border: 2px solid #e5e7eb; border-radius: 8px; font-size: 1em; background: white; pointer-events: auto;">
                            </div>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-lock"></i> Update Password
                            </button>
                        </form>
                    </div>
                    
                    <!-- Two-Factor Authentication -->
                    <div style="padding: 20px; background: #f9fafb; border-radius: 10px;">
                        <div style="display: flex; justify-content: space-between; align-items: center;">
                            <div>
                                <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                    <i class="fas fa-mobile-alt"></i> Two-Factor Authentication
                                </h4>
                                <p style="color: #6b7280; font-size: 0.95em;">Add an extra layer of security to your account</p>
                            </div>
                            <button class="btn btn-secondary" disabled>
                                <i class="fas fa-plus"></i> Enable
                            </button>
                        </div>
                    </div>
                </div>
                
                <!-- Privacy Settings -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-user-shield"></i> Privacy Settings
                    </h3>
                    <form action="{{ route('student.privacy.update') }}" method="POST" class="ajax-form">
                        @csrf
                        <div style="display: grid; gap: 20px;">
                            <!-- Profile Visibility -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-eye"></i> Profile Visibility
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Make your profile visible to other learners</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="profile_visible" value="1" checked>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>
                            
                            <!-- Show on Leaderboard -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-ranking-star"></i> Leaderboard Participation
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Show your rank on the leaderboard</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="show_leaderboard" value="1" checked>
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>
                            
                            <!-- Share Progress -->
                            <div style="display: flex; justify-content: space-between; align-items: center; padding: 20px; background: #f9fafb; border-radius: 10px;">
                                <div>
                                    <h4 style="font-weight: 600; color: #333; margin-bottom: 5px;">
                                        <i class="fas fa-share-nodes"></i> Share Learning Progress
                                    </h4>
                                    <p style="color: #6b7280; font-size: 0.95em;">Allow others to see your learning progress</p>
                                </div>
                                <label class="toggle-switch">
                                    <input type="checkbox" name="share_progress" value="1">
                                    <span class="toggle-slider"></span>
                                </label>
                            </div>
                        </div>
                        <div style="margin-top: 25px; text-align: center;">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> Save Privacy Settings
                            </button>
                        </div>
                    </form>
                </div>
                
                <!-- Data Management -->
                <div class="card" style="margin-bottom: 30px;">
                    <h3 style="font-size: 1.5em; color: #333; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 2px solid #e5e7eb;">
                        <i class="fas fa-database"></i> Data Management
                    </h3>
                    <div style="display: grid; gap: 20px;">
                        <div style="padding: 20px; background: #f0f9ff; border-radius: 10px; border-left: 4px solid #3b82f6;">
                            <h4 style="font-weight: 600; color: #333; margin-bottom: 10px;">
                                <i class="fas fa-download"></i> Download Your Data
                            </h4>
                            <p style="color: #6b7280; margin-bottom: 15px;">Get a copy of all your learning data, progress, and certificates</p>
                            <button class="btn btn-primary">
                                <i class="fas fa-file-download"></i> Request Data Export
                            </button>
                        </div>
                        
                        <div style="padding: 20px; background: #fef2f2; border-radius: 10px; border-left: 4px solid #ef4444;">
                            <h4 style="font-weight: 600; color: #dc2626; margin-bottom: 10px;">
                                <i class="fas fa-exclamation-triangle"></i> Delete Account
                            </h4>
                            <p style="color: #6b7280; margin-bottom: 15px;">Permanently delete your account and all associated data. This action cannot be undone.</p>
                            <button class="btn" style="background: #ef4444; color: white;">
                                <i class="fas fa-trash-alt"></i> Delete My Account
                            </button>
                        </div>
                    </div>
                </div>
            </section>

            <style>
                .toggle-switch {
                    position: relative;
                    display: inline-block;
                    width: 60px;
                    height: 34px;
                }

                .toggle-switch input {
                    opacity: 0;
                    width: 0;
                    height: 0;
                }

                .toggle-slider {
                    position: absolute;
                    cursor: pointer;
                    top: 0;
                    left: 0;
                    right: 0;
                    bottom: 0;
                    background-color: #ccc;
                    transition: .4s;
                    border-radius: 34px;
                }

                .toggle-slider:before {
                    position: absolute;
                    content: "";
                    height: 26px;
                    width: 26px;
                    left: 4px;
                    bottom: 4px;
                    background-color: white;
                    transition: .4s;
                    border-radius: 50%;
                }

                input:checked + .toggle-slider {
                    background-color: #667eea;
                }

                input:focus + .toggle-slider {
                    box-shadow: 0 0 1px #667eea;
                }

                input:checked + .toggle-slider:before {
                    transform: translateX(26px);
                }
                
                /* Notification animations */
                @keyframes slideIn {
                    from {
                        transform: translateX(400px);
                        opacity: 0;
                    }
                    to {
                        transform: translateX(0);
                        opacity: 1;
                    }
                }
                
                @keyframes slideOut {
                    from {
                        transform: translateX(0);
                        opacity: 1;
                    }
                    to {
                        transform: translateX(400px);
                        opacity: 0;
                    }
                }
            </style>

            <script>
                // Auto-save settings when toggles change using AJAX
                if(document.getElementById('settingsForm')) {
                    document.getElementById('settingsForm').addEventListener('change', function(e) {
                        if(e.target.type === 'checkbox') {
                            e.preventDefault();
                            
                            // Submit via AJAX to stay on page
                            const formData = new FormData(this);
                            
                            fetch(this.action, {
                                method: 'POST',
                                body: formData,
                                headers: {
                                    'X-Requested-With': 'XMLHttpRequest'
                                }
                            })
                            .then(response => response.json())
                            .then(data => {
                                // Show success message
                                showNotification('Settings saved successfully!', 'success');
                            })
                            .catch(error => {
                                console.error('Error:', error);
                                showNotification('Error saving settings', 'error');
                            });
                        }
                    });
                }
                
                // Handle profile-related forms with AJAX
                document.querySelectorAll('form.ajax-form').forEach(form => {
                    form.addEventListener('submit', function(e) {
                        e.preventDefault();
                        
                        console.log('Form submitting:', this.action);
                        
                        const formData = new FormData(this);
                        const hasFile = this.querySelector('input[type="file"]');
                        
                        // Debug: Log form data
                        for (let pair of formData.entries()) {
                            console.log(pair[0] + ': ' + pair[1]);
                        }
                        
                        // Show loading notification for file uploads
                        if(hasFile && hasFile.files.length > 0) {
                            showNotification('Uploading...', 'info');
                        }
                        
                        fetch(this.action, {
                            method: 'POST',
                            body: formData,
                            headers: {
                                'X-Requested-With': 'XMLHttpRequest',
                                'Accept': 'application/json'
                            },
                            credentials: 'same-origin'
                        })
                        .then(response => {
                            console.log('Response status:', response.status);
                            if (!response.ok) {
                                const contentType = response.headers.get('content-type') || '';
                                if (contentType.includes('application/json')) {
                                    return response.json().then(err => {
                                        console.error('Server error:', err);
                                        return Promise.reject(err);
                                    });
                                }
                                return response.text().then(text => {
                                    console.error('Server error (non-JSON):', text);
                                    return Promise.reject({ message: 'An unexpected error occurred. Please try again.' });
                                });
                            }
                            const contentType = response.headers.get('content-type') || '';
                            if (response.status === 204 || !contentType.includes('application/json')) {
                                return {};
                            }
                            return response.json();
                        })
                        .then(data => {
                            console.log('Success response:', data);
                            const message = data && data.message ? data.message : 'Changes saved successfully!';
                            showNotification(message, 'success');
                            
                            // Reload page data if needed (e.g., profile updated, avatar changed)
                            if ((data && data.reload) || hasFile) {
                                setTimeout(() => location.reload(), 1500);
                            }
                            
                            // Reset form if it's password change
                            if(this.querySelector('input[name="current_password"]')) {
                                this.reset();
                            }
                        })
                        .catch(error => {
                            console.error('Error:', error);
                            let errorMessage = 'Error saving changes';
                            
                            // Handle validation errors
                            if(error.errors) {
                                errorMessage = Object.values(error.errors).flat().join(', ');
                            } else if(error.message) {
                                errorMessage = error.message;
                            }
                            
                            showNotification(errorMessage, 'error');
                        });
                    });
                });
                
                // Notification function
                function showNotification(message, type = 'success') {
                    const notification = document.createElement('div');
                    const colors = {
                        success: '#10b981',
                        error: '#ef4444',
                        info: '#3b82f6'
                    };
                    const icons = {
                        success: 'check-circle',
                        error: 'exclamation-circle',
                        info: 'info-circle'
                    };
                    
                    notification.style.cssText = `
                        position: fixed;
                        top: 20px;
                        right: 20px;
                        background: ${colors[type] || colors.success};
                        color: white;
                        padding: 15px 25px;
                        border-radius: 8px;
                        box-shadow: 0 10px 30px rgba(0,0,0,0.3);
                        z-index: 10000;
                        font-weight: 600;
                        animation: slideIn 0.3s ease;
                    `;
                    notification.innerHTML = `<i class="fas fa-${icons[type] || icons.success}"></i> ${message}`;
                    document.body.appendChild(notification);
                    
                    setTimeout(() => {
                        notification.style.animation = 'slideOut 0.3s ease';
                        setTimeout(() => notification.remove(), 300);
                    }, 3000);
                }
            </script>
        </main>
    </div>

    <!-- Mobile Menu Toggle -->
    <button class="mobile-toggle" onclick="toggleMobileMenu()">
        <i class="fas fa-bars"></i>
    </button>

    <script>
        // Navigation - Use event delegation to handle all data-section clicks
        document.addEventListener('click', function(e) {
            const target = e.target.closest('[data-section]');
            if (target) {
                const section = target.dataset.section;
                
                if (section) {
                    e.preventDefault();
                    showSection(section);
                    
                    // Update active state for sidebar nav items only
                    if (target.classList.contains('sidebar-nav')) {
                        document.querySelectorAll('.sidebar-nav').forEach(n => n.classList.remove('active'));
                        target.classList.add('active');
                    }
                    
                    // Close mobile menu on any click
                    if (window.innerWidth <= 768) {
                        document.getElementById('sidebar').classList.remove('mobile-open');
                    }
                }
            }
        });

        function showSection(sectionName) {
            // Hide all sections
            document.querySelectorAll('section[id^="section-"]').forEach(section => {
                section.style.display = 'none';
            });
            
            // Show selected section
            const targetSection = document.getElementById('section-' + sectionName);
            if (targetSection) {
                targetSection.style.display = 'block';
                // Scroll to top of content
                window.scrollTo({ top: 0, behavior: 'smooth' });
            }
        }

        function showCourseTab(tabName) {
            // Update tab buttons
            document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
            event.target.classList.add('active');
            
            // Update tab content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
            });
            document.getElementById('tab-' + tabName).classList.add('active');
        }

        function toggleMobileMenu() {
            document.getElementById('sidebar').classList.toggle('mobile-open');
        }

        // Chart.js Progress Chart
        const ctx = document.getElementById('progressChart');
        if (ctx) {
            new Chart(ctx, {
                type: 'line',
                data: {
                    labels: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    datasets: [{
                        label: 'Learning Time (hours)',
                        data: [2, 3.5, 2.5, 4, 3, 5, 4.5],
                        borderColor: '#667eea',
                        backgroundColor: 'rgba(102, 126, 234, 0.1)',
                        tension: 0.4,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        y: {
                            beginAtZero: true,
                            ticks: {
                                callback: function(value) {
                                    return value + 'h';
                                }
                            }
                        }
                    }
                }
            });
        }

        // Add page load animation
        window.addEventListener('load', function() {
            document.body.style.opacity = '0';
            setTimeout(() => {
                document.body.style.transition = 'opacity 0.5s ease';
                document.body.style.opacity = '1';
            }, 100);
        });
    </script>
</body>
</html>
