<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>My Badges - EE Micro LMS</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
        }

        .header {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
            margin-bottom: 30px;
            text-align: center;
        }

        .header h1 {
            font-size: 2.5em;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            margin-bottom: 10px;
        }

        .header p {
            color: #6c757d;
            font-size: 1.1em;
        }

        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }

        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            text-align: center;
            transition: transform 0.3s;
        }

        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }

        .stat-icon {
            font-size: 3em;
            margin-bottom: 10px;
        }

        .stat-value {
            font-size: 2.5em;
            font-weight: bold;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }

        .stat-label {
            color: #6c757d;
            font-size: 0.9em;
            margin-top: 5px;
        }

        .section {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 30px;
        }

        .section-title {
            font-size: 1.8em;
            color: #2c3e50;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .badges-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
            gap: 20px;
        }

        .badge-card {
            background: #f8f9fa;
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            transition: all 0.3s;
            position: relative;
            border: 3px solid transparent;
        }

        .badge-card.earned {
            background: linear-gradient(135deg, #fff 0%, #f0f4ff 100%);
            border-color: var(--badge-color, #667eea);
        }

        .badge-card.locked {
            opacity: 0.6;
            filter: grayscale(0.5);
        }

        .badge-card:hover {
            transform: scale(1.05);
            box-shadow: 0 10px 25px rgba(0,0,0,0.15);
        }

        .badge-icon {
            font-size: 4em;
            margin-bottom: 10px;
            filter: drop-shadow(0 4px 6px rgba(0,0,0,0.1));
        }

        .badge-card.locked .badge-icon {
            filter: grayscale(1) opacity(0.5);
        }

        .badge-name {
            font-size: 1.2em;
            font-weight: 600;
            color: #2c3e50;
            margin-bottom: 8px;
        }

        .badge-description {
            color: #6c757d;
            font-size: 0.9em;
            margin-bottom: 12px;
            line-height: 1.4;
        }

        .badge-points {
            display: inline-block;
            background: linear-gradient(135deg, #FFD700, #FFA500);
            color: white;
            padding: 5px 15px;
            border-radius: 20px;
            font-weight: 600;
            font-size: 0.9em;
        }

        .badge-earned-date {
            color: #28a745;
            font-size: 0.85em;
            margin-top: 10px;
            font-weight: 600;
        }

        .badge-progress {
            margin-top: 10px;
            background: #e9ecef;
            border-radius: 10px;
            height: 8px;
            overflow: hidden;
        }

        .badge-progress-bar {
            height: 100%;
            background: linear-gradient(90deg, #667eea, #764ba2);
            transition: width 0.5s ease;
        }

        .badge-progress-text {
            font-size: 0.8em;
            color: #6c757d;
            margin-top: 5px;
        }

        .earned-badge {
            position: absolute;
            top: 10px;
            right: 10px;
            background: #28a745;
            color: white;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 0.75em;
            font-weight: 600;
        }

        .locked-badge {
            position: absolute;
            top: 10px;
            right: 10px;
            background: #6c757d;
            color: white;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 0.75em;
            font-weight: 600;
        }

        .filter-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            flex-wrap: wrap;
        }

        .filter-tab {
            padding: 10px 20px;
            border-radius: 25px;
            border: 2px solid #e1e5e9;
            background: white;
            cursor: pointer;
            transition: all 0.3s;
            font-weight: 600;
        }

        .filter-tab:hover {
            border-color: #667eea;
            color: #667eea;
        }

        .filter-tab.active {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-color: transparent;
        }

        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: #6c757d;
        }

        .empty-state-icon {
            font-size: 5em;
            margin-bottom: 20px;
            opacity: 0.5;
        }

        .back-button {
            display: inline-block;
            padding: 12px 30px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            text-decoration: none;
            border-radius: 25px;
            font-weight: 600;
            transition: transform 0.3s;
            margin-bottom: 20px;
        }

        .back-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        @media (max-width: 768px) {
            .header h1 {
                font-size: 1.8em;
            }

            .badges-grid {
                grid-template-columns: 1fr;
            }

            .stats-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <a href="{{ url()->previous() }}" class="back-button">← Back to Dashboard</a>

        <div class="header">
            <h1>🏆 My Badges & Achievements</h1>
            <p>Collect badges by completing courses, scoring high on quizzes, and achieving milestones!</p>
        </div>

        <div class="stats-grid" id="statsGrid">
            <div class="stat-card">
                <div class="stat-icon">🏆</div>
                <div class="stat-value" id="earnedCount">-</div>
                <div class="stat-label">Badges Earned</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">🎯</div>
                <div class="stat-value" id="totalCount">-</div>
                <div class="stat-label">Total Badges</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">⭐</div>
                <div class="stat-value" id="totalPoints">-</div>
                <div class="stat-label">Total Points</div>
            </div>
            <div class="stat-card">
                <div class="stat-icon">📊</div>
                <div class="stat-value" id="completionPercentage">-%</div>
                <div class="stat-label">Completion</div>
            </div>
        </div>

        <div class="section">
            <h2 class="section-title">
                <span>🎮</span>
                <span>Badge Collection</span>
            </h2>

            <div class="filter-tabs">
                <button class="filter-tab active" data-filter="all">All Badges</button>
                <button class="filter-tab" data-filter="earned">Earned</button>
                <button class="filter-tab" data-filter="locked">Locked</button>
                <button class="filter-tab" data-filter="course">Course</button>
                <button class="filter-tab" data-filter="quiz">Quiz</button>
                <button class="filter-tab" data-filter="special">Special</button>
            </div>

            <div class="badges-grid" id="badgesGrid">
                <div class="empty-state">
                    <div class="empty-state-icon">⏳</div>
                    <h3>Loading badges...</h3>
                </div>
            </div>
        </div>
    </div>

    <script>
        let allBadges = [];
        let currentFilter = 'all';

        // Load badges on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadBadges();
            setupFilters();
        });

        async function loadBadges() {
            try {
                const response = await fetch('/badges', {
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });

                const data = await response.json();

                if (data.success) {
                    allBadges = data.badges;
                    updateStats(data.stats);
                    renderBadges();
                }
            } catch (error) {
                console.error('Failed to load badges:', error);
                document.getElementById('badgesGrid').innerHTML = `
                    <div class="empty-state">
                        <div class="empty-state-icon">😕</div>
                        <h3>Failed to load badges</h3>
                        <p>Please refresh the page and try again</p>
                    </div>
                `;
            }
        }

        function updateStats(stats) {
            document.getElementById('earnedCount').textContent = stats.earned_badges;
            document.getElementById('totalCount').textContent = stats.total_badges;
            document.getElementById('totalPoints').textContent = stats.total_points;
            document.getElementById('completionPercentage').textContent = stats.completion_percentage + '%';
        }

        function renderBadges() {
            const grid = document.getElementById('badgesGrid');
            
            let filteredBadges = allBadges;

            // Apply filters
            if (currentFilter === 'earned') {
                filteredBadges = allBadges.filter(b => b.earned);
            } else if (currentFilter === 'locked') {
                filteredBadges = allBadges.filter(b => !b.earned);
            } else if (currentFilter !== 'all') {
                filteredBadges = allBadges.filter(b => b.category === currentFilter);
            }

            if (filteredBadges.length === 0) {
                grid.innerHTML = `
                    <div class="empty-state">
                        <div class="empty-state-icon">🔍</div>
                        <h3>No badges found</h3>
                        <p>Try selecting a different filter</p>
                    </div>
                `;
                return;
            }

            grid.innerHTML = filteredBadges.map(badge => {
                const earnedClass = badge.earned ? 'earned' : 'locked';
                const earnedBadge = badge.earned 
                    ? '<span class="earned-badge">✓ Earned</span>' 
                    : '<span class="locked-badge">🔒 Locked</span>';
                
                const earnedDate = badge.earned_at 
                    ? `<div class="badge-earned-date">Earned: ${new Date(badge.earned_at).toLocaleDateString()}</div>`
                    : '';

                const progress = badge.progress 
                    ? `
                        <div class="badge-progress">
                            <div class="badge-progress-bar" style="width: ${badge.progress.percentage}%"></div>
                        </div>
                        <div class="badge-progress-text">${badge.progress.current} / ${badge.progress.required}</div>
                    `
                    : '';

                return `
                    <div class="badge-card ${earnedClass}" style="--badge-color: ${badge.color};">
                        ${earnedBadge}
                        <div class="badge-icon">${badge.icon}</div>
                        <div class="badge-name">${badge.name}</div>
                        <div class="badge-description">${badge.description}</div>
                        <div class="badge-points">⭐ ${badge.points} pts</div>
                        ${earnedDate}
                        ${progress}
                    </div>
                `;
            }).join('');
        }

        function setupFilters() {
            document.querySelectorAll('.filter-tab').forEach(tab => {
                tab.addEventListener('click', function() {
                    // Update active state
                    document.querySelectorAll('.filter-tab').forEach(t => t.classList.remove('active'));
                    this.classList.add('active');

                    // Update filter and re-render
                    currentFilter = this.dataset.filter;
                    renderBadges();
                });
            });
        }
    </script>
</body>
</html>
