import { Head, Link, usePage } from '@inertiajs/react';
import { useState } from 'react';

interface Course {
    id: number;
    name: string;
    description: string;
}

interface ResumePoint {
    id: number;
    topic_id: number;
    course_id: number;
}

interface BrandingColors {
    gradient?: string;
}

interface Branding {
    name?: string;
    tagline?: string;
    logo_url?: string;
    colors?: BrandingColors;
}

interface Props {
    enrolledCourses: Course[];
    progress: { [key: number]: number };
    resumePoint: ResumePoint | null;
    branding?: Branding;
}

export default function UserDashboard({ enrolledCourses, progress, resumePoint }: Props) {
    const [activeTab, setActiveTab] = useState('courses');
    const { branding } = usePage<{ branding?: Branding }>().props;
    const gradient = branding?.colors?.gradient ?? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
    const tagline = branding?.tagline ?? 'Entrepreneurship Training Hub';
    const resumeCourse = resumePoint
        ? enrolledCourses?.find(course => course.id === resumePoint.course_id)
        : undefined;
    const resumeLink = resumePoint
        ? `/courses/${resumePoint.course_id}${resumePoint.topic_id ? `?topic=${resumePoint.topic_id}` : ''}`
        : undefined;

    return (
        <>
            <Head title="My Dashboard" />

            <div className="min-h-screen bg-gray-100">
                {/* Header - Mobile Optimized */}
                <header className="bg-white shadow sticky top-0 z-50">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="flex items-center justify-between py-4 md:py-6">
                            {/* Left: Logo + Brand (Responsive) */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Logo */}
                                {branding?.logo_url ? (
                                    <img
                                        src={branding.logo_url}
                                        alt={branding.name ?? 'Site logo'}
                                        className="h-10 w-auto object-contain sm:h-12"
                                    />
                                ) : (
                                    <div
                                        className="h-10 w-10 sm:h-12 sm:w-12 flex items-center justify-center rounded-xl text-white flex-shrink-0"
                                        style={{ background: gradient }}
                                    >
                                        <i className="fas fa-lightbulb text-lg sm:text-xl" aria-hidden />
                                    </div>
                                )}
                                
                                {/* Brand Text - Hidden on small mobile */}
                                <div className="leading-tight hidden xs:block">
                                    <span className="block text-base sm:text-lg font-semibold text-gray-900">
                                        {branding?.name ?? 'EE Micro LMS'}
                                    </span>
                                    {/* Tagline - Hidden on mobile, shown on desktop */}
                                    <span className="hidden lg:block text-xs sm:text-sm text-gray-500">{tagline}</span>
                                </div>
                                
                                {/* Divider */}
                                <div className="hidden lg:block h-12 w-px bg-gray-200" />
                                
                                {/* Dashboard Title - Hidden on mobile, shown on tablet+ */}
                                <h1 className="hidden lg:block text-xl xl:text-2xl font-bold text-gray-900">My Dashboard</h1>
                            </div>
                            
                            {/* Right: Actions */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Student Badge - Hidden on small mobile */}
                                <span className="hidden sm:inline-flex items-center px-3 py-1.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                    Student
                                </span>
                                
                                {/* Logout Button */}
                                <Link
                                    href="/logout"
                                    method="post"
                                    className="inline-flex items-center gap-2 px-3 sm:px-4 py-2 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-100 transition-colors"
                                >
                                    <i className="fas fa-sign-out-alt text-sm" />
                                    <span className="hidden xs:inline">Logout</span>
                                </Link>
                            </div>
                        </div>
                        
                        {/* Mobile Dashboard Title - Only on small screens */}
                        <div className="lg:hidden pb-3 border-t pt-3 mt-1">
                            <h1 className="text-lg font-bold text-gray-900">My Dashboard</h1>
                        </div>
                    </div>
                </header>

                <main className="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                    {/* Navigation Tabs */}
                    <div className="mb-8 overflow-x-auto">
                        <nav className="flex space-x-4 sm:space-x-8 min-w-max sm:min-w-0">
                            {[
                                { id: 'courses', label: 'My Courses' },
                                { id: 'certificates', label: 'Certificates' },
                                { id: 'profile', label: 'Profile' }
                            ].map(tab => (
                                <button
                                    key={tab.id}
                                    onClick={() => setActiveTab(tab.id)}
                                    className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === tab.id
                                            ? 'border-indigo-500 text-indigo-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    {tab.label}
                                </button>
                            ))}
                        </nav>
                    </div>

                    {/* Content */}
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        {activeTab === 'courses' && (
                            <div className="p-4 sm:p-6">
                                {resumePoint && resumeLink && (
                                    <div
                                        className="mb-6 rounded-lg p-4 sm:p-6 text-white shadow"
                                        style={{ background: gradient }}
                                    >
                                        <div className="flex flex-col gap-4 md:flex-row md:items-center md:justify-between">
                                            <div className="min-w-0">
                                                <p className="text-xs sm:text-sm uppercase tracking-wide text-white/80">
                                                    Continue from where you stopped
                                                </p>
                                                <h3 className="text-lg sm:text-xl font-semibold truncate">
                                                    {resumeCourse?.name ?? 'Your course progress'}
                                                </h3>
                                                {resumeCourse?.description && (
                                                    <p className="text-sm text-white/80 line-clamp-2">
                                                        {resumeCourse.description}
                                                    </p>
                                                )}
                                            </div>
                                            <div className="flex items-center gap-3 flex-shrink-0">
                                                <i className="fas fa-play-circle text-2xl sm:text-3xl" aria-hidden />
                                                <Link
                                                    href={resumeLink}
                                                    className="whitespace-nowrap rounded-md bg-white/20 px-4 py-2 text-sm font-medium backdrop-blur transition hover:bg-white/30"
                                                >
                                                    Resume
                                                </Link>
                                            </div>
                                        </div>
                                    </div>
                                )}
                                <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-3 mb-4">
                                    <h2 className="text-base sm:text-lg font-medium text-gray-900">My Enrolled Courses</h2>
                                    <Link
                                        href="/courses"
                                        className="text-indigo-600 hover:text-indigo-900 text-sm whitespace-nowrap"
                                    >
                                        Browse All
                                    </Link>
                                </div>
                                <div className="space-y-4">
                                    {enrolledCourses?.map(course => (
                                        <div key={course.id} className="border rounded-lg p-4">
                                            <div className="flex flex-col sm:flex-row sm:justify-between sm:items-start gap-4">
                                                <div className="flex-1 min-w-0">
                                                    <h3 className="text-base sm:text-lg font-medium text-gray-900 truncate">{course.name}</h3>
                                                    <p className="text-sm text-gray-600 line-clamp-2">{course.description}</p>
                                                    <div className="mt-2 flex items-center space-x-4 text-sm text-gray-500">
                                                        <span>Progress: {progress?.[course.id] || 0}%</span>
                                                    </div>
                                                    <div className="mt-3">
                                                        <div className="w-full bg-gray-200 rounded-full h-2">
                                                            <div
                                                                className="bg-indigo-600 h-2 rounded-full transition-all duration-300"
                                                                style={{ width: `${progress?.[course.id] || 0}%` }}
                                                            ></div>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="flex-shrink-0 w-full sm:w-auto">
                                                    <Link
                                                        href={`/courses/${course.id}`}
                                                        className="block w-full sm:w-auto text-center bg-indigo-600 text-white px-4 py-2 rounded-md text-sm font-medium hover:bg-indigo-700 transition-colors"
                                                    >
                                                        {progress?.[course.id] > 0 ? 'Continue' : 'Start Course'}
                                                    </Link>
                                                </div>
                                            </div>
                                        </div>
                                    )) || (
                                        <div className="text-center py-8">
                                            <p className="text-gray-600 mb-4">You haven't enrolled in any courses yet.</p>
                                            <Link
                                                href="/courses"
                                                className="inline-block bg-indigo-600 text-white px-6 py-3 rounded-md text-sm font-medium hover:bg-indigo-700 transition-colors"
                                            >
                                                Browse Available Courses
                                            </Link>
                                        </div>
                                    )}
                                </div>
                            </div>
                        )}

                        {activeTab === 'certificates' && (
                            <div className="p-4 sm:p-6">
                                <h2 className="text-base sm:text-lg font-medium text-gray-900 mb-4">My Certificates</h2>
                                <p className="text-gray-600 text-sm sm:text-base">Certificate management coming soon...</p>
                            </div>
                        )}

                        {activeTab === 'profile' && (
                            <div className="p-4 sm:p-6">
                                <h2 className="text-base sm:text-lg font-medium text-gray-900 mb-4">Profile Settings</h2>
                                <p className="text-gray-600 text-sm sm:text-base">Profile management coming soon...</p>
                            </div>
                        )}
                    </div>
                </main>
            </div>
        </>
    );
}