import { Head, Link, usePage, router } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import axios from 'axios';
import CourseForm from '@/Components/CourseForm';

interface Course {
    id: number;
    name: string;
    short_description: string;
    category?: string;
    lesson_count: number;
    is_active: boolean;
}

interface Student {
    id: number;
    name: string;
    email: string;
    enrolled_courses?: number;
}

interface Category {
    id: number;
    name: string;
    is_active: boolean;
}

interface BrandingColors {
    gradient?: string;
}

interface Branding {
    name?: string;
    tagline?: string;
    logo_url?: string;
    colors?: BrandingColors;
}

interface Props {
    courses: Course[];
    students: Student[];
    branding?: Branding;
}

export default function TutorDashboard({ courses, students }: Props) {
    const [activeTab, setActiveTab] = useState('courses');
    const [showCourseForm, setShowCourseForm] = useState(false);
    const [editingCourse, setEditingCourse] = useState<Course | null>(null);
    const [categories, setCategories] = useState<Category[]>([]);
    const [selectedCategory, setSelectedCategory] = useState<string>('');
    const { branding } = usePage<{ branding?: Branding }>().props;
    const gradient = branding?.colors?.gradient ?? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
    const tagline = branding?.tagline ?? 'Entrepreneurship Training Hub';

    useEffect(() => {
        fetchCategories();
    }, []);

    const fetchCategories = async () => {
        try {
            const response = await axios.get('/api/categories', {
                headers: {
                    'Cache-Control': 'no-cache',
                    'Pragma': 'no-cache'
                }
            });
            setCategories(response.data.filter((cat: Category) => cat.is_active));
        } catch (error) {
            console.error('Error fetching categories:', error);
        }
    };

    const filteredCourses = selectedCategory
        ? courses?.filter(course => course.category === selectedCategory)
        : courses;

    const handleCourseSuccess = () => {
        setShowCourseForm(false);
        setEditingCourse(null);
        router.visit(window.location.href);
    };

    return (
        <>
            <Head title="Tutor Dashboard" />

            <div className="min-h-screen bg-gray-100">
                {/* Header - Mobile Optimized */}
                <header className="bg-white shadow sticky top-0 z-50">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="flex items-center justify-between py-4 md:py-6">
                            {/* Left: Logo + Brand (Responsive) */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Logo */}
                                {branding?.logo_url ? (
                                    <img
                                        src={branding.logo_url}
                                        alt={branding.name ?? 'Site logo'}
                                        className="h-10 w-auto object-contain sm:h-12"
                                    />
                                ) : (
                                    <div
                                        className="h-10 w-10 sm:h-12 sm:w-12 flex items-center justify-center rounded-xl text-white flex-shrink-0"
                                        style={{ background: gradient }}
                                    >
                                        <i className="fas fa-lightbulb text-lg sm:text-xl" aria-hidden />
                                    </div>
                                )}
                                
                                {/* Brand Text - Hidden on small mobile */}
                                <div className="leading-tight hidden xs:block">
                                    <span className="block text-base sm:text-lg font-semibold text-gray-900">
                                        {branding?.name ?? 'EE Micro LMS'}
                                    </span>
                                    {/* Tagline - Hidden on mobile, shown on desktop */}
                                    <span className="hidden lg:block text-xs sm:text-sm text-gray-500">{tagline}</span>
                                </div>
                                
                                {/* Divider */}
                                <div className="hidden lg:block h-12 w-px bg-gray-200" />
                                
                                {/* Dashboard Title - Hidden on mobile, shown on tablet+ */}
                                <h1 className="hidden lg:block text-xl xl:text-2xl font-bold text-gray-900">Tutor Dashboard</h1>
                            </div>
                            
                            {/* Right: Actions */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Tutor Badge - Hidden on small mobile */}
                                <span className="hidden sm:inline-flex items-center px-3 py-1.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                                    Tutor
                                </span>
                                
                                {/* Logout Button */}
                                <Link
                                    href="/logout"
                                    method="post"
                                    className="inline-flex items-center gap-2 px-3 sm:px-4 py-2 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-100 transition-colors"
                                >
                                    <i className="fas fa-sign-out-alt text-sm" />
                                    <span className="hidden xs:inline">Logout</span>
                                </Link>
                            </div>
                        </div>
                        
                        {/* Mobile Dashboard Title - Only on small screens */}
                        <div className="lg:hidden pb-3 border-t pt-3 mt-1">
                            <h1 className="text-lg font-bold text-gray-900">Tutor Dashboard</h1>
                        </div>
                    </div>
                </header>

                <main className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
                    {/* Navigation Tabs */}
                    <div className="mb-8">
                        <nav className="flex space-x-8">
                            {[
                                { id: 'courses', label: 'My Courses' },
                                { id: 'students', label: 'Students' },
                                { id: 'progress', label: 'Progress' },
                                { id: 'chat', label: 'Chat' }
                            ].map(tab => (
                                <button
                                    key={tab.id}
                                    onClick={() => setActiveTab(tab.id)}
                                    className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === tab.id
                                            ? 'border-indigo-500 text-indigo-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    {tab.label}
                                </button>
                            ))}
                        </nav>
                    </div>

                    {/* Content */}
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        {activeTab === 'courses' && (
                            <div className="p-6">
                                <div className="flex justify-between items-center mb-4 flex-wrap gap-4">
                                    <h2 className="text-lg font-medium text-gray-900">My Courses</h2>
                                    <div className="flex items-center gap-4">
                                        <div className="flex items-center gap-2">
                                            <label htmlFor="category-filter" className="text-sm font-medium text-gray-700">
                                                Filter:
                                            </label>
                                            <select
                                                id="category-filter"
                                                value={selectedCategory}
                                                onChange={(e) => setSelectedCategory(e.target.value)}
                                                className="rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 text-sm"
                                            >
                                                <option value="">All Categories</option>
                                                {categories.map((cat) => (
                                                    <option key={cat.id} value={cat.name}>
                                                        {cat.name}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>
                                        <button
                                            onClick={() => setShowCourseForm(true)}
                                            className="bg-indigo-600 text-white px-4 py-2 rounded-md text-sm font-medium hover:bg-indigo-700"
                                        >
                                            Create Course
                                        </button>
                                    </div>
                                </div>

                                {showCourseForm && (
                                    <div className="mb-6 border rounded-lg p-4 bg-gray-50">
                                        <h3 className="text-lg font-medium mb-4">
                                            {editingCourse ? 'Edit Course' : 'Create New Course'}
                                        </h3>
                                        <CourseForm
                                            course={editingCourse || undefined}
                                            onSuccess={handleCourseSuccess}
                                        />
                                        <button
                                            onClick={() => {
                                                setShowCourseForm(false);
                                                setEditingCourse(null);
                                            }}
                                            className="mt-4 text-gray-600 hover:text-gray-800 text-sm"
                                        >
                                            Cancel
                                        </button>
                                    </div>
                                )}

                                <div className="space-y-4">
                                    {filteredCourses?.map(course => (
                                        <div key={course.id} className="border rounded-lg p-4">
                                            <div className="flex justify-between items-start">
                                                <div>
                                                    <div className="flex items-center gap-2 mb-1">
                                                        <h3 className="text-lg font-medium text-gray-900">{course.name}</h3>
                                                        {course.category && (
                                                            <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                                                {course.category}
                                                            </span>
                                                        )}
                                                    </div>
                                                    <p className="text-sm text-gray-600">{course.short_description}</p>
                                                    <div className="mt-2 flex items-center space-x-4 text-sm text-gray-500">
                                                        <span>{course.lesson_count} lessons</span>
                                                        <span>{course.is_active ? 'Active' : 'Inactive'}</span>
                                                    </div>
                                                </div>
                                                <div className="flex space-x-2">
                                                    <button
                                                        onClick={() => {
                                                            setEditingCourse(course);
                                                            setShowCourseForm(true);
                                                        }}
                                                        className="text-indigo-600 hover:text-indigo-900 text-sm"
                                                    >
                                                        Edit
                                                    </button>
                                                    <Link
                                                        href={`/courses/${course.id}`}
                                                        className="text-gray-600 hover:text-gray-900 text-sm"
                                                    >
                                                        Manage
                                                    </Link>
                                                </div>
                                            </div>
                                        </div>
                                    )) || (
                                        <p className="text-gray-600">No courses yet. Create your first course!</p>
                                    )}
                                    {filteredCourses?.length === 0 && courses?.length > 0 && (
                                        <p className="text-gray-600">No courses found in this category.</p>
                                    )}
                                </div>
                            </div>
                        )}

                        {activeTab === 'students' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">My Students</h2>
                                <div className="space-y-4">
                                    {students?.map(student => (
                                        <div key={student.id} className="border rounded-lg p-4">
                                            <div className="flex justify-between items-center">
                                                <div>
                                                    <h3 className="text-lg font-medium text-gray-900">{student.name}</h3>
                                                    <p className="text-sm text-gray-600">{student.email}</p>
                                                </div>
                                                <div className="text-sm text-gray-500">
                                                    Enrolled: {student.enrolled_courses || 0} courses
                                                </div>
                                            </div>
                                        </div>
                                    )) || (
                                        <p className="text-gray-600">No students enrolled yet.</p>
                                    )}
                                </div>
                            </div>
                        )}

                        {activeTab === 'progress' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">Student Progress</h2>
                                <p className="text-gray-600">Progress tracking interface coming soon...</p>
                            </div>
                        )}

                        {activeTab === 'chat' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">Chat with Students</h2>
                                <p className="text-gray-600">Chat interface coming soon...</p>
                            </div>
                        )}
                    </div>
                </main>
            </div>
        </>
    );
}