import { Head, Link, usePage } from '@inertiajs/react';
import { useState } from 'react';

interface BrandingColors {
    gradient?: string;
    primary?: string;
}

interface Branding {
    name?: string;
    tagline?: string;
    logo_url?: string;
    colors?: BrandingColors;
}

interface AdminPageProps {
    stats: any;
    branding?: Branding;
}

export default function AdminDashboard({ stats }: AdminPageProps) {
    const [activeTab, setActiveTab] = useState('overview');
    const { branding } = usePage<{ branding?: Branding }>().props;
    const gradient = branding?.colors?.gradient ?? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
    const tagline = branding?.tagline ?? 'Entrepreneurship Training Hub';

    return (
        <>
            <Head title="Admin Dashboard" />

            <div className="min-h-screen bg-gray-100">
                {/* Header - Mobile Optimized */}
                <header className="bg-white shadow sticky top-0 z-50">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="flex items-center justify-between py-4 md:py-6">
                            {/* Left: Logo + Brand (Responsive) */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Logo */}
                                {branding?.logo_url ? (
                                    <img
                                        src={branding.logo_url}
                                        alt={branding.name ?? 'Site logo'}
                                        className="h-10 w-auto object-contain sm:h-12"
                                    />
                                ) : (
                                    <div
                                        className="h-10 w-10 sm:h-12 sm:w-12 flex items-center justify-center rounded-xl text-white flex-shrink-0"
                                        style={{ background: gradient }}
                                    >
                                        <i className="fas fa-lightbulb text-lg sm:text-xl" aria-hidden />
                                    </div>
                                )}
                                
                                {/* Brand Text - Hidden on small mobile */}
                                <div className="leading-tight hidden xs:block">
                                    <span className="block text-base sm:text-lg font-semibold text-gray-900">
                                        {branding?.name ?? 'EE Micro LMS'}
                                    </span>
                                    {/* Tagline - Hidden on mobile, shown on desktop */}
                                    <span className="hidden lg:block text-xs sm:text-sm text-gray-500">{tagline}</span>
                                </div>
                                
                                {/* Divider */}
                                <div className="hidden lg:block h-12 w-px bg-gray-200" />
                                
                                {/* Dashboard Title - Hidden on mobile, shown on tablet+ */}
                                <h1 className="hidden lg:block text-xl xl:text-2xl font-bold text-gray-900">Admin Dashboard</h1>
                            </div>
                            
                            {/* Right: Actions */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Admin Badge - Hidden on small mobile */}
                                <span className="hidden sm:inline-flex items-center px-3 py-1.5 rounded-full text-xs font-medium bg-red-100 text-red-800">
                                    Admin
                                </span>
                                
                                {/* Logout Button */}
                                <Link
                                    href="/logout"
                                    method="post"
                                    className="inline-flex items-center gap-2 px-3 sm:px-4 py-2 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-100 transition-colors"
                                >
                                    <i className="fas fa-sign-out-alt text-sm" />
                                    <span className="hidden xs:inline">Logout</span>
                                </Link>
                            </div>
                        </div>
                        
                        {/* Mobile Dashboard Title - Only on small screens */}
                        <div className="lg:hidden pb-3 border-t pt-3 mt-1">
                            <h1 className="text-lg font-bold text-gray-900">Admin Dashboard</h1>
                        </div>
                    </div>
                </header>

                <main className="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
                    {/* Navigation Tabs */}
                    <div className="mb-8">
                        <nav className="flex space-x-8">
                            {[
                                { id: 'overview', label: 'Overview' },
                                { id: 'users', label: 'Users' },
                                { id: 'courses', label: 'Courses' },
                                { id: 'categories', label: 'Categories' },
                                { id: 'settings', label: 'Settings' }
                            ].map(tab => (
                                <button
                                    key={tab.id}
                                    onClick={() => setActiveTab(tab.id)}
                                    className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                        activeTab === tab.id
                                            ? 'border-indigo-500 text-indigo-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    {tab.label}
                                </button>
                            ))}
                        </nav>
                    </div>

                    {/* Content */}
                    <div className="bg-white overflow-hidden shadow rounded-lg">
                        {activeTab === 'overview' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">System Overview</h2>
                                <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
                                    <div className="bg-blue-50 p-4 rounded-lg">
                                        <div className="text-2xl font-bold text-blue-600">{stats?.total_users || 0}</div>
                                        <div className="text-sm text-blue-600">Total Users</div>
                                    </div>
                                    <div className="bg-green-50 p-4 rounded-lg">
                                        <div className="text-2xl font-bold text-green-600">{stats?.total_courses || 0}</div>
                                        <div className="text-sm text-green-600">Total Courses</div>
                                    </div>
                                    <div className="bg-yellow-50 p-4 rounded-lg">
                                        <div className="text-2xl font-bold text-yellow-600">{stats?.total_lessons || 0}</div>
                                        <div className="text-sm text-yellow-600">Total Lessons</div>
                                    </div>
                                    <div className="bg-purple-50 p-4 rounded-lg">
                                        <div className="text-2xl font-bold text-purple-600">{stats?.total_topics || 0}</div>
                                        <div className="text-sm text-purple-600">Total Topics</div>
                                    </div>
                                </div>
                            </div>
                        )}

                        {activeTab === 'users' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">User Management</h2>
                                <p className="text-gray-600">User management interface coming soon...</p>
                            </div>
                        )}

                        {activeTab === 'courses' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">Course Management</h2>
                                <p className="text-gray-600">Course management interface coming soon...</p>
                            </div>
                        )}

                        {activeTab === 'categories' && (
                            <div className="p-6">
                                <div className="flex justify-between items-center mb-4">
                                    <h2 className="text-lg font-medium text-gray-900">Category Management</h2>
                                    <Link
                                        href="/admin/categories"
                                        className="inline-flex items-center gap-2 px-4 py-2 bg-indigo-600 text-white rounded-md hover:bg-indigo-700"
                                    >
                                        <i className="fas fa-cog" />
                                        Manage Categories
                                    </Link>
                                </div>
                                <p className="text-gray-600">Click "Manage Categories" to create, edit, and delete course categories.</p>
                            </div>
                        )}

                        {activeTab === 'settings' && (
                            <div className="p-6">
                                <h2 className="text-lg font-medium text-gray-900 mb-4">System Settings</h2>
                                <p className="text-gray-600">Settings interface coming soon...</p>
                            </div>
                        )}
                    </div>
                </main>
            </div>
        </>
    );
}