import { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import LessonForm from '@/Components/LessonForm';

interface Lesson {
    id: number;
    name: string;
    description: string;
    order_index: number;
    is_active: boolean;
    topics_count: number;
}

interface Course {
    id: number;
    name: string;
    short_description: string;
    is_active: boolean;
    lessons: Lesson[];
}

interface Props {
    course: Course;
}

export default function CourseDetail({ course }: Props) {
    const [showLessonForm, setShowLessonForm] = useState(false);
    const [editingLesson, setEditingLesson] = useState<Lesson | null>(null);

    const handleLessonSuccess = () => {
        setShowLessonForm(false);
        setEditingLesson(null);
        router.visit(window.location.href);
    };

    const handleDeleteLesson = (lessonId: number) => {
        if (confirm('Are you sure you want to delete this lesson?')) {
            router.delete(route('lessons.destroy', lessonId), {
                onSuccess: handleLessonSuccess,
            });
        }
    };

    return (
        <>
            <Head title={`${course.name} - Course Details`} />

            <div className="min-h-screen bg-gray-50">
                <div className="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <Link
                                    href="/dashboard/tutor"
                                    className="text-indigo-600 hover:text-indigo-800 text-sm"
                                >
                                    ← Back to Dashboard
                                </Link>
                                <h1 className="text-3xl font-bold text-gray-900 mt-2">{course.name}</h1>
                                <p className="mt-2 text-gray-600">{course.short_description}</p>
                            </div>
                            <div className="flex items-center space-x-4">
                                <span className={`px-3 py-1 text-sm rounded-full ${
                                    course.is_active
                                        ? 'bg-green-100 text-green-800'
                                        : 'bg-yellow-100 text-yellow-800'
                                }`}>
                                    {course.is_active ? 'Active' : 'Inactive'}
                                </span>
                                <Link
                                    href={`/courses/${course.id}/edit`}
                                    className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700 text-sm"
                                >
                                    Edit Course
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Lessons Section */}
                    <div className="bg-white shadow rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200">
                            <div className="flex justify-between items-center">
                                <h2 className="text-lg font-medium text-gray-900">Lessons</h2>
                                <button
                                    onClick={() => setShowLessonForm(true)}
                                    className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700 text-sm"
                                >
                                    Add Lesson
                                </button>
                            </div>
                        </div>

                        <div className="p-6">
                            {showLessonForm && (
                                <div className="mb-6 border rounded-lg p-4 bg-gray-50">
                                    <h3 className="text-lg font-medium mb-4">
                                        {editingLesson ? 'Edit Lesson' : 'Create New Lesson'}
                                    </h3>
                                    <LessonForm
                                        courseId={course.id}
                                        lesson={editingLesson || undefined}
                                        onSuccess={handleLessonSuccess}
                                    />
                                    <button
                                        onClick={() => {
                                            setShowLessonForm(false);
                                            setEditingLesson(null);
                                        }}
                                        className="mt-4 text-gray-600 hover:text-gray-800 text-sm"
                                    >
                                        Cancel
                                    </button>
                                </div>
                            )}

                            {course.lessons && course.lessons.length > 0 ? (
                                <div className="space-y-4">
                                    {course.lessons
                                        .sort((a, b) => a.order_index - b.order_index)
                                        .map((lesson) => (
                                            <div key={lesson.id} className="border rounded-lg p-4">
                                                <div className="flex justify-between items-start">
                                                    <div className="flex-1">
                                                        <div className="flex items-center space-x-3">
                                                            <span className="text-sm text-gray-500 font-medium">
                                                                #{lesson.order_index}
                                                            </span>
                                                            <h3 className="text-lg font-medium text-gray-900">
                                                                {lesson.name}
                                                            </h3>
                                                            <span className={`px-2 py-1 text-xs rounded-full ${
                                                                lesson.is_active
                                                                    ? 'bg-green-100 text-green-800'
                                                                    : 'bg-gray-100 text-gray-800'
                                                            }`}>
                                                                {lesson.is_active ? 'Active' : 'Inactive'}
                                                            </span>
                                                        </div>
                                                        <p className="mt-2 text-gray-600">{lesson.description}</p>
                                                        <div className="mt-2 text-sm text-gray-500">
                                                            {lesson.topics_count} topics
                                                        </div>
                                                    </div>
                                                    <div className="flex space-x-2 ml-4">
                                                        <button
                                                            onClick={() => {
                                                                setEditingLesson(lesson);
                                                                setShowLessonForm(true);
                                                            }}
                                                            className="text-indigo-600 hover:text-indigo-800 text-sm"
                                                        >
                                                            Edit
                                                        </button>
                                                        <button
                                                            onClick={() => handleDeleteLesson(lesson.id)}
                                                            className="text-red-600 hover:text-red-800 text-sm"
                                                        >
                                                            Delete
                                                        </button>
                                                        <Link
                                                            href={`/lessons/${lesson.id}`}
                                                            className="text-gray-600 hover:text-gray-800 text-sm"
                                                        >
                                                            Manage Topics
                                                        </Link>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                </div>
                            ) : (
                                <div className="text-center py-8">
                                    <p className="text-gray-600 mb-4">No lessons created yet.</p>
                                    <button
                                        onClick={() => setShowLessonForm(true)}
                                        className="bg-indigo-600 text-white px-4 py-2 rounded-md hover:bg-indigo-700"
                                    >
                                        Create Your First Lesson
                                    </button>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </>
    );
}