import { Head, Link, usePage } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import axios from 'axios';

interface Category {
    id: number;
    name: string;
    slug: string;
    description: string | null;
    icon: string | null;
    is_active: boolean;
    order: number;
}

interface BrandingColors {
    gradient?: string;
    primary?: string;
}

interface Branding {
    name?: string;
    tagline?: string;
    logo_url?: string;
    colors?: BrandingColors;
}

export default function Categories() {
    const { branding } = usePage<{ branding?: Branding }>().props;
    const gradient = branding?.colors?.gradient ?? 'linear-gradient(135deg, #667eea 0%, #764ba2 100%)';
    const tagline = branding?.tagline ?? 'Entrepreneurship Training Hub';
    const [categories, setCategories] = useState<Category[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [editingCategory, setEditingCategory] = useState<Category | null>(null);
    const [formData, setFormData] = useState({
        name: '',
        description: '',
        icon: '',
        is_active: true,
        order: 0
    });

    useEffect(() => {
        fetchCategories();
    }, []);

    const fetchCategories = async () => {
        try {
            const response = await axios.get('/api/categories');
            setCategories(response.data);
        } catch (error) {
            console.error('Error fetching categories:', error);
        } finally {
            setLoading(false);
        }
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        
        try {
            if (editingCategory) {
                await axios.put(`/api/categories/${editingCategory.id}`, formData);
            } else {
                await axios.post('/api/categories', formData);
            }
            
            setShowModal(false);
            setEditingCategory(null);
            setFormData({ name: '', description: '', icon: '', is_active: true, order: 0 });
            fetchCategories();
        } catch (error: any) {
            console.error('Error saving category:', error);
            alert(error.response?.data?.message || 'Error saving category');
        }
    };

    const handleEdit = (category: Category) => {
        setEditingCategory(category);
        setFormData({
            name: category.name,
            description: category.description || '',
            icon: category.icon || '',
            is_active: category.is_active,
            order: category.order
        });
        setShowModal(true);
    };

    const handleDelete = async (id: number) => {
        if (!confirm('Are you sure you want to delete this category?')) return;
        
        try {
            await axios.delete(`/api/categories/${id}`);
            fetchCategories();
        } catch (error) {
            console.error('Error deleting category:', error);
            alert('Error deleting category');
        }
    };

    const openNewModal = () => {
        setEditingCategory(null);
        setFormData({ name: '', description: '', icon: '', is_active: true, order: 0 });
        setShowModal(true);
    };

    return (
        <>
            <Head title="Manage Categories" />

            <div className="min-h-screen bg-gray-100">
                {/* Header - Mobile Optimized */}
                <header className="bg-white shadow sticky top-0 z-50">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="flex items-center justify-between py-4 md:py-6">
                            {/* Left: Logo + Brand (Responsive) */}
                            <div className="flex items-center gap-2 sm:gap-3 md:gap-4">
                                {/* Logo */}
                                {branding?.logo_url ? (
                                    <img
                                        src={branding.logo_url}
                                        alt={branding.name ?? 'Site logo'}
                                        className="h-10 w-auto object-contain sm:h-12"
                                    />
                                ) : (
                                    <div
                                        className="h-10 w-10 sm:h-12 sm:w-12 flex items-center justify-center rounded-xl text-white flex-shrink-0"
                                        style={{ background: gradient }}
                                    >
                                        <i className="fas fa-lightbulb text-lg sm:text-xl" aria-hidden />
                                    </div>
                                )}

                                {/* Brand Text - Hidden on small screens */}
                                <div className="hidden sm:block">
                                    <h1 className="text-lg md:text-xl font-bold text-gray-900">
                                        {branding?.name ?? 'Learning Portal'}
                                    </h1>
                                    <p className="text-xs md:text-sm text-gray-600 hidden md:block">{tagline}</p>
                                </div>
                            </div>

                            {/* Right: Navigation Links */}
                            <nav className="flex items-center gap-2 sm:gap-4">
                                <Link
                                    href="/admin/dashboard"
                                    className="text-sm sm:text-base text-gray-700 hover:text-indigo-600 transition px-2 sm:px-3 py-2"
                                >
                                    <i className="fas fa-home mr-1 sm:mr-2" />
                                    <span className="hidden sm:inline">Dashboard</span>
                                </Link>
                                <Link
                                    href="/admin/categories"
                                    className="text-sm sm:text-base text-indigo-600 font-semibold px-2 sm:px-3 py-2 border-b-2 border-indigo-600"
                                >
                                    <i className="fas fa-folder mr-1 sm:mr-2" />
                                    <span className="hidden sm:inline">Categories</span>
                                </Link>
                            </nav>
                        </div>
                    </div>
                </header>

                {/* Main Content */}
                <div className="py-6 sm:py-12">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                            <div className="p-4 sm:p-6 text-gray-900">
                                {/* Header */}
                                <div className="mb-6 flex flex-col sm:flex-row sm:justify-between sm:items-center gap-4">
                                    <h1 className="text-xl sm:text-2xl font-bold">Course Categories</h1>
                                    <button
                                        onClick={openNewModal}
                                        className="bg-indigo-600 hover:bg-indigo-700 text-white px-4 py-2 rounded-md flex items-center justify-center gap-2 w-full sm:w-auto"
                                    >
                                        <i className="fas fa-plus" />
                                        Add Category
                                    </button>
                                </div>

                                {/* Categories List */}
                                {loading ? (
                                    <div className="p-6 text-center">Loading...</div>
                                ) : categories.length === 0 ? (
                                    <div className="p-6 text-center text-gray-500">
                                        No categories found. Create your first category!
                                    </div>
                                ) : (
                                    <div className="overflow-x-auto -mx-4 sm:mx-0">
                                        <table className="min-w-full divide-y divide-gray-200">
                                            <thead className="bg-gray-50">
                                                <tr>
                                                    <th className="px-4 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Name
                                                    </th>
                                                    <th className="px-4 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden md:table-cell">
                                                        Description
                                                    </th>
                                                    <th className="px-4 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden sm:table-cell">
                                                        Icon
                                                    </th>
                                                    <th className="px-4 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider hidden lg:table-cell">
                                                        Order
                                                    </th>
                                                    <th className="px-4 sm:px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Status
                                                    </th>
                                                    <th className="px-4 sm:px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                        Actions
                                                    </th>
                                                </tr>
                                            </thead>
                                            <tbody className="bg-white divide-y divide-gray-200">
                                                {categories.map((category) => (
                                                    <tr key={category.id}>
                                                        <td className="px-4 sm:px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                            {category.name}
                                                        </td>
                                                        <td className="px-4 sm:px-6 py-4 text-sm text-gray-500 hidden md:table-cell">
                                                            {category.description || '-'}
                                                        </td>
                                                        <td className="px-4 sm:px-6 py-4 whitespace-nowrap text-sm text-gray-500 hidden sm:table-cell">
                                                            {category.icon ? (
                                                                <i className={`fas fa-${category.icon}`} />
                                                            ) : '-'}
                                                        </td>
                                                        <td className="px-4 sm:px-6 py-4 whitespace-nowrap text-sm text-gray-500 hidden lg:table-cell">
                                                            {category.order}
                                                        </td>
                                                        <td className="px-4 sm:px-6 py-4 whitespace-nowrap">
                                                            <span className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                                category.is_active 
                                                                    ? 'bg-green-100 text-green-800' 
                                                                    : 'bg-red-100 text-red-800'
                                                            }`}>
                                                                {category.is_active ? 'Active' : 'Inactive'}
                                                            </span>
                                                        </td>
                                                        <td className="px-4 sm:px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                            <button
                                                                onClick={() => handleEdit(category)}
                                                                className="text-indigo-600 hover:text-indigo-900 mr-3"
                                                                title="Edit"
                                                            >
                                                                <i className="fas fa-edit" />
                                                            </button>
                                                            <button
                                                                onClick={() => handleDelete(category.id)}
                                                                className="text-red-600 hover:text-red-900"
                                                                title="Delete"
                                                            >
                                                                <i className="fas fa-trash" />
                                                            </button>
                                                        </td>
                                                    </tr>
                                                ))}
                                            </tbody>
                                        </table>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {/* Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50 flex items-center justify-center p-4">
                    <div className="relative mx-auto w-full max-w-lg">
                        <div className="bg-white overflow-hidden shadow-xl sm:rounded-lg">
                            <div className="p-6 sm:p-8 text-gray-900">
                                <div className="mb-6 flex items-center justify-between">
                                    <h3 className="text-xl sm:text-2xl font-bold">
                                        {editingCategory ? 'Edit Category' : 'Add New Category'}
                                    </h3>
                                    <button
                                        type="button"
                                        onClick={() => {
                                            setShowModal(false);
                                            setEditingCategory(null);
                                        }}
                                        className="text-gray-400 hover:text-gray-600 transition"
                                    >
                                        <i className="fas fa-times text-xl" />
                                    </button>
                                </div>
                                
                                <form onSubmit={handleSubmit} className="space-y-5">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Category Name <span className="text-red-500">*</span>
                                        </label>
                                        <input
                                            type="text"
                                            value={formData.name}
                                            onChange={(e) => setFormData({ ...formData, name: e.target.value })}
                                            className="w-full px-4 py-2.5 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition"
                                            placeholder="Enter category name"
                                            required
                                        />
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Description
                                        </label>
                                        <textarea
                                            value={formData.description}
                                            onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                                            className="w-full px-4 py-2.5 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition"
                                            rows={3}
                                            placeholder="Describe this category"
                                        />
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-2">
                                            Icon (FontAwesome)
                                        </label>
                                        <div className="relative">
                                            <input
                                                type="text"
                                                value={formData.icon}
                                                onChange={(e) => setFormData({ ...formData, icon: e.target.value })}
                                                className="w-full px-4 py-2.5 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition"
                                                placeholder="briefcase, book, rocket"
                                            />
                                            {formData.icon && (
                                                <div className="absolute right-3 top-2.5 text-gray-400">
                                                    <i className={`fas fa-${formData.icon}`} />
                                                </div>
                                            )}
                                        </div>
                                        <p className="text-xs text-gray-500 mt-1.5">
                                            <i className="fas fa-info-circle mr-1" />
                                            Enter FontAwesome icon name without 'fa-' prefix
                                        </p>
                                    </div>

                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                                Display Order
                                            </label>
                                            <input
                                                type="number"
                                                value={formData.order}
                                                onChange={(e) => setFormData({ ...formData, order: parseInt(e.target.value) })}
                                                className="w-full px-4 py-2.5 border border-gray-300 rounded-lg shadow-sm focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 transition"
                                                min="0"
                                            />
                                        </div>

                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-2">
                                                Status
                                            </label>
                                            <div className="flex items-center h-[42px]">
                                                <label className="flex items-center cursor-pointer">
                                                    <input
                                                        type="checkbox"
                                                        checked={formData.is_active}
                                                        onChange={(e) => setFormData({ ...formData, is_active: e.target.checked })}
                                                        className="w-4 h-4 rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500"
                                                    />
                                                    <span className="ml-2 text-sm font-medium text-gray-700">Active</span>
                                                </label>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="flex flex-col-reverse sm:flex-row justify-end gap-3 pt-6 border-t border-gray-200">
                                        <button
                                            type="button"
                                            onClick={() => {
                                                setShowModal(false);
                                                setEditingCategory(null);
                                            }}
                                            className="w-full sm:w-auto px-5 py-2.5 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition"
                                        >
                                            Cancel
                                        </button>
                                        <button
                                            type="submit"
                                            className="w-full sm:w-auto px-5 py-2.5 text-sm font-medium text-white bg-indigo-600 rounded-lg hover:bg-indigo-700 transition shadow-sm"
                                        >
                                            <i className={`fas fa-${editingCategory ? 'check' : 'plus'} mr-2`} />
                                            {editingCategory ? 'Update Category' : 'Create Category'}
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                </div>
            )}
        </>
    );
}
