import { useForm } from '@inertiajs/react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';

interface TopicFormProps {
    lessonId: number;
    topic?: {
        id: number;
        title: string;
        content: string;
        order_index: number;
        is_published: boolean;
    };
    onSuccess?: () => void;
}

export default function TopicForm({ lessonId, topic, onSuccess }: TopicFormProps) {
    const isEditing = !!topic;

    const { data, setData, post, put, processing, errors } = useForm({
        lesson_id: lessonId,
        title: topic?.title || '',
        content: topic?.content || '',
        order_index: topic?.order_index || 1,
        is_published: topic?.is_published ?? false,
    });

    const submit = (e: React.FormEvent) => {
        e.preventDefault();

        if (isEditing) {
            put(route('topics.update', topic.id), {
                onSuccess: onSuccess,
            });
        } else {
            post(route('topics.store'), {
                onSuccess: onSuccess,
            });
        }
    };

    return (
        <form onSubmit={submit} className="space-y-6">
            <div>
                <InputLabel htmlFor="title" value="Topic Title" />
                <TextInput
                    id="title"
                    type="text"
                    className="mt-1 block w-full"
                    value={data.title}
                    onChange={(e) => setData('title', e.target.value)}
                    required
                />
                <InputError message={errors.title} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="content" value="Topic Content" />
                <textarea
                    id="content"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    rows={8}
                    value={data.content}
                    onChange={(e) => setData('content', e.target.value)}
                    placeholder="Enter the topic content here..."
                    required
                />
                <InputError message={errors.content} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="order_index" value="Order" />
                <TextInput
                    id="order_index"
                    type="number"
                    className="mt-1 block w-full"
                    value={data.order_index}
                    onChange={(e) => setData('order_index', parseInt(e.target.value) || 1)}
                    min="1"
                    required
                />
                <InputError message={errors.order_index} className="mt-2" />
            </div>

            <div className="flex items-center">
                <input
                    id="is_published"
                    type="checkbox"
                    className="rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500"
                    checked={data.is_published}
                    onChange={(e) => setData('is_published', e.target.checked)}
                />
                <InputLabel htmlFor="is_published" value="Publish Topic" className="ml-2" />
            </div>

            <div className="flex items-center justify-end">
                <PrimaryButton disabled={processing}>
                    {isEditing ? 'Update Topic' : 'Create Topic'}
                </PrimaryButton>
            </div>
        </form>
    );
}