import { useForm } from '@inertiajs/react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';

interface LessonFormProps {
    courseId: number;
    lesson?: {
        id: number;
        name: string;
        description: string;
        order_index: number;
        is_active: boolean;
    };
    onSuccess?: () => void;
}

export default function LessonForm({ courseId, lesson, onSuccess }: LessonFormProps) {
    const isEditing = !!lesson;

    const { data, setData, post, put, processing, errors } = useForm({
        course_id: courseId,
        name: lesson?.name || '',
        description: lesson?.description || '',
        order_index: lesson?.order_index || 1,
        is_active: lesson?.is_active ?? true,
    });

    const submit = (e: React.FormEvent) => {
        e.preventDefault();

        if (isEditing) {
            put(route('lessons.update', lesson.id), {
                onSuccess: onSuccess,
            });
        } else {
            post(route('lessons.store'), {
                onSuccess: onSuccess,
            });
        }
    };

    return (
        <form onSubmit={submit} className="space-y-6">
            <div>
                <InputLabel htmlFor="name" value="Lesson Title" />
                <TextInput
                    id="name"
                    type="text"
                    className="mt-1 block w-full"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    required
                />
                <InputError message={errors.name} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="description" value="Lesson Description" />
                <textarea
                    id="description"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    rows={3}
                    value={data.description}
                    onChange={(e) => setData('description', e.target.value)}
                />
                <InputError message={errors.description} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="order_index" value="Order" />
                <TextInput
                    id="order_index"
                    type="number"
                    className="mt-1 block w-full"
                    value={data.order_index}
                    onChange={(e) => setData('order_index', parseInt(e.target.value) || 1)}
                    min="1"
                    required
                />
                <InputError message={errors.order_index} className="mt-2" />
            </div>

            <div className="flex items-center">
                <input
                    id="is_active"
                    type="checkbox"
                    className="rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500"
                    checked={data.is_active}
                    onChange={(e) => setData('is_active', e.target.checked)}
                />
                <InputLabel htmlFor="is_active" value="Active" className="ml-2" />
            </div>

            <div className="flex items-center justify-end">
                <PrimaryButton disabled={processing}>
                    {isEditing ? 'Update Lesson' : 'Create Lesson'}
                </PrimaryButton>
            </div>
        </form>
    );
}