import { useForm } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import axios from 'axios';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';

interface Category {
    id: number;
    name: string;
    is_active: boolean;
}

interface CourseFormProps {
    course?: {
        id: number;
        name: string;
        short_description: string;
        is_active: boolean;
        category?: string;
    };
    onSuccess?: () => void;
}

export default function CourseForm({ course, onSuccess }: CourseFormProps) {
    const isEditing = !!course;
    const [categories, setCategories] = useState<Category[]>([]);

    useEffect(() => {
        fetchCategories();
    }, []);

    const fetchCategories = async () => {
        try {
            const response = await axios.get('/api/categories', {
                headers: {
                    'Cache-Control': 'no-cache',
                    'Pragma': 'no-cache'
                }
            });
            setCategories(response.data.filter((cat: Category) => cat.is_active));
        } catch (error) {
            console.error('Error fetching categories:', error);
        }
    };

    const { data, setData, post, put, processing, errors } = useForm({
        name: course?.name || '',
        short_description: course?.short_description || '',
        is_active: course?.is_active || false,
        category: course?.category || '',
    });

    const submit = (e: React.FormEvent) => {
        e.preventDefault();

        if (isEditing) {
            put(route('courses.update', course.id), {
                onSuccess: onSuccess,
            });
        } else {
            post(route('courses.store'), {
                onSuccess: onSuccess,
            });
        }
    };

    return (
        <form onSubmit={submit} className="space-y-6">
            <div>
                <InputLabel htmlFor="name" value="Course Title" />
                <TextInput
                    id="name"
                    type="text"
                    className="mt-1 block w-full"
                    value={data.name}
                    onChange={(e) => setData('name', e.target.value)}
                    required
                />
                <InputError message={errors.name} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="short_description" value="Course Description" />
                <textarea
                    id="short_description"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    rows={4}
                    value={data.short_description}
                    onChange={(e) => setData('short_description', e.target.value)}
                    required
                />
                <InputError message={errors.short_description} className="mt-2" />
            </div>

            <div>
                <InputLabel htmlFor="category" value="Course Category" />
                <select
                    id="category"
                    className="mt-1 block w-full border-gray-300 focus:border-indigo-500 focus:ring-indigo-500 rounded-md shadow-sm"
                    value={data.category}
                    onChange={(e) => setData('category', e.target.value)}
                    required
                >
                    <option value="">Select a category</option>
                    {categories.map((cat) => (
                        <option key={cat.id} value={cat.name}>
                            {cat.name}
                        </option>
                    ))}
                </select>
                <InputError message={errors.category} className="mt-2" />
            </div>

            <div className="flex items-center">
                <input
                    id="is_active"
                    type="checkbox"
                    className="rounded border-gray-300 text-indigo-600 shadow-sm focus:ring-indigo-500"
                    checked={data.is_active}
                    onChange={(e) => setData('is_active', e.target.checked)}
                />
                <InputLabel htmlFor="is_active" value="Publish Course" className="ml-2" />
            </div>

            <div className="flex items-center justify-end">
                <PrimaryButton disabled={processing}>
                    {isEditing ? 'Update Course' : 'Create Course'}
                </PrimaryButton>
            </div>
        </form>
    );
}