<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;

class DemoCoursesSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Course 1: Introduction to Web Development
        $course1 = Course::create([
            'name' => 'Introduction to Web Development',
            'slug' => 'intro-to-web-dev',
            'short_description' => 'Learn the fundamentals of web development including HTML, CSS, and JavaScript basics.',
            'chunk_words' => 800,
            'has_prerequisite' => false,
            'is_active' => true,
            'certificate_template' => '<div style="text-align: center; padding: 50px; border: 3px solid #667eea;">
                <h1 style="color: #667eea;">Certificate of Completion</h1>
                <p style="font-size: 18px; margin: 20px 0;">This is to certify that</p>
                <h2 style="color: #764ba2;">{{user_name}}</h2>
                <p style="font-size: 16px;">has successfully completed</p>
                <h3 style="color: #667eea;">Introduction to Web Development</h3>
                <p style="margin-top: 30px;">Date: {{completion_date}}</p>
                <p style="font-style: italic; color: #666;">Score: {{score}}%</p>
            </div>',
            'completion_instructions' => null,
        ]);

        // Lesson 1.1: HTML Basics
        $lesson1_1 = Lesson::create([
            'course_id' => $course1->id,
            'name' => 'HTML Fundamentals',
            'intro' => 'Learn the building blocks of web pages with HTML elements and structure.',
            'order_index' => 1,
            'share_enabled' => false,
        ]);

        Topic::create([
            'lesson_id' => $lesson1_1->id,
            'title' => 'What is HTML?',
            'body' => '<h2>Introduction to HTML</h2><p>HTML (HyperText Markup Language) is the standard markup language for creating web pages. It describes the structure of a web page semantically and originally derived from SGML.</p><p><strong>Key Points:</strong></p><ul><li>HTML elements are the building blocks of web pages</li><li>Elements are represented by tags enclosed in angle brackets</li><li>Tags usually come in pairs: opening tag and closing tag</li><li>HTML5 is the latest version with many new features</li></ul><p style="background: #f0f9ff; padding: 15px; border-left: 4px solid #3b82f6; margin: 20px 0;"><strong>💡 Fun Fact:</strong> The first website ever created is still online at CERN and was published on August 6, 1991!</p>',
            'video_url' => 'https://www.youtube.com/watch?v=UB1O30fR-EE',
            'image_url' => 'https://images.unsplash.com/photo-1542831371-29b0f74f9713?w=800',
            'order_index' => 1,
            'is_published' => true,
        ]);

        Topic::create([
            'lesson_id' => $lesson1_1->id,
            'title' => 'HTML Document Structure',
            'body' => '<h2>Basic HTML Structure</h2><p>Every HTML document follows a standard structure. Let\'s break down each part:</p><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px; overflow-x: auto;"><code>&lt;!DOCTYPE html&gt;  &lt;!-- Declares this is HTML5 --&gt;
&lt;html lang="en"&gt;  &lt;!-- Root element --&gt;
  &lt;head&gt;
    &lt;meta charset="UTF-8"&gt;  &lt;!-- Character encoding --&gt;
    &lt;meta name="viewport" content="width=device-width, initial-scale=1.0"&gt;
    &lt;title&gt;My First Web Page&lt;/title&gt;  &lt;!-- Browser tab title --&gt;
  &lt;/head&gt;
  &lt;body&gt;
    &lt;h1&gt;Welcome to My Website&lt;/h1&gt;
    &lt;p&gt;This is my first paragraph.&lt;/p&gt;
  &lt;/body&gt;
&lt;/html&gt;</code></pre><p><strong>Structure Breakdown:</strong></p><table style="width: 100%; border-collapse: collapse; margin: 20px 0;"><tr style="background: #f3f4f6;"><th style="border: 1px solid #ddd; padding: 12px; text-align: left;">Element</th><th style="border: 1px solid #ddd; padding: 12px; text-align: left;">Purpose</th></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><code>&lt;!DOCTYPE html&gt;</code></td><td style="border: 1px solid #ddd; padding: 8px;">Document type declaration</td></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><code>&lt;html&gt;</code></td><td style="border: 1px solid #ddd; padding: 8px;">Root element of the page</td></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><code>&lt;head&gt;</code></td><td style="border: 1px solid #ddd; padding: 8px;">Contains metadata and links</td></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><code>&lt;body&gt;</code></td><td style="border: 1px solid #ddd; padding: 8px;">Contains visible page content</td></tr></table>',
            'video_url' => 'https://www.youtube.com/watch?v=ok-plXXHlWw',
            'image_url' => 'https://images.unsplash.com/photo-1461749280684-dccba630e2f6?w=800',
            'order_index' => 2,
            'is_published' => true,
        ]);

        // Lesson 1.2: CSS Basics
        $lesson1_2 = Lesson::create([
            'course_id' => $course1->id,
            'name' => 'CSS Styling',
            'intro' => 'Make your web pages beautiful with Cascading Style Sheets.',
            'order_index' => 2,
            'share_enabled' => false,
        ]);

        Topic::create([
            'lesson_id' => $lesson1_2->id,
            'title' => 'Introduction to CSS',
            'body' => '<h2>What is CSS?</h2><p>CSS (Cascading Style Sheets) is the language used to style and layout web pages. With CSS, you can control colors, fonts, spacing, positioning, animations, and create responsive designs that work on any device.</p><p><strong>CSS Syntax:</strong></p><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px;"><code>selector {
  property: value;
}

/* Example */
h1 {
  color: #667eea;
  font-size: 32px;
  font-weight: bold;
}</code></pre><div style="background: #fef3c7; border-left: 4px solid #f59e0b; padding: 15px; margin: 20px 0;"><strong>⚡ Pro Tip:</strong> Always use external CSS files linked in the &lt;head&gt; section for better organization and caching!</div>',
            'video_url' => 'https://www.youtube.com/watch?v=yfoY53QXEnI',
            'image_url' => 'https://images.unsplash.com/photo-1507721999472-8ed4421c4af2?w=800',
            'order_index' => 1,
            'is_published' => true,
        ]);

        Topic::create([
            'lesson_id' => $lesson1_2->id,
            'title' => 'CSS Selectors',
            'body' => '<h2>Understanding CSS Selectors</h2><p>Selectors are patterns used to select the elements you want to style. Mastering selectors is key to efficient CSS.</p><table style="width: 100%; border-collapse: collapse; margin: 20px 0;"><thead><tr style="background: #667eea; color: white;"><th style="border: 1px solid #ddd; padding: 12px; text-align: left;">Selector Type</th><th style="border: 1px solid #ddd; padding: 12px; text-align: left;">Syntax</th><th style="border: 1px solid #ddd; padding: 12px; text-align: left;">Example</th></tr></thead><tbody><tr><td style="border: 1px solid #ddd; padding: 8px;"><strong>Element</strong></td><td style="border: 1px solid #ddd; padding: 8px;"><code>element</code></td><td style="border: 1px solid #ddd; padding: 8px;"><code>p { color: blue; }</code></td></tr><tr style="background: #f9fafb;"><td style="border: 1px solid #ddd; padding: 8px;"><strong>Class</strong></td><td style="border: 1px solid #ddd; padding: 8px;"><code>.classname</code></td><td style="border: 1px solid #ddd; padding: 8px;"><code>.highlight { background: yellow; }</code></td></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><strong>ID</strong></td><td style="border: 1px solid #ddd; padding: 8px;"><code>#idname</code></td><td style="border: 1px solid #ddd; padding: 8px;"><code>#header { font-size: 24px; }</code></td></tr><tr style="background: #f9fafb;"><td style="border: 1px solid #ddd; padding: 8px;"><strong>Universal</strong></td><td style="border: 1px solid #ddd; padding: 8px;"><code>*</code></td><td style="border: 1px solid #ddd; padding: 8px;"><code>* { margin: 0; }</code></td></tr><tr><td style="border: 1px solid #ddd; padding: 8px;"><strong>Descendant</strong></td><td style="border: 1px solid #ddd; padding: 8px;"><code>parent child</code></td><td style="border: 1px solid #ddd; padding: 8px;"><code>div p { color: red; }</code></td></tr></tbody></table><div style="background: #dbeafe; border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0;"><strong>📚 Remember:</strong> Classes can be reused on multiple elements, but IDs should be unique on a page!</div>',
            'video_url' => 'https://www.youtube.com/watch?v=l1mER1bV0N0',
            'image_url' => 'https://images.unsplash.com/photo-1523437113738-bbd3cc89fb19?w=800',
            'order_index' => 2,
            'is_published' => true,
        ]);

        // Course 2: Advanced JavaScript Programming
        $course2 = Course::create([
            'name' => 'Advanced JavaScript Programming',
            'slug' => 'advanced-javascript',
            'short_description' => 'Master advanced JavaScript concepts including ES6+, async programming, and modern frameworks.',
            'chunk_words' => 1000,
            'has_prerequisite' => true,
            'prerequisite_course_id' => $course1->id,
            'is_active' => true,
            'certificate_template' => null,
            'completion_instructions' => 'Congratulations on completing Advanced JavaScript Programming! 

Next Steps:
1. Practice building real-world projects using the concepts you learned
2. Explore modern JavaScript frameworks like React, Vue, or Angular
3. Join our advanced masterclass for hands-on project development
4. Connect with fellow developers in our community forum

Keep coding and stay curious! 🚀',
        ]);

        // Lesson 2.1: ES6+ Features
        $lesson2_1 = Lesson::create([
            'course_id' => $course2->id,
            'name' => 'Modern JavaScript Features',
            'intro' => 'Explore ES6+ features that make JavaScript more powerful and elegant.',
            'order_index' => 1,
            'share_enabled' => false,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_1->id,
            'title' => 'Arrow Functions',
            'body' => '<h2>Arrow Functions in ES6</h2><p>Arrow functions provide a more concise syntax for writing function expressions. They also lexically bind the <code>this</code> value, which makes them perfect for callbacks and array methods.</p><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px; overflow-x: auto;"><code>// Traditional function
function add(a, b) {
  return a + b;
}

// Arrow function (concise)
const add = (a, b) => a + b;

// Arrow function with body
const multiply = (a, b) => {
  const result = a * b;
  return result;
};

// Single parameter (parentheses optional)
const square = x => x * x;

// No parameters
const greet = () => console.log("Hello!");</code></pre><h3>Benefits of Arrow Functions:</h3><ul><li><strong>Shorter syntax</strong> - Less boilerplate code</li><li><strong>Implicit return</strong> - One-liners return automatically</li><li><strong>Lexical this</strong> - No need to bind context</li><li><strong>Perfect for array methods</strong> - map, filter, reduce</li></ul><div style="background: #fef3c7; border-left: 4px solid #f59e0b; padding: 15px; margin: 20px 0;"><strong>⚠️ Note:</strong> Arrow functions cannot be used as constructors and don\'t have their own <code>arguments</code> object!</div>',
            'video_url' => 'https://www.youtube.com/watch?v=h33Srr5J9nY',
            'image_url' => 'https://images.unsplash.com/photo-1627398242454-45a1465c2479?w=800',
            'order_index' => 1,
            'is_published' => true,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_1->id,
            'title' => 'Destructuring Assignment',
            'body' => '<h2>ES6 Destructuring</h2><p>Destructuring is a convenient way to extract multiple values from arrays or objects into distinct variables. It makes your code cleaner and more readable.</p><h3>Array Destructuring:</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px;"><code>// Basic array destructuring
const [first, second, third] = [1, 2, 3];
console.log(first);  // 1
console.log(second); // 2

// Skip elements
const [a, , c] = [1, 2, 3];
console.log(a, c);  // 1 3

// Rest operator
const [head, ...tail] = [1, 2, 3, 4, 5];
console.log(head);  // 1
console.log(tail);  // [2, 3, 4, 5]</code></pre><h3>Object Destructuring:</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px;"><code>// Basic object destructuring
const {name, age, city} = {
  name: "John",
  age: 30,
  city: "New York"
};

// Rename variables
const {name: fullName, age: years} = person;

// Default values
const {country = "USA"} = person;

// Nested destructuring
const {address: {street, city}} = person;</code></pre><div style="background: #dcfce7; border-left: 4px solid #22c55e; padding: 15px; margin: 20px 0;"><strong>✨ Pro Tip:</strong> Use destructuring in function parameters to make your APIs cleaner and more readable!</div>',
            'video_url' => 'https://www.youtube.com/watch?v=NIq3qLaHCIs',
            'image_url' => 'https://images.unsplash.com/photo-1555066931-4365d14bab8c?w=800',
            'order_index' => 2,
            'is_published' => true,
        ]);

        // Lesson 2.2: Async JavaScript
        $lesson2_2 = Lesson::create([
            'course_id' => $course2->id,
            'name' => 'Asynchronous Programming',
            'intro' => 'Master promises, async/await, and handling asynchronous operations.',
            'order_index' => 2,
            'share_enabled' => false,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_2->id,
            'title' => 'Understanding Promises',
            'body' => '<h2>JavaScript Promises</h2><p>A Promise is an object representing the eventual completion or failure of an asynchronous operation. Think of it as a "promise" to complete a task in the future.</p><h3>Promise States:</h3><ul><li><strong style="color: #f59e0b;">Pending:</strong> Initial state, neither fulfilled nor rejected</li><li><strong style="color: #22c55e;">Fulfilled:</strong> Operation completed successfully</li><li><strong style="color: #ef4444;">Rejected:</strong> Operation failed</li></ul><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px; overflow-x: auto;"><code>// Creating a Promise
const myPromise = new Promise((resolve, reject) => {
  setTimeout(() => {
    const success = true;
    
    if (success) {
      resolve("Operation successful! 🎉");
    } else {
      reject("Operation failed 😢");
    }
  }, 2000);
});

// Using the Promise
myPromise
  .then(result => {
    console.log(result);
    return "Next step";
  })
  .then(nextResult => {
    console.log(nextResult);
  })
  .catch(error => {
    console.error("Error:", error);
  })
  .finally(() => {
    console.log("Cleanup code here");
  });</code></pre><div style="background: #dbeafe; border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0;"><strong>💡 Key Concept:</strong> Promises allow you to chain multiple asynchronous operations together cleanly, avoiding "callback hell"!</div>',
            'video_url' => 'https://www.youtube.com/watch?v=DHvZLI7Db8E',
            'image_url' => 'https://images.unsplash.com/photo-1516116216624-53e697fedbea?w=800',
            'order_index' => 1,
            'is_published' => true,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_2->id,
            'title' => 'Async/Await',
            'body' => '<h2>Modern Async with Async/Await</h2><p>Async/await is syntactic sugar built on top of Promises. It makes asynchronous code look and behave more like synchronous code, making it easier to read and maintain.</p><h3>Basic Async/Await Syntax:</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px; overflow-x: auto;"><code>// Async function always returns a Promise
async function fetchData() {
  try {
    // Wait for the fetch to complete
    const response = await fetch("https://api.example.com/data");
    
    // Wait for the JSON parsing
    const data = await response.json();
    
    console.log(data);
    return data;
  } catch (error) {
    console.error("Error fetching data:", error);
    throw error;
  }
}

// Calling async function
fetchData()
  .then(data => console.log("Success:", data))
  .catch(err => console.log("Failed:", err));</code></pre><h3>Multiple Parallel Requests:</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px;"><code>async function fetchMultiple() {
  // Run requests in parallel
  const [users, posts, comments] = await Promise.all([
    fetch("/api/users").then(r => r.json()),
    fetch("/api/posts").then(r => r.json()),
    fetch("/api/comments").then(r => r.json())
  ]);
  
  return { users, posts, comments };
}</code></pre><div style="background: #dcfce7; border-left: 4px solid #22c55e; padding: 15px; margin: 20px 0;"><strong>✅ Best Practice:</strong> Always use try/catch blocks with async/await to handle errors properly!</div><div style="background: #fee2e2; border-left: 4px solid #ef4444; padding: 15px; margin: 20px 0;"><strong>⚠️ Warning:</strong> Don\'t forget the <code>await</code> keyword! Without it, you\'ll get a Promise instead of the resolved value.</div>',
            'video_url' => 'https://www.youtube.com/watch?v=V_Kr9OSfDeU',
            'image_url' => 'https://images.unsplash.com/photo-1555949963-ff9fe0c870eb?w=800',
            'order_index' => 2,
            'is_published' => true,
        ]);

        // Lesson 2.3: Project: Build a Todo App
        $lesson2_3 = Lesson::create([
            'course_id' => $course2->id,
            'name' => 'Final Project: Todo Application',
            'intro' => 'Apply everything you learned to build a fully functional todo application.',
            'order_index' => 3,
            'share_enabled' => false,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_3->id,
            'title' => 'Project Requirements',
            'body' => '<h2>Todo Application Requirements</h2><p>In this final project, you\'ll build a fully functional Todo application that demonstrates all the concepts you\'ve learned throughout this course.</p><h3>Core Features (Required):</h3><table style="width: 100%; border-collapse: collapse; margin: 20px 0;"><thead><tr style="background: #f1f5f9;"><th style="padding: 12px; border: 1px solid #cbd5e1; text-align: left;">Feature</th><th style="padding: 12px; border: 1px solid #cbd5e1; text-align: left;">Description</th></tr></thead><tbody><tr><td style="padding: 10px; border: 1px solid #cbd5e1;"><strong>Add Todos</strong></td><td style="padding: 10px; border: 1px solid #cbd5e1;">Users can enter todo text and add it to the list</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;"><strong>Complete Todos</strong></td><td style="padding: 10px; border: 1px solid #cbd5e1;">Click to mark todos as done (with visual indication)</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;"><strong>Delete Todos</strong></td><td style="padding: 10px; border: 1px solid #cbd5e1;">Remove todos from the list permanently</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;"><strong>Filter Todos</strong></td><td style="padding: 10px; border: 1px solid #cbd5e1;">Show All, Active only, or Completed only</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;"><strong>Local Storage</strong></td><td style="padding: 10px; border: 1px solid #cbd5e1;">Persist data between browser sessions</td></tr></tbody></table><h3>Bonus Features (Optional):</h3><ul><li>✨ Edit existing todos inline</li><li>✨ Drag-and-drop to reorder todos</li><li>✨ Due dates with date picker</li><li>✨ Priority levels (high, medium, low)</li><li>✨ Categories or tags</li><li>✨ Dark mode toggle</li></ul><div style="background: #dbeafe; border-left: 4px solid #3b82f6; padding: 15px; margin: 20px 0;"><strong>💡 Pro Tip:</strong> Start with the core features first. Get them working perfectly before adding bonus features. Quality over quantity!</div><h3>Technical Requirements:</h3><ul><li>Use modern JavaScript (ES6+): arrow functions, destructuring, async/await</li><li>Implement proper error handling</li><li>Write clean, commented code</li><li>Ensure responsive design (mobile-friendly)</li><li>Validate user input</li></ul>',
            'video_url' => 'https://www.youtube.com/watch?v=Ttf3CEsEwMQ',
            'image_url' => 'https://images.unsplash.com/photo-1484480974693-6ca0a78fb36b?w=800',
            'order_index' => 1,
            'is_published' => true,
        ]);

        Topic::create([
            'lesson_id' => $lesson2_3->id,
            'title' => 'Getting Started',
            'body' => '<h2>Project Setup Guide</h2><p>Follow these steps to set up your Todo application project from scratch.</p><h3>Step 1: Create Project Structure</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px;"><code>todo-app/
├── index.html
├── css/
│   └── style.css
├── js/
│   ├── app.js
│   ├── storage.js
│   └── ui.js
└── README.md</code></pre><h3>Step 2: HTML Template Starter</h3><pre style="background: #1e293b; color: #e2e8f0; padding: 20px; border-radius: 8px; overflow-x: auto;"><code>&lt;!DOCTYPE html&gt;
&lt;html lang="en"&gt;
&lt;head&gt;
  &lt;meta charset="UTF-8"&gt;
  &lt;meta name="viewport" content="width=device-width, initial-scale=1.0"&gt;
  &lt;title&gt;My Todo App&lt;/title&gt;
  &lt;link rel="stylesheet" href="css/style.css"&gt;
&lt;/head&gt;
&lt;body&gt;
  &lt;div class="container"&gt;
    &lt;h1&gt;My Todos&lt;/h1&gt;
    &lt;div class="input-section"&gt;
      &lt;input type="text" id="todo-input" placeholder="What needs to be done?"&gt;
      &lt;button id="add-btn"&gt;Add&lt;/button&gt;
    &lt;/div&gt;
    &lt;ul id="todo-list"&gt;&lt;/ul&gt;
  &lt;/div&gt;
  &lt;script src="js/app.js" type="module"&gt;&lt;/script&gt;
&lt;/body&gt;
&lt;/html&gt;</code></pre><h3>Step 3: Implementation Checklist</h3><table style="width: 100%; border-collapse: collapse; margin: 20px 0;"><thead><tr style="background: #f1f5f9;"><th style="padding: 12px; border: 1px solid #cbd5e1;">Task</th><th style="padding: 12px; border: 1px solid #cbd5e1; width: 80px;">Status</th></tr></thead><tbody><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Set up HTML structure</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Create CSS styling</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Implement add todo functionality</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Implement complete/uncomplete toggle</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Implement delete functionality</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Add filter buttons</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Implement local storage</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Test on multiple browsers</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr><tr><td style="padding: 10px; border: 1px solid #cbd5e1;">Make responsive for mobile</td><td style="padding: 10px; border: 1px solid #cbd5e1; text-align: center;">☐</td></tr></tbody></table><div style="background: #dcfce7; border-left: 4px solid #22c55e; padding: 15px; margin: 20px 0;"><strong>✅ Testing Tips:</strong><ul style="margin: 10px 0;"><li>Test with empty input</li><li>Test with very long todo text</li><li>Test local storage by refreshing page</li><li>Test all filter combinations</li><li>Test on mobile device or browser dev tools</li></ul></div><div style="background: #fef3c7; border-left: 4px solid #f59e0b; padding: 15px; margin: 20px 0;"><strong>� Resources:</strong> Use MDN Web Docs for LocalStorage API, CSS Flexbox/Grid guides, and JavaScript Event Handling references.</div>',
            'video_url' => 'https://www.youtube.com/watch?v=W7FaYfuwu70',
            'image_url' => 'https://images.unsplash.com/photo-1507925921958-8a62f3d1a50d?w=800',
            'order_index' => 2,
            'is_published' => true,
        ]);

        $this->command->info('✅ Demo courses created successfully!');
        $this->command->info('   - Course 1: Introduction to Web Development (2 lessons, 4 topics)');
        $this->command->info('   - Course 2: Advanced JavaScript Programming (3 lessons, 6 topics)');
    }
}
