<?php

namespace App\Traits;

use Illuminate\Support\Str;

trait HasSlug
{
    /**
     * Boot the trait and add event listeners
     */
    protected static function bootHasSlug()
    {
        static::creating(function ($model) {
            if (empty($model->slug)) {
                $model->slug = $model->generateUniqueSlug($model->getSlugSourceAttribute());
            }
        });

        static::updating(function ($model) {
            if ($model->isDirty($model->getSlugSourceAttribute()) && empty($model->slug)) {
                $model->slug = $model->generateUniqueSlug($model->getSlugSourceAttribute());
            }
        });
    }

    /**
     * Generate a unique slug
     */
    public function generateUniqueSlug($source)
    {
        $slug = Str::slug($source);
        $originalSlug = $slug;
        $counter = 1;

        $query = static::where('slug', $slug);

        // Add scope constraint if method exists (e.g., for lessons within a course)
        if (method_exists($this, 'getSlugScope')) {
            $scopeKey = $this->getSlugScope();
            $query->where($scopeKey, $this->{$scopeKey});
        }

        // Exclude current model if updating
        if ($this->exists) {
            $query->where('id', '!=', $this->id);
        }

        while ($query->exists()) {
            $slug = $originalSlug . '-' . $counter;
            $counter++;
            $query = static::where('slug', $slug);
            
            if (method_exists($this, 'getSlugScope')) {
                $scopeKey = $this->getSlugScope();
                $query->where($scopeKey, $this->{$scopeKey});
            }
            
            if ($this->exists) {
                $query->where('id', '!=', $this->id);
            }
        }

        return $slug;
    }

    /**
     * Get the attribute to use for slug generation
     */
    abstract public function getSlugSourceAttribute();
}
