<?php

namespace App\Support;

use App\Models\SiteSetting;

class BrandSettings
{
    /**
     * Cached brand payload per request.
     */
    protected static ?array $cache = null;

    /**
     * Get branding data shared across views and Inertia props.
     */
    public static function data(): array
    {
        if (static::$cache !== null) {
            return static::$cache;
        }

        $settings = SiteSetting::getAllAsArray();

        $siteName = self::normalize($settings['site_name'] ?? config('app.name', 'EE Micro LMS'), 'EE Micro LMS');
        $tagline = self::normalize($settings['site_description'] ?? null, 'Entrepreneurship Training Hub');
        $description = self::normalize($settings['site_description'] ?? null, $tagline);

        $primary = self::normalizeColor($settings['primary_color'] ?? null, '#667eea');
        $secondary = self::normalizeColor($settings['secondary_color'] ?? null, '#764ba2');
        $success = self::normalizeColor($settings['success_color'] ?? null, '#10b981');
        $warning = self::normalizeColor($settings['warning_color'] ?? null, '#f59e0b');
        $danger = self::normalizeColor($settings['danger_color'] ?? null, '#ef4444');
        $info = self::normalizeColor($settings['info_color'] ?? null, '#3b82f6');

        $logoUrl = self::resolveMediaUrl($settings['logo_path'] ?? null);
        $logoDarkUrl = self::resolveMediaUrl($settings['logo_dark_path'] ?? null);
        $faviconUrl = self::resolveMediaUrl($settings['favicon_path'] ?? null);

        $email = self::normalize($settings['contact_email'] ?? null, 'info@eemicrolms.com');
        $phone = self::normalize($settings['contact_phone'] ?? null);
        $address = self::normalize($settings['contact_address'] ?? null);

        $social = array_filter([
            'facebook' => self::normalize($settings['facebook_url'] ?? null),
            'twitter' => self::normalize($settings['twitter_url'] ?? null),
            'linkedin' => self::normalize($settings['linkedin_url'] ?? null),
            'instagram' => self::normalize($settings['instagram_url'] ?? null),
            'youtube' => self::normalize($settings['youtube_url'] ?? null),
        ]);

        $copyright = self::normalize(
            $settings['copyright_text'] ?? null,
            sprintf('%s © %s', date('Y'), $siteName)
        );

        return static::$cache = [
            'name' => $siteName,
            'tagline' => $tagline,
            'description' => $description,
            'logo_url' => $logoUrl,
            'logo_dark_url' => $logoDarkUrl,
            'favicon_url' => $faviconUrl,
            'colors' => [
                'primary' => $primary,
                'secondary' => $secondary,
                'success' => $success,
                'warning' => $warning,
                'danger' => $danger,
                'info' => $info,
                'primary_dark' => self::adjustBrightness($primary, -18),
                'primary_light' => self::adjustBrightness($primary, 16),
                'secondary_light' => self::adjustBrightness($secondary, 18),
                'gradient' => sprintf('linear-gradient(135deg, %1$s 0%%, %2$s 100%%)', $primary, $secondary),
                'primary_alpha_10' => self::hexToRgba($primary, 0.1),
                'primary_alpha_20' => self::hexToRgba($primary, 0.2),
            ],
            'contact' => [
                'email' => $email,
                'phone' => $phone,
                'phone_href' => $phone ? preg_replace('/[^0-9+]/', '', $phone) : null,
                'address' => $address,
            ],
            'social' => $social,
            'copyright' => $copyright,
        ];
    }

    /**
     * Reset cached result (useful for tests or after settings update).
     */
    public static function forget(): void
    {
        static::$cache = null;
    }

    private static function normalize(?string $value, ?string $fallback = null): ?string
    {
        $value = trim((string) ($value ?? ''));
        if ($value === '') {
            return $fallback;
        }

        return $value;
    }

    private static function normalizeColor(?string $value, string $fallback): string
    {
        $value = trim((string) ($value ?? ''));
        if ($value === '') {
            return $fallback;
        }

        if ($value[0] !== '#') {
            $value = '#' . ltrim($value, '#');
        }

        if (strlen($value) === 4) {
            $value = sprintf('#%1$s%1$s%2$s%2$s%3$s%3$s', $value[1], $value[2], $value[3]);
        }

        return strtolower($value);
    }

    private static function adjustBrightness(string $hex, int $percent): string
    {
        $hex = ltrim($hex, '#');
        if (strlen($hex) !== 6) {
            $hex = str_pad($hex, 6, '0');
        }

        $channels = [
            hexdec(substr($hex, 0, 2)),
            hexdec(substr($hex, 2, 2)),
            hexdec(substr($hex, 4, 2)),
        ];

        $factor = $percent / 100;

        $adjusted = array_map(function ($channel) use ($factor) {
            $value = (int) round($channel + ($channel * $factor));
            return max(0, min(255, $value));
        }, $channels);

        return sprintf('#%02x%02x%02x', $adjusted[0], $adjusted[1], $adjusted[2]);
    }

    private static function hexToRgba(string $hex, float $alpha): string
    {
        $hex = ltrim($hex, '#');
        if (strlen($hex) !== 6) {
            $hex = str_pad($hex, 6, '0');
        }

        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));

        $alpha = max(0, min(1, $alpha));

        return sprintf('rgba(%d, %d, %d, %.2f)', $r, $g, $b, $alpha);
    }

    private static function resolveMediaUrl(?string $path): ?string
    {
        $path = trim((string) ($path ?? ''));
        if ($path === '') {
            return null;
        }

        if (preg_match('/^https?:\/\//i', $path)) {
            return $path;
        }

        return asset(ltrim($path, '/'));
    }
}
