<?php

namespace App\Services;

use App\Models\Course;
use App\Models\StudentDailyStudyTime;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;

class StudyTimeService
{
    /**
     * Get today's study time for a user in a specific course
     */
    public function getTodayStudyTime(int $userId, int $courseId): StudentDailyStudyTime
    {
        return StudentDailyStudyTime::firstOrCreate(
            [
                'user_id' => $userId,
                'course_id' => $courseId,
                'study_date' => Carbon::today()
            ],
            [
                'total_minutes_studied' => 0
            ]
        );
    }

    /**
     * Add study minutes to today's record
     */
    public function addStudyTime(int $userId, int $courseId, int $minutes): StudentDailyStudyTime
    {
        $record = $this->getTodayStudyTime($userId, $courseId);
        $record->increment('total_minutes_studied', $minutes);
        return $record->fresh();
    }

    /**
     * Check if user has reached daily limit
     */
    public function hasReachedDailyLimit(int $userId, int $courseId): array
    {
        $course = Course::find($courseId);
        
        if (!$course || !$course->enable_study_time_limit) {
            return [
                'limited' => false,
                'can_study' => true,
                'message' => 'No time limit set'
            ];
        }

        $todayRecord = $this->getTodayStudyTime($userId, $courseId);
        $minutesStudied = $todayRecord->total_minutes_studied;

        // Check minimum requirement
        $minMinutes = $course->min_study_minutes_per_day ?? 0;
        $maxMinutes = $course->max_study_minutes_per_day;

        // If max limit is set and reached
        if ($maxMinutes && $minutesStudied >= $maxMinutes) {
            return [
                'limited' => true,
                'can_study' => false,
                'minutes_studied' => $minutesStudied,
                'max_minutes' => $maxMinutes,
                'min_minutes' => $minMinutes,
                'remaining_minutes' => 0,
                'message' => "You've reached your daily study limit of {$maxMinutes} minutes. Come back tomorrow!"
            ];
        }

        // Can still study
        $remainingMinutes = $maxMinutes ? ($maxMinutes - $minutesStudied) : null;
        
        return [
            'limited' => false,
            'can_study' => true,
            'minutes_studied' => $minutesStudied,
            'max_minutes' => $maxMinutes,
            'min_minutes' => $minMinutes,
            'remaining_minutes' => $remainingMinutes,
            'message' => $remainingMinutes 
                ? "You have {$remainingMinutes} minutes remaining today" 
                : 'No limit - study as long as you need'
        ];
    }

    /**
     * Check if user has met minimum daily requirement
     */
    public function hasMetMinimumRequirement(int $userId, int $courseId): array
    {
        $course = Course::find($courseId);
        
        if (!$course || !$course->enable_study_time_limit || !$course->min_study_minutes_per_day) {
            return [
                'has_minimum' => false,
                'met_minimum' => true,
                'message' => 'No minimum requirement'
            ];
        }

        $todayRecord = $this->getTodayStudyTime($userId, $courseId);
        $minutesStudied = $todayRecord->total_minutes_studied;
        $minMinutes = $course->min_study_minutes_per_day;

        $metMinimum = $minutesStudied >= $minMinutes;
        $remainingMinutes = max(0, $minMinutes - $minutesStudied);

        return [
            'has_minimum' => true,
            'met_minimum' => $metMinimum,
            'minutes_studied' => $minutesStudied,
            'min_minutes' => $minMinutes,
            'remaining_minutes' => $remainingMinutes,
            'message' => $metMinimum 
                ? "Great! You've met today's minimum of {$minMinutes} minutes" 
                : "Study {$remainingMinutes} more minutes to meet today's goal"
        ];
    }

    /**
     * Get study statistics for a user in a course
     */
    public function getStudyStats(int $userId, int $courseId, int $days = 7): array
    {
        $startDate = Carbon::today()->subDays($days - 1);
        
        $records = StudentDailyStudyTime::where('user_id', $userId)
            ->where('course_id', $courseId)
            ->where('study_date', '>=', $startDate)
            ->orderBy('study_date', 'asc')
            ->get();

        $course = Course::find($courseId);
        
        return [
            'daily_records' => $records,
            'total_minutes' => $records->sum('total_minutes_studied'),
            'average_minutes' => $records->avg('total_minutes_studied') ?? 0,
            'days_studied' => $records->where('total_minutes_studied', '>', 0)->count(),
            'total_days' => $days,
            'course_settings' => [
                'has_limits' => $course && $course->enable_study_time_limit,
                'min_minutes' => $course?->min_study_minutes_per_day,
                'max_minutes' => $course?->max_study_minutes_per_day
            ]
        ];
    }

    /**
     * Start a study session (returns session start time)
     */
    public function startSession(int $userId, int $courseId): array
    {
        $limitStatus = $this->hasReachedDailyLimit($userId, $courseId);
        
        if (!$limitStatus['can_study']) {
            return [
                'can_start' => false,
                'reason' => $limitStatus['message']
            ];
        }

        return [
            'can_start' => true,
            'session_start' => now(),
            'remaining_minutes' => $limitStatus['remaining_minutes']
        ];
    }

    /**
     * End a study session and record the time
     */
    public function endSession(int $userId, int $courseId, Carbon $sessionStart): array
    {
        $minutesStudied = max(1, (int) $sessionStart->diffInMinutes(now()));
        $record = $this->addStudyTime($userId, $courseId, $minutesStudied);
        
        return [
            'minutes_added' => $minutesStudied,
            'total_today' => $record->total_minutes_studied,
            'limit_status' => $this->hasReachedDailyLimit($userId, $courseId)
        ];
    }
}
