<?php

namespace App\Services;

use App\Models\User;
use App\Models\Course;
use App\Models\Topic;
use App\Models\UserProgress;
use Illuminate\Support\Facades\DB;

class ProgressService
{
    /**
     * Get comprehensive user statistics
     */
    public function getUserStatistics(User $user)
    {
        $totalTopics = Topic::whereHas('section.course', function($query) {
            $query->where('is_active', true);
        })->where('is_published', true)->count();

        $completedTopics = UserProgress::where('user_id', $user->id)
            ->where('completed', true)
            ->count();

        $averageQuizScore = UserProgress::where('user_id', $user->id)
            ->whereNotNull('quiz_score')
            ->avg('quiz_score') ?? 0;

        $totalVideoWatchTime = UserProgress::where('user_id', $user->id)
            ->sum('video_watch_seconds');

        $progressPercentage = $totalTopics > 0 
            ? round(($completedTopics / $totalTopics) * 100, 2) 
            : 0;

        return [
            'total_topics' => $totalTopics,
            'completed_topics' => $completedTopics,
            'in_progress_topics' => $this->getInProgressTopicsCount($user->id),
            'average_quiz_score' => round($averageQuizScore, 2),
            'total_video_watch_time' => $totalVideoWatchTime,
            'progress_percentage' => $progressPercentage,
            'total_attempts' => UserProgress::where('user_id', $user->id)->sum('attempts'),
            'last_activity' => UserProgress::where('user_id', $user->id)
                ->latest('updated_at')
                ->value('updated_at'),
        ];
    }

    /**
     * Get in-progress topics count (started but not completed)
     */
    private function getInProgressTopicsCount($userId)
    {
        return UserProgress::where('user_id', $userId)
            ->where('completed', false)
            ->where(function($query) {
                $query->where('video_watch_seconds', '>', 0)
                    ->orWhere('attempts', '>', 0);
            })
            ->count();
    }

    /**
     * Get all users with their progress statistics
     */
    public function getAllUsersWithProgress()
    {
        $users = User::with('roles')->get();
        $totalTopics = Topic::whereHas('section.course', function($query) {
            $query->where('is_active', true);
        })->where('is_published', true)->count();

        return $users->map(function ($user) use ($totalTopics) {
            $completedTopics = UserProgress::where('user_id', $user->id)
                ->where('completed', true)
                ->count();

            $averageQuizScore = UserProgress::where('user_id', $user->id)
                ->whereNotNull('quiz_score')
                ->avg('quiz_score') ?? 0;

            $progressPercentage = $totalTopics > 0 
                ? round(($completedTopics / $totalTopics) * 100, 2) 
                : 0;

            $lastActivity = UserProgress::where('user_id', $user->id)
                ->latest('updated_at')
                ->value('updated_at');

            return [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'roles' => $user->roles->pluck('name')->toArray(),
                'is_admin' => $user->hasRole('admin'),
                'created_at' => $user->created_at->format('Y-m-d H:i:s'),
                'email_verified_at' => $user->email_verified_at ? $user->email_verified_at->format('Y-m-d H:i:s') : null,
                'progress' => [
                    'total_topics' => $totalTopics,
                    'completed_topics' => $completedTopics,
                    'progress_percentage' => $progressPercentage,
                    'average_quiz_score' => round($averageQuizScore, 2),
                    'last_activity' => $lastActivity ? $lastActivity->format('Y-m-d H:i:s') : null,
                ],
            ];
        });
    }

    /**
     * Get course-specific progress for a user
     */
    public function getUserCourseProgress(User $user, Course $course)
    {
        $courseTopics = Topic::whereHas('lesson', function($query) use ($course) {
            $query->where('course_id', $course->id);
        })->where('is_published', true)->pluck('id');

        $completedTopics = UserProgress::where('user_id', $user->id)
            ->whereIn('topic_id', $courseTopics)
            ->where('completed', true)
            ->count();

        $totalTopics = $courseTopics->count();
        $progressPercentage = $totalTopics > 0 
            ? round(($completedTopics / $totalTopics) * 100, 2) 
            : 0;

        return [
            'course_id' => $course->id,
            'course_name' => $course->name,
            'total_topics' => $totalTopics,
            'completed_topics' => $completedTopics,
            'progress_percentage' => $progressPercentage,
        ];
    }

    /**
     * Get dashboard analytics data
     */
    public function getDashboardAnalytics()
    {
        $totalUsers = User::count();
        $activeCourses = Course::where('is_active', true)->count();
        
        // Users who completed all topics
        $totalTopics = Topic::whereHas('section.course', function($query) {
            $query->where('is_active', true);
        })->where('is_published', true)->count();

        $completedUsers = 0;
        if ($totalTopics > 0) {
            $completedUsers = User::whereHas('progress', function($query) use ($totalTopics) {
                $query->where('completed', true);
            })
            ->withCount(['progress as completed_count' => function($query) {
                $query->where('completed', true);
            }])
            ->get()
            ->filter(function($user) use ($totalTopics) {
                return $user->completed_count >= $totalTopics;
            })
            ->count();
        }

        // Users with at least one topic in progress
        $inProgressUsers = User::whereHas('progress', function($query) {
            $query->where('completed', false)
                ->where(function($q) {
                    $q->where('video_watch_seconds', '>', 0)
                      ->orWhere('attempts', '>', 0);
                });
        })->count();

        $avgScore = UserProgress::whereNotNull('quiz_score')->avg('quiz_score') ?? 0;

        return [
            'users' => $totalUsers,
            'courses' => $activeCourses,
            'completed' => $completedUsers,
            'in_progress' => $inProgressUsers,
            'avg_score' => round($avgScore, 2),
        ];
    }

    /**
     * Get top performing users
     */
    public function getTopPerformers($limit = 10)
    {
        return User::withCount(['progress as completed_count' => function($query) {
            $query->where('completed', true);
        }])
        ->with('roles')
        ->having('completed_count', '>', 0)
        ->orderBy('completed_count', 'desc')
        ->limit($limit)
        ->get()
        ->map(function($user) {
            $avgScore = UserProgress::where('user_id', $user->id)
                ->whereNotNull('quiz_score')
                ->avg('quiz_score') ?? 0;

            return [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'completed_topics' => $user->completed_count,
                'average_score' => round($avgScore, 2),
            ];
        });
    }

    /**
     * Get at-risk users (low engagement)
     */
    public function getAtRiskUsers($daysInactive = 7)
    {
        $cutoffDate = now()->subDays($daysInactive);

        return User::whereDoesntHave('progress', function($query) use ($cutoffDate) {
            $query->where('updated_at', '>', $cutoffDate);
        })
        ->orWhereHas('progress', function($query) use ($cutoffDate) {
            $query->where('updated_at', '<', $cutoffDate);
        })
        ->with('roles')
        ->limit(20)
        ->get()
        ->map(function($user) {
            $lastActivity = UserProgress::where('user_id', $user->id)
                ->latest('updated_at')
                ->value('updated_at');

            return [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'last_activity' => $lastActivity ? $lastActivity->diffForHumans() : 'Never',
            ];
        });
    }
}
