<?php

namespace App\Services;

use App\Models\User;
use App\Models\PenaltyRule;
use App\Models\UserPenaltyCard;
use App\Models\PenaltyHistory;
use App\Models\UserActivityLog;
use App\Notifications\YellowCardNotification;
use App\Notifications\RedCardNotification;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;

class PenaltyService
{
    /**
     * Check all active rules and issue penalties where needed
     */
    public function checkAndIssuePenalties(): array
    {
        $issued = [];
        $rules = PenaltyRule::active()->orderBy('order')->get();

        foreach ($rules as $rule) {
            $affectedUsers = $this->checkRuleForAllUsers($rule);
            $issued = array_merge($issued, $affectedUsers);
        }

        return $issued;
    }

    /**
     * Check a specific rule for all users
     */
    protected function checkRuleForAllUsers(PenaltyRule $rule): array
    {
        $issued = [];
        $users = User::whereDoesntHave('roles', function ($query) {
            $query->whereIn('name', ['admin', 'tutor']);
        })->get();

        foreach ($users as $user) {
            if ($this->shouldIssuePenalty($user, $rule)) {
                $card = $this->issuePenaltyCard($user, $rule);
                if ($card) {
                    $issued[] = [
                        'user' => $user,
                        'card' => $card,
                        'rule' => $rule
                    ];
                }
            }
        }

        return $issued;
    }

    /**
     * Check if penalty should be issued to user
     */
    protected function shouldIssuePenalty(User $user, PenaltyRule $rule): bool
    {
        // Don't issue if user already has active card of this type from this rule
        $existingCard = $user->activePenaltyCards()
            ->where('penalty_rule_id', $rule->id)
            ->where('card_type', $rule->card_type)
            ->first();

        if ($existingCard) {
            return false;
        }

        switch ($rule->trigger_type) {
            case 'inactivity_days':
                return $this->checkInactivity($user, $rule->threshold_value);
            
            case 'missed_deadline':
                return $this->checkMissedDeadlines($user, $rule->threshold_value);
            
            case 'failed_quiz':
                return $this->checkFailedQuizzes($user, $rule->threshold_value);
            
            default:
                return false;
        }
    }

    /**
     * Check if user has been inactive for threshold days
     */
    protected function checkInactivity(User $user, int $thresholdDays): bool
    {
        $lastActivity = $user->getLastActivityAt();
        
        if (!$lastActivity) {
            // User has never been active
            $daysSinceCreation = $user->created_at->diffInDays(now());
            return $daysSinceCreation >= $thresholdDays;
        }

        $daysSinceActive = $lastActivity->diffInDays(now());
        return $daysSinceActive >= $thresholdDays;
    }

    /**
     * Check missed deadlines (placeholder for future implementation)
     */
    protected function checkMissedDeadlines(User $user, int $threshold): bool
    {
        // TODO: Implement deadline tracking
        return false;
    }

    /**
     * Check failed quizzes (placeholder for future implementation)
     */
    protected function checkFailedQuizzes(User $user, int $threshold): bool
    {
        // TODO: Implement quiz failure tracking
        return false;
    }

    /**
     * Issue a penalty card to a user
     */
    public function issuePenaltyCard(
        User $user, 
        PenaltyRule $rule, 
        ?User $issuedBy = null, 
        ?string $customReason = null
    ): ?UserPenaltyCard {
        DB::beginTransaction();
        
        try {
            // Create the penalty card
            $card = UserPenaltyCard::create([
                'user_id' => $user->id,
                'penalty_rule_id' => $rule->id,
                'issued_by' => $issuedBy?->id,
                'card_type' => $rule->card_type,
                'reason' => $customReason ?? $rule->name,
                'points_deducted' => $rule->points_deduction,
                'issued_at' => now(),
                'is_active' => true,
                'metadata' => [
                    'auto_issued' => !$issuedBy,
                    'threshold_value' => $rule->threshold_value,
                    'trigger_type' => $rule->trigger_type
                ]
            ]);

            // Deduct points from user's badge points
            $this->deductPoints($user, $rule->points_deduction, $card);

            // Log to history
            PenaltyHistory::create([
                'user_id' => $user->id,
                'penalty_card_id' => $card->id,
                'action_type' => 'issued',
                'card_type' => $card->card_type,
                'points_affected' => $rule->points_deduction,
                'description' => "Penalty card issued: {$card->reason}",
                'action_by' => $issuedBy?->id,
                'metadata' => [
                    'rule_id' => $rule->id,
                    'auto_issued' => !$issuedBy
                ]
            ]);

            // Send notification
            $this->sendPenaltyNotification($user, $card);

            DB::commit();

            Log::info("Penalty card issued", [
                'user_id' => $user->id,
                'card_type' => $card->card_type,
                'points_deducted' => $rule->points_deduction
            ]);

            return $card;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to issue penalty card", [
                'user_id' => $user->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Deduct points from user (from badge points)
     */
    protected function deductPoints(User $user, int $points, UserPenaltyCard $card): void
    {
        // Log the point deduction
        PenaltyHistory::create([
            'user_id' => $user->id,
            'penalty_card_id' => $card->id,
            'action_type' => 'points_deducted',
            'card_type' => $card->card_type,
            'points_affected' => $points,
            'description' => "Points deducted due to {$card->card_type} card",
            'metadata' => ['reason' => $card->reason]
        ]);

        // Note: Actual point deduction can be calculated dynamically
        // by summing badge points minus penalty points
    }

    /**
     * Check and escalate yellow cards to red cards
     */
    public function checkAndEscalateCards(): array
    {
        $escalated = [];
        
        $yellowCards = UserPenaltyCard::active()
            ->yellowCards()
            ->whereNotNull('penalty_rule_id')
            ->whereNull('escalated_at')
            ->with('penaltyRule', 'user')
            ->get();

        foreach ($yellowCards as $card) {
            if ($this->shouldEscalate($card)) {
                $redCard = $this->escalateToRedCard($card);
                if ($redCard) {
                    $escalated[] = [
                        'yellow_card' => $card,
                        'red_card' => $redCard,
                        'user' => $card->user
                    ];
                }
            }
        }

        return $escalated;
    }

    /**
     * Check if yellow card should be escalated to red
     */
    protected function shouldEscalate(UserPenaltyCard $card): bool
    {
        if (!$card->penaltyRule || !$card->penaltyRule->escalation_days) {
            return false;
        }

        $daysSinceIssued = $card->daysSinceIssued();
        return $daysSinceIssued >= $card->penaltyRule->escalation_days;
    }

    /**
     * Escalate yellow card to red card
     */
    public function escalateToRedCard(UserPenaltyCard $yellowCard): ?UserPenaltyCard
    {
        DB::beginTransaction();

        try {
            $rule = $yellowCard->penaltyRule;
            $user = $yellowCard->user;

            // Mark yellow card as escalated
            $yellowCard->update(['escalated_at' => now()]);

            // Create red card
            $redCard = UserPenaltyCard::create([
                'user_id' => $user->id,
                'penalty_rule_id' => $rule->id,
                'issued_by' => null,
                'card_type' => 'red',
                'reason' => "Escalated from yellow card: {$yellowCard->reason}",
                'points_deducted' => $rule->escalation_points ?? $rule->points_deduction,
                'issued_at' => now(),
                'is_active' => true,
                'metadata' => [
                    'escalated_from' => $yellowCard->id,
                    'auto_escalated' => true
                ]
            ]);

            // Deduct additional points
            if ($rule->escalation_points) {
                $this->deductPoints($user, $rule->escalation_points, $redCard);
            }

            // Log to history
            PenaltyHistory::create([
                'user_id' => $user->id,
                'penalty_card_id' => $redCard->id,
                'action_type' => 'escalated',
                'card_type' => 'red',
                'points_affected' => $rule->escalation_points ?? 0,
                'description' => "Yellow card escalated to red card",
                'metadata' => [
                    'yellow_card_id' => $yellowCard->id,
                    'days_since_yellow' => $yellowCard->daysSinceIssued()
                ]
            ]);

            // Send notification
            $this->sendPenaltyNotification($user, $redCard);

            DB::commit();

            Log::info("Yellow card escalated to red", [
                'user_id' => $user->id,
                'yellow_card_id' => $yellowCard->id,
                'red_card_id' => $redCard->id
            ]);

            return $redCard;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to escalate card", [
                'yellow_card_id' => $yellowCard->id,
                'error' => $e->getMessage()
            ]);
            return null;
        }
    }

    /**
     * Revoke a penalty card
     */
    public function revokeCard(UserPenaltyCard $card, User $revokedBy, string $reason): bool
    {
        DB::beginTransaction();

        try {
            $card->update([
                'is_active' => false,
                'revoked_at' => now(),
                'revoked_by' => $revokedBy->id,
                'revoke_reason' => $reason
            ]);

            // Log to history
            PenaltyHistory::create([
                'user_id' => $card->user_id,
                'penalty_card_id' => $card->id,
                'action_type' => 'revoked',
                'card_type' => $card->card_type,
                'points_affected' => 0,
                'description' => "Card revoked: {$reason}",
                'action_by' => $revokedBy->id,
                'metadata' => ['revoke_reason' => $reason]
            ]);

            DB::commit();

            Log::info("Penalty card revoked", [
                'card_id' => $card->id,
                'user_id' => $card->user_id,
                'revoked_by' => $revokedBy->id
            ]);

            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Failed to revoke card", [
                'card_id' => $card->id,
                'error' => $e->getMessage()
            ]);
            return false;
        }
    }

    /**
     * Send penalty notification to user
     */
    protected function sendPenaltyNotification(User $user, UserPenaltyCard $card): void
    {
        try {
            if ($card->card_type === 'yellow') {
                $user->notify(new YellowCardNotification($card));
            } else {
                $user->notify(new RedCardNotification($card));
            }

            // Log notification sent
            PenaltyHistory::create([
                'user_id' => $user->id,
                'penalty_card_id' => $card->id,
                'action_type' => 'notification_sent',
                'card_type' => $card->card_type,
                'points_affected' => 0,
                'description' => "Penalty notification sent to {$user->email}",
                'metadata' => ['notification_type' => $card->card_type . '_card']
            ]);

        } catch (\Exception $e) {
            Log::error("Failed to send penalty notification", [
                'user_id' => $user->id,
                'card_id' => $card->id,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get penalty statistics for a user
     */
    public function getUserPenaltyStats(User $user): array
    {
        return [
            'total_cards' => $user->penaltyCards()->count(),
            'active_cards' => $user->activePenaltyCards()->count(),
            'yellow_cards' => $user->penaltyCards()->yellowCards()->count(),
            'red_cards' => $user->penaltyCards()->redCards()->count(),
            'total_points_lost' => $user->getTotalPenaltyPoints(),
            'days_since_last_activity' => $user->getDaysSinceLastActivity(),
            'has_active_yellow' => $user->hasActiveYellowCard(),
            'has_active_red' => $user->hasActiveRedCard()
        ];
    }

    /**
     * Get system-wide penalty statistics
     */
    public function getSystemStats(): array
    {
        return [
            'total_cards_issued' => UserPenaltyCard::count(),
            'active_cards' => UserPenaltyCard::active()->count(),
            'yellow_cards' => UserPenaltyCard::yellowCards()->count(),
            'red_cards' => UserPenaltyCard::redCards()->count(),
            'total_points_deducted' => UserPenaltyCard::sum('points_deducted'),
            'users_with_cards' => UserPenaltyCard::distinct('user_id')->count('user_id'),
            'cards_today' => UserPenaltyCard::whereDate('issued_at', today())->count(),
            'escalations_count' => UserPenaltyCard::whereNotNull('escalated_at')->count()
        ];
    }
}
