<?php

namespace App\Services;

use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;
use App\Models\QuizQuestion;
use App\Models\UserProgress;
use App\Models\User;
use App\Models\SiteSetting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CourseService
{
    /**
     * Get all courses with lesson counts and prerequisites
     */
     public function getAllCourses($languageId = null)
    {
         return Course::withCount('lessons')
            ->when($languageId, function ($query) use ($languageId) {
                $query->where('language_id', $languageId);
            })
            ->with([
                'prerequisiteCourse:id,name',
                'language:id,name,code,native_name,flag_emoji',
            ])
            ->orderBy('created_at', 'desc')
            ->get()
            ->map(function ($course) {
                return [
                    'id' => $course->id,
                    'name' => $course->name,
                    'slug' => $course->slug,
                    'short_description' => $course->short_description,
                    'category' => $course->category,
                    'has_prerequisite' => $course->has_prerequisite,
                    'prerequisite_course_id' => $course->prerequisite_course_id,
                    'prerequisite_course_name' => $course->prerequisiteCourse ? $course->prerequisiteCourse->name : null,
                    'is_active' => $course->is_active,
                    'is_free' => $course->is_free,
                    'price' => $course->price,
                    'discount' => $course->discount,
                    'discount_type' => $course->discount_type,
                    'discounted_price' => $course->discounted_price,
                    'currency' => $course->currency,
                    'image_url' => $course->image_url,
                    'thumbnail_url' => $course->thumbnail_url,
                    'chunk_words' => $course->chunk_words,
                    'certificate_template' => $course->certificate_template,
                    'completion_instructions' => $course->completion_instructions,
                    'enable_study_time_limit' => $course->enable_study_time_limit,
                    'min_study_time_per_day' => $course->min_study_time_per_day,
                    'max_study_time_per_day' => $course->max_study_time_per_day,
                    'enable_course_survey' => $course->enable_course_survey,
                    'survey_mode' => $course->survey_mode,
                    'survey_title' => $course->survey_title,
                    'survey_description' => $course->survey_description,
                    'language_id' => $course->language_id,
                    'language' => $course->language ? $course->language->code : null,
                    'language_name' => $course->language->name ?? null,
                    'language_native_name' => $course->language->native_name ?? null,
                    'language_flag_emoji' => $course->language->flag_emoji ?? null,
                    'lesson_count' => $course->lessons_count,
                    'created_at' => $course->created_at->format('Y-m-d H:i:s'),
                    'updated_at' => $course->updated_at->format('Y-m-d H:i:s'),
                ];
            });
    }

    /**
     * Get a single course by ID
     */
    public function getCourseById($courseId)
    {
        $course = Course::with([
                'prerequisiteCourse:id,name',
                'language:id,name,code,native_name,flag_emoji',
            ])
            ->find($courseId);

        if (!$course) {
            return null;
        }

        return [
            'id' => $course->id,
            'name' => $course->name,
            'slug' => $course->slug,
            'short_description' => $course->short_description,
            'category' => $course->category,
            'has_prerequisite' => $course->has_prerequisite,
            'prerequisite_course_id' => $course->prerequisite_course_id,
            'prerequisite_course_name' => $course->prerequisiteCourse ? $course->prerequisiteCourse->name : null,
            'is_active' => $course->is_active,
            'is_free' => $course->is_free,
            'price' => $course->price,
            'discount' => $course->discount,
            'discount_type' => $course->discount_type,
            'discounted_price' => $course->discounted_price,
            'currency' => $course->currency,
            'image_url' => $course->image_url,
            'thumbnail_url' => $course->thumbnail_url,
            'chunk_words' => $course->chunk_words ?? SiteSetting::getChunkWordsDefault(),
            'certificate_template' => $course->certificate_template,
            'completion_instructions' => $course->completion_instructions,
            'enable_study_time_limit' => $course->enable_study_time_limit,
            'min_study_time_per_day' => $course->min_study_time_per_day,
            'max_study_time_per_day' => $course->max_study_time_per_day,
            'enable_course_survey' => $course->enable_course_survey,
            'survey_mode' => $course->survey_mode,
            'survey_title' => $course->survey_title,
            'survey_description' => $course->survey_description,
            'language_id' => $course->language_id,
            'language' => $course->language ? $course->language->code : null,
            'language_name' => $course->language->name ?? null,
            'language_native_name' => $course->language->native_name ?? null,
            'language_flag_emoji' => $course->language->flag_emoji ?? null,
            'created_at' => $course->created_at->format('Y-m-d H:i:s'),
            'updated_at' => $course->updated_at->format('Y-m-d H:i:s'),
        ];
    }

    /**
     * Create a new course
     */
    public function createCourse(array $data)
    {
        // Handle language code conversion to language_id
        if (isset($data['language']) && !isset($data['language_id'])) {
            $language = \App\Models\Language::where('code', $data['language'])->first();
            if ($language) {
                $data['language_id'] = $language->id;
            }
            unset($data['language']);
        }
        
        // If no language specified, use default
        if (empty($data['language_id'])) {
            $defaultLanguage = \App\Models\Language::getDefault();
            if ($defaultLanguage) {
                $data['language_id'] = $defaultLanguage->id;
            }
        }
        
        // Map survey and study-time field names from UI to DB fields
        if (array_key_exists('enable_survey', $data)) {
            $data['enable_course_survey'] = (bool) $data['enable_survey'];
            unset($data['enable_survey']);
        }
        if (!isset($data['enable_course_survey'])) {
            $data['enable_course_survey'] = false;
        }
        if (array_key_exists('min_study_minutes_per_day', $data)) {
            $data['min_study_time_per_day'] = $data['min_study_minutes_per_day'];
            unset($data['min_study_minutes_per_day']);
        }
        if (array_key_exists('max_study_minutes_per_day', $data)) {
            $data['max_study_time_per_day'] = $data['max_study_minutes_per_day'];
            unset($data['max_study_minutes_per_day']);
        }

        $data['currency'] = isset($data['currency']) ? strtoupper(substr($data['currency'], 0, 3)) : 'USD';
        $isFree = array_key_exists('is_free', $data) ? filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) : null;
        $data['is_free'] = $isFree === null ? true : $isFree;
        if ($data['is_free']) {
            $data['price'] = 0;
            $data['discount'] = null;
            $data['discount_type'] = $data['discount_type'] ?? 'amount';
        } else {
            $data['price'] = isset($data['price']) ? (float) $data['price'] : 0;
            $data['is_free'] = false;
            $data['discount_type'] = $data['discount_type'] ?? 'amount';
        }

        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']);
        }

        // Ensure slug is unique
        $originalSlug = $data['slug'];
        $counter = 1;
        while (Course::where('slug', $data['slug'])->exists()) {
            $data['slug'] = $originalSlug . '-' . $counter;
            $counter++;
        }

        // Normalize boolean values
        $data['has_prerequisite'] = isset($data['has_prerequisite']) ? (bool)$data['has_prerequisite'] : false;
        $data['is_active'] = isset($data['is_active']) ? (bool)$data['is_active'] : true;
        
        // Set default chunk_words if not provided
        $data['chunk_words'] = $data['chunk_words'] ?? SiteSetting::getChunkWordsDefault();

        // Normalize survey fields
        if (isset($data['enable_course_survey'])) {
            $data['enable_course_survey'] = (bool) $data['enable_course_survey'];
        }
        if (empty($data['survey_mode']) || !in_array($data['survey_mode'], ['feedback', 'survey'])) {
            $data['survey_mode'] = 'feedback';
        }

        // Handle prerequisite
        if ($data['has_prerequisite'] && empty($data['prerequisite_course_id'])) {
            throw new \Exception('Prerequisite course ID is required when has_prerequisite is true');
        }
        
        if (!$data['has_prerequisite']) {
            $data['prerequisite_course_id'] = null;
        }

        $course = Course::create($data);

        // Create certificate assignment if certificate is selected
        if (!empty($data['certificate_template'])) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Course::class,
                    'assignable_id' => $course->id,
                ],
                [
                    'certificate_template_id' => $data['certificate_template'],
                ]
            );
        }

        return $this->getCourseById($course->id);
    }

    /**
     * Update an existing course
     */
    public function updateCourse($courseId, array $data)
    {
        $course = Course::find($courseId);

        if (!$course) {
            throw new \Exception('Course not found');
        }

        // Handle language code conversion to language_id
        if (isset($data['language']) && !isset($data['language_id'])) {
            $language = \App\Models\Language::where('code', $data['language'])->first();
            if ($language) {
                $data['language_id'] = $language->id;
            }
            unset($data['language']);
        }

        // Map survey and study-time field names from UI to DB fields
        if (array_key_exists('enable_survey', $data)) {
            $data['enable_course_survey'] = (bool) $data['enable_survey'];
            unset($data['enable_survey']);
        }
        if (array_key_exists('min_study_minutes_per_day', $data)) {
            $data['min_study_time_per_day'] = $data['min_study_minutes_per_day'];
            unset($data['min_study_minutes_per_day']);
        }
        if (array_key_exists('max_study_minutes_per_day', $data)) {
            $data['max_study_time_per_day'] = $data['max_study_minutes_per_day'];
            unset($data['max_study_minutes_per_day']);
        }

        if (isset($data['currency'])) {
            $data['currency'] = strtoupper(substr($data['currency'], 0, 3));
        } else {
            $data['currency'] = $course->currency ?? 'USD';
        }

        if (array_key_exists('is_free', $data)) {
            $isFree = filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
            $data['is_free'] = $isFree === null ? true : $isFree;
        } else {
            $data['is_free'] = $course->is_free ?? true;
        }

        if ($data['is_free']) {
            $data['price'] = 0;
            $data['discount'] = null;
            $data['discount_type'] = $data['discount_type'] ?? 'amount';
        } else {
            $data['price'] = isset($data['price']) ? (float) $data['price'] : ($course->price ?? 0);
            $data['is_free'] = false;
            $data['discount_type'] = $data['discount_type'] ?? ($course->discount_type ?? 'amount');
        }

        // Update slug if name changed
        if (isset($data['name']) && $data['name'] !== $course->name) {
            $data['slug'] = Str::slug($data['name']);
            
            // Ensure slug is unique
            $originalSlug = $data['slug'];
            $counter = 1;
            while (Course::where('slug', $data['slug'])->where('id', '!=', $courseId)->exists()) {
                $data['slug'] = $originalSlug . '-' . $counter;
                $counter++;
            }
        }

        // Normalize boolean values
        if (isset($data['has_prerequisite'])) {
            $data['has_prerequisite'] = (bool)$data['has_prerequisite'];
        }
        if (isset($data['is_active'])) {
            $data['is_active'] = (bool)$data['is_active'];
        }
        if (isset($data['enable_course_survey'])) {
            $data['enable_course_survey'] = (bool) $data['enable_course_survey'];
        }
        if (!array_key_exists('survey_mode', $data) || !in_array($data['survey_mode'], ['feedback', 'survey'])) {
            $data['survey_mode'] = 'feedback';
        }

        // Handle prerequisite
        if (isset($data['has_prerequisite'])) {
            if ($data['has_prerequisite'] && empty($data['prerequisite_course_id'])) {
                throw new \Exception('Prerequisite course ID is required when has_prerequisite is true');
            }
            
            if (!$data['has_prerequisite']) {
                $data['prerequisite_course_id'] = null;
            }
        }

        $course->update($data);

        // Update or delete certificate assignment
        if (!empty($data['certificate_template'])) {
            \App\Models\CertificateAssignment::updateOrCreate(
                [
                    'assignable_type' => \App\Models\Course::class,
                    'assignable_id' => $course->id,
                ],
                [
                    'certificate_template_id' => $data['certificate_template'],
                ]
            );
        } else {
            // Remove certificate assignment if empty
            \App\Models\CertificateAssignment::where([
                'assignable_type' => \App\Models\Course::class,
                'assignable_id' => $course->id,
            ])->delete();
        }

        return $this->getCourseById($course->id);
    }

    /**
     * Delete a course and all related data (cascade)
     */
    public function deleteCourse($courseId)
    {
        $course = Course::find($courseId);

        if (!$course) {
            throw new \Exception('Course not found');
        }

        DB::beginTransaction();
        
        try {
            // Get all lessons in this course
            $lessons = Lesson::where('course_id', $courseId)->pluck('id');

            if ($lessons->isNotEmpty()) {
                // Delete quiz questions first (they have FK to lessons)
                QuizQuestion::whereIn('lesson_id', $lessons)->delete();
                
                // Get all topics in these lessons
                $topics = Topic::whereIn('lesson_id', $lessons)->pluck('id');

                if ($topics->isNotEmpty()) {
                    // Delete user progress for these topics
                    UserProgress::whereIn('topic_id', $topics)->delete();

                    // Delete topics (quiz questions will cascade if foreign key is set)
                    Topic::whereIn('id', $topics)->delete();
                }

                // Delete lessons
                Lesson::whereIn('id', $lessons)->delete();
            }

            // Remove this course as prerequisite from other courses
            Course::where('prerequisite_course_id', $courseId)->update([
                'has_prerequisite' => false,
                'prerequisite_course_id' => null
            ]);

            // Delete the course
            $course->delete();

            DB::commit();

            return true;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    /**
     * Get prerequisites for a course
     */
    public function getPrerequisites($courseId)
    {
        $course = Course::find($courseId);

        if (!$course) {
            return [];
        }

        if (!$course->has_prerequisite || !$course->prerequisite_course_id) {
            return [];
        }

        $prerequisite = Course::where('id', $course->prerequisite_course_id)
            ->where('is_active', true)
            ->first();

        return $prerequisite ? [$prerequisite] : [];
    }

    /**
     * Check if a user has completed prerequisites for a course
     */
    public function checkPrerequisites(User $user, $courseId)
    {
        $course = Course::find($courseId);

        if (!$course || !$course->has_prerequisite || !$course->prerequisite_course_id) {
            return [
                'has_access' => true,
                'prerequisites' => [],
                'completed' => []
            ];
        }

        $prerequisite = Course::find($course->prerequisite_course_id);
        
        if (!$prerequisite) {
            return [
                'has_access' => true,
                'prerequisites' => [],
                'completed' => []
            ];
        }

        // Get all topics in prerequisite course
        $prerequisiteTopics = Topic::whereHas('lesson', function ($query) use ($course) {
            $query->where('course_id', $course->prerequisite_course_id);
        })->pluck('id');

        // Check if user completed all topics
        $completedTopics = UserProgress::where('user_id', $user->id)
            ->whereIn('topic_id', $prerequisiteTopics)
            ->where('completed', true)
            ->pluck('topic_id');

        $hasAccess = $prerequisiteTopics->count() === $completedTopics->count();

        return [
            'has_access' => $hasAccess,
            'prerequisites' => [$prerequisite],
            'completed' => $completedTopics->count(),
            'total' => $prerequisiteTopics->count()
        ];
    }

    /**
     * Get course statistics
     */
    public function getCourseStats($courseId)
    {
        $course = Course::withCount('lessons')->find($courseId);

        if (!$course) {
            return null;
        }

        // Get total topics
        $topicCount = Topic::whereHas('lesson', function ($query) use ($courseId) {
            $query->where('course_id', $courseId);
        })->count();

        // Get total enrolled users (users who have progress in this course)
        $enrolledUsers = UserProgress::whereHas('topic', function ($query) use ($courseId) {
            $query->whereHas('lesson', function ($q) use ($courseId) {
                $q->where('course_id', $courseId);
            });
        })->distinct('user_id')->count('user_id');

        // Get completed users (users who completed all topics)
        $topicIds = Topic::whereHas('lesson', function ($query) use ($courseId) {
            $query->where('course_id', $courseId);
        })->pluck('id');

        $completedUsers = 0;
        if ($topicIds->isNotEmpty()) {
            $usersWithProgress = UserProgress::whereIn('topic_id', $topicIds)
                ->select('user_id', DB::raw('COUNT(DISTINCT topic_id) as completed_topics'))
                ->where('completed', true)
                ->groupBy('user_id')
                ->having('completed_topics', '=', $topicIds->count())
                ->count();
            
            $completedUsers = $usersWithProgress;
        }

        // Get average quiz score
        $avgQuizScore = UserProgress::whereHas('topic', function ($query) use ($courseId) {
            $query->whereHas('lesson', function ($q) use ($courseId) {
                $q->where('course_id', $courseId);
            });
        })->where('quiz_score', '>', 0)->avg('quiz_score');

        return [
            'course_id' => $courseId,
            'course_name' => $course->name,
            'lesson_count' => $course->lessons_count,
            'topic_count' => $topicCount,
            'enrolled_users' => $enrolledUsers,
            'completed_users' => $completedUsers,
            'completion_rate' => $enrolledUsers > 0 ? round(($completedUsers / $enrolledUsers) * 100, 1) : 0,
            'avg_quiz_score' => $avgQuizScore ? round($avgQuizScore, 1) : 0
        ];
    }

    /**
     * Get all available courses for prerequisites (excluding the specified course)
     */
    public function getAvailablePrerequisites($excludeCourseId = null)
    {
        $query = Course::where('is_active', true)
            ->select('id', 'name', 'slug');

        if ($excludeCourseId) {
            $query->where('id', '!=', $excludeCourseId);
        }

        return $query->orderBy('name')->get();
    }
}
