<?php

namespace App\Notifications;

use App\Models\UserPenaltyCard;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class YellowCardNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $penaltyCard;

    /**
     * Create a new notification instance.
     */
    public function __construct(UserPenaltyCard $penaltyCard)
    {
        $this->penaltyCard = $penaltyCard;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $rule = $this->penaltyCard->penaltyRule;
        $pointsLost = $this->penaltyCard->points_deducted;
        
        return (new MailMessage)
            ->subject('⚠️ Yellow Card Warning - Action Required')
            ->greeting("Hello {$notifiable->name},")
            ->line('**You have received a Yellow Card warning in the EE Micro LMS.**')
            ->line("**Reason:** {$this->penaltyCard->reason}")
            ->line("**Points Deducted:** {$pointsLost} points")
            ->line('---')
            ->line('**What does this mean?**')
            ->line('You have been inactive or missed important deadlines. This is a warning to help you get back on track.')
            ->line('---')
            ->line('**Next Steps:**')
            ->line('1. Log in to your account and resume your courses')
            ->line('2. Complete any pending lessons or quizzes')
            ->line('3. Stay active to avoid further penalties')
            ->line('---')
            ->line('**⚠️ Important Warning:**')
            ->line($this->getEscalationWarning())
            ->action('Resume Learning', url('/dashboard'))
            ->line('If you have any questions or need assistance, please contact your instructor or administrator.')
            ->salutation('Best regards, EE Micro LMS Team');
    }

    /**
     * Get escalation warning text
     */
    protected function getEscalationWarning(): string
    {
        $rule = $this->penaltyCard->penaltyRule;
        
        if ($rule && $rule->escalation_days) {
            return "If you remain inactive for {$rule->escalation_days} more days, this yellow card will be escalated to a RED CARD with additional point deductions of {$rule->escalation_points} points.";
        }
        
        return "Continued inactivity may result in further penalties including a RED CARD and complete point reset.";
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'yellow_card',
            'card_id' => $this->penaltyCard->id,
            'reason' => $this->penaltyCard->reason,
            'points_deducted' => $this->penaltyCard->points_deducted,
            'issued_at' => $this->penaltyCard->issued_at->toDateTimeString(),
            'message' => "You've received a Yellow Card warning. {$this->penaltyCard->points_deducted} points deducted."
        ];
    }
}
