<?php

namespace App\Notifications;

use App\Models\UserPenaltyCard;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class RedCardNotification extends Notification implements ShouldQueue
{
    use Queueable;

    protected $penaltyCard;

    /**
     * Create a new notification instance.
     */
    public function __construct(UserPenaltyCard $penaltyCard)
    {
        $this->penaltyCard = $penaltyCard;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $pointsLost = $this->penaltyCard->points_deducted;
        $isEscalated = !empty($this->penaltyCard->metadata['escalated_from']);
        
        $subject = $isEscalated 
            ? '🚫 RED CARD - Yellow Card Escalated' 
            : '🚫 RED CARD - Immediate Action Required';
        
        $message = (new MailMessage)
            ->subject($subject)
            ->greeting("Hello {$notifiable->name},")
            ->line('**You have received a RED CARD penalty in the EE Micro LMS.**')
            ->line("**Reason:** {$this->penaltyCard->reason}")
            ->line("**Points Deducted:** {$pointsLost} points");
        
        if ($isEscalated) {
            $message->line('---')
                ->line('**Escalation Notice:**')
                ->line('Your previous Yellow Card warning has been escalated to a Red Card due to continued inactivity.');
        }
        
        $message->line('---')
            ->line('**What does this mean?**')
            ->line('This is a serious penalty indicating extended inactivity or repeated violations.')
            ->line('Your progress and achievements have been significantly impacted.')
            ->line('---')
            ->line('**Immediate Action Required:**')
            ->line('1. **Log in immediately** and resume your courses')
            ->line('2. **Contact your instructor** if you need assistance')
            ->line('3. **Complete outstanding work** to demonstrate commitment')
            ->line('4. **Maintain consistent activity** to avoid future penalties')
            ->line('---')
            ->line('**🚫 Critical Warning:**')
            ->line('Continued inactivity may result in course suspension or additional administrative actions.')
            ->action('Resume Learning NOW', url('/dashboard'))
            ->line('---')
            ->line('**Need Help?**')
            ->line('If you\'re facing difficulties or challenges that prevented your participation, please reach out to your instructor or administrator. We\'re here to support your success.')
            ->salutation('Urgent regards, EE Micro LMS Team');
        
        return $message;
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'red_card',
            'card_id' => $this->penaltyCard->id,
            'reason' => $this->penaltyCard->reason,
            'points_deducted' => $this->penaltyCard->points_deducted,
            'issued_at' => $this->penaltyCard->issued_at->toDateTimeString(),
            'is_escalated' => !empty($this->penaltyCard->metadata['escalated_from']),
            'message' => "You've received a RED CARD penalty. {$this->penaltyCard->points_deducted} points deducted. Immediate action required!"
        ];
    }
}
