<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;

class CustomPage extends Model
{
    use SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'title',
        'slug',
        'content',
        'meta_description',
        'meta_keywords',
        'template',
        'is_published',
        'is_static',
        'order',
        'published_at',
        'created_by',
        'updated_by',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'content' => 'array',
        'is_published' => 'boolean',
        'is_static' => 'boolean',
        'published_at' => 'datetime',
        'order' => 'integer',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [];

    /**
     * Boot the model.
     */
    protected static function boot()
    {
        parent::boot();

        // Automatically generate slug from title if not provided
        static::creating(function ($page) {
            if (empty($page->slug)) {
                $page->slug = Str::slug($page->title);
            }
        });

        // Update slug if title changes
        static::updating(function ($page) {
            if ($page->isDirty('title') && empty($page->slug)) {
                $page->slug = Str::slug($page->title);
            }
        });
    }

    /**
     * Get the user who created the page.
     *
     * @return BelongsTo
     */
    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the user who last updated the page.
     *
     * @return BelongsTo
     */
    public function updater(): BelongsTo
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /**
     * Scope a query to only include published pages.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopePublished($query)
    {
        return $query->where('is_published', true)
                     ->whereNotNull('published_at')
                     ->where('published_at', '<=', now());
    }

    /**
     * Scope a query to only include static pages.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeStatic($query)
    {
        return $query->where('is_static', true);
    }

    /**
     * Scope a query to only include custom (non-static) pages.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeCustom($query)
    {
        return $query->where('is_static', false);
    }

    /**
     * Scope a query to order pages by their order column.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public function scopeOrdered($query)
    {
        return $query->orderBy('order', 'asc')->orderBy('title', 'asc');
    }

    /**
     * Get content blocks parsed from JSON.
     *
     * @return array
     */
    public function getBlocks(): array
    {
        if (empty($this->content)) {
            return [];
        }

        // If content is already an array, return it
        if (is_array($this->content)) {
            return $this->content;
        }

        // If content is a JSON string, decode it
        if (is_string($this->content)) {
            $decoded = json_decode($this->content, true);
            return $decoded ?? [];
        }

        return [];
    }

    /**
     * Check if the page is published.
     *
     * @return bool
     */
    public function isPublished(): bool
    {
        return $this->is_published 
               && $this->published_at !== null 
               && $this->published_at->lte(now());
    }

    /**
     * Publish the page.
     *
     * @return bool
     */
    public function publish(): bool
    {
        $this->is_published = true;
        if ($this->published_at === null) {
            $this->published_at = now();
        }
        return $this->save();
    }

    /**
     * Unpublish the page.
     *
     * @return bool
     */
    public function unpublish(): bool
    {
        $this->is_published = false;
        return $this->save();
    }

    /**
     * Get the page URL.
     *
     * @return string
     */
    public function getUrlAttribute(): string
    {
        return route('page.show', $this->slug);
    }

    /**
     * Get a summary of the page content.
     *
     * @param int $length
     * @return string
     */
    public function getSummary(int $length = 200): string
    {
        $blocks = $this->getBlocks();
        $text = '';

        // Extract text from blocks
        foreach ($blocks as $block) {
            if (!isset($block['type']) || !isset($block['content'])) {
                continue;
            }

            $content = $block['content'];
            
            // Handle different block types
            switch ($block['type']) {
                case 'text':
                    // Text blocks have HTML string content
                    if (is_string($content)) {
                        $text .= strip_tags($content) . ' ';
                    } elseif (is_array($content) && isset($content['html'])) {
                        $text .= strip_tags($content['html']) . ' ';
                    }
                    break;
                    
                case 'hero':
                case 'cta':
                    // Hero/CTA blocks have heading and subheading
                    if (is_array($content)) {
                        if (isset($content['heading'])) {
                            $text .= $content['heading'] . ' ';
                        }
                        if (isset($content['subheading'])) {
                            $text .= $content['subheading'] . ' ';
                        }
                    }
                    break;
                    
                case 'heading':
                    // Heading blocks have text
                    if (is_array($content) && isset($content['text'])) {
                        $text .= $content['text'] . ' ';
                    }
                    break;
                    
                case 'features':
                    // Features have heading
                    if (is_array($content) && isset($content['heading'])) {
                        $text .= $content['heading'] . ' ';
                    }
                    break;
            }
        }

        // If no text found in blocks, try meta description
        if (empty(trim($text)) && !empty($this->meta_description)) {
            $text = $this->meta_description;
        }

        // If still no text, use title
        if (empty(trim($text))) {
            $text = $this->title;
        }

        return Str::limit(trim($text), $length);
    }
}
