<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Course extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'description',
        'short_description',
        'image_url',
        'thumbnail_url',
        'chunk_words',
        'completion_mode',
        'certificate_template',
        'completion_instructions',
        'has_prerequisite',
        'prerequisite_course_id',
        'is_active',
        'is_free',
        'price',
        'discount',
        'discount_type',
        'currency',
        'created_by',
        'language_id',
        'category',
        // Study time limit fields
        'enable_study_time_limit',
        'min_study_time_per_day',
        'max_study_time_per_day',
        // Survey fields
        'enable_course_survey',
        'survey_mode',
        'survey_title',
        'survey_description',
    ];

    protected $casts = [
        'is_active' => 'boolean',
        'is_free' => 'boolean',
        'has_prerequisite' => 'boolean',
        'chunk_words' => 'integer',
        'price' => 'decimal:2',
        'discount' => 'decimal:2',
        'discount_type' => 'string',
        'enable_study_time_limit' => 'boolean',
        'min_study_time_per_day' => 'integer',
        'max_study_time_per_day' => 'integer',
        'enable_course_survey' => 'boolean',
    ];

    protected $appends = ['discounted_price'];

    public function lessons()
    {
        return $this->hasMany(Lesson::class, 'course_id')->orderBy('order_index');
    }

    // Alias for backwards compatibility
    public function sections()
    {
        return $this->lessons();
    }

    public function creator()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function prerequisiteCourse()
    {
        return $this->belongsTo(Course::class, 'prerequisite_course_id');
    }

    // Alias for backwards compatibility
    public function prerequisite()
    {
        return $this->prerequisiteCourse();
    }

    public function requiredBy()
    {
        return $this->belongsToMany(Course::class, 'course_prerequisites', 'required_course_id', 'course_id');
    }

    public function surveys()
    {
        return $this->hasMany(CourseSurvey::class);
    }

    public function activeSurvey()
    {
        return $this->hasOne(CourseSurvey::class)->where('is_active', true)->latest();
    }

    public function studentStudyTimes()
    {
        return $this->hasMany(StudentDailyStudyTime::class);
    }

    public function enrollments()
    {
        return $this->hasMany(CourseEnrollment::class);
    }

    public function students()
    {
        return $this->belongsToMany(User::class, 'course_enrollments', 'course_id', 'user_id')
            ->withPivot('status', 'enrolled_at', 'completed_at', 'progress')
            ->withTimestamps();
    }

    public function setNameAttribute($value)
    {
        $this->attributes['name'] = $value;
        $this->attributes['slug'] = Str::slug($value);
    }

    public function getLessonCountAttribute()
    {
        return $this->lessons()->count();
    }

    public function getDiscountedPriceAttribute(): float
    {
        return $this->calculateDiscountedPrice($this->price, $this->discount, $this->discount_type);
    }

    private function calculateDiscountedPrice($price, $discount, $discountType): float
    {
        $priceValue = (float) ($price ?? 0);
        $discountValue = (float) ($discount ?? 0);
        if ($priceValue <= 0 || $discountValue <= 0) {
            return round($priceValue, 2);
        }

        $type = $discountType ?: 'amount';
        if ($type === 'percent') {
            $priceValue -= ($priceValue * ($discountValue / 100));
        } else {
            $priceValue -= $discountValue;
        }

        return round(max($priceValue, 0), 2);
    }

    /**
     * Get available course categories
     */
    public static function getCategories(): array
    {
        return [
            'Mindset' => 'Mindset',
            'Business' => 'Business',
            'Personal Development' => 'Personal Development',
            'Spiritual Training' => 'Spiritual Training',
        ];
    }

    public function language()
    {
        return $this->belongsTo(Language::class);
    }

    public function scopeByLanguage($query, $languageId)
    {
        if ($languageId) {
            return $query->where('language_id', $languageId);
        }
        return $query;
    }
}
