<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Payment;
use App\Models\CourseEnrollment;
use Illuminate\Support\Facades\Log;
use Stripe\Stripe;
use Stripe\Webhook;

class WebhookController extends Controller
{
    /**
     * Handle Stripe webhook
     */
    public function handleStripe(Request $request)
    {
        Stripe::setApiKey(env('STRIPE_SECRET'));
        $endpoint_secret = env('STRIPE_WEBHOOK_SECRET');

        $payload = $request->getContent();
        $sig_header = $request->header('Stripe-Signature');

        try {
            $event = Webhook::constructEvent($payload, $sig_header, $endpoint_secret);
        } catch(\UnexpectedValueException $e) {
            Log::error('Stripe webhook invalid payload: ' . $e->getMessage());
            return response()->json(['error' => 'Invalid payload'], 400);
        } catch(\Stripe\Exception\SignatureVerificationException $e) {
            Log::error('Stripe webhook signature verification failed: ' . $e->getMessage());
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        // Handle the event
        switch ($event->type) {
            case 'checkout.session.completed':
                $session = $event->data->object;
                $this->handleStripeCheckoutCompleted($session);
                break;
            
            case 'payment_intent.succeeded':
                $paymentIntent = $event->data->object;
                Log::info('Stripe payment succeeded: ' . $paymentIntent->id);
                break;

            case 'payment_intent.payment_failed':
                $paymentIntent = $event->data->object;
                $this->handleStripePaymentFailed($paymentIntent);
                break;

            default:
                Log::info('Received unknown Stripe event type: ' . $event->type);
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Handle completed Stripe checkout
     */
    protected function handleStripeCheckoutCompleted($session)
    {
        $payment = Payment::where('transaction_id', $session->id)->first();

        if ($payment && $payment->isPending()) {
            $payment->markAsCompleted();

            // Create course enrollment
            CourseEnrollment::firstOrCreate([
                'user_id' => $payment->user_id,
                'course_id' => $payment->course_id,
            ]);

            Log::info('Stripe checkout completed for payment: ' . $payment->id);
        }
    }

    /**
     * Handle failed Stripe payment
     */
    protected function handleStripePaymentFailed($paymentIntent)
    {
        $payment = Payment::where('gateway_reference', $paymentIntent->id)->first();

        if ($payment && $payment->isPending()) {
            $payment->markAsFailed();
            Log::info('Stripe payment failed for payment: ' . $payment->id);
        }
    }

    /**
     * Handle PayPal webhook
     */
    public function handlePayPal(Request $request)
    {
        // PayPal IPN verification
        $raw_post_data = file_get_contents('php://input');
        $raw_post_array = explode('&', $raw_post_data);
        $myPost = [];
        
        foreach ($raw_post_array as $keyval) {
            $keyval = explode('=', $keyval);
            if (count($keyval) == 2) {
                $myPost[$keyval[0]] = urldecode($keyval[1]);
            }
        }

        $req = 'cmd=_notify-validate';
        foreach ($myPost as $key => $value) {
            $value = urlencode($value);
            $req .= "&$key=$value";
        }

        $paypalUrl = env('PAYPAL_MODE') === 'live' 
            ? 'https://ipnpb.paypal.com/cgi-bin/webscr'
            : 'https://ipnpb.sandbox.paypal.com/cgi-bin/webscr';

        $ch = curl_init($paypalUrl);
        curl_setopt($ch, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_1);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $req);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 1);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 2);
        curl_setopt($ch, CURLOPT_FORBID_REUSE, 1);
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Connection: Close']);

        $res = curl_exec($ch);
        curl_close($ch);

        if (strcmp($res, "VERIFIED") == 0) {
            $payment_status = $request->input('payment_status');
            $txn_id = $request->input('txn_id');

            if ($payment_status == 'Completed') {
                $payment = Payment::where('transaction_id', $txn_id)->first();
                
                if ($payment && $payment->isPending()) {
                    $payment->markAsCompleted();
                    
                    CourseEnrollment::firstOrCreate([
                        'user_id' => $payment->user_id,
                        'course_id' => $payment->course_id,
                    ]);

                    Log::info('PayPal IPN verified and payment completed: ' . $payment->id);
                }
            }
        } else {
            Log::error('PayPal IPN verification failed');
        }

        return response()->json(['status' => 'success']);
    }

    /**
     * Handle Paystack webhook
     */
    public function handlePaystack(Request $request)
    {
        // Verify webhook signature
        $signature = $request->header('x-paystack-signature');
        $body = $request->getContent();
        
        if (!$signature || ($signature !== hash_hmac('sha512', $body, env('PAYSTACK_SECRET_KEY')))) {
            Log::error('Paystack webhook signature verification failed');
            return response()->json(['error' => 'Invalid signature'], 400);
        }

        $event = $request->all();

        if ($event['event'] === 'charge.success') {
            $reference = $event['data']['reference'];
            
            $payment = Payment::where('gateway_reference', $reference)->first();
            
            if ($payment && $payment->isPending()) {
                $payment->update([
                    'status' => 'completed',
                    'transaction_id' => $event['data']['id'],
                ]);

                CourseEnrollment::firstOrCreate([
                    'user_id' => $payment->user_id,
                    'course_id' => $payment->course_id,
                ]);

                Log::info('Paystack webhook processed for payment: ' . $payment->id);
            }
        }

        return response()->json(['status' => 'success']);
    }
}
