<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Models\Course;
use App\Models\UserProgress;
use App\Models\UserLessonProgress;
use App\Models\IssuedCertificate;
use App\Models\User;
use App\Models\UserPenaltyCard;
use Carbon\Carbon;

class UserDashboardController extends Controller
{
    public function index()
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        // Redirect based on role
        if ($user->hasRole('admin')) {
            return redirect()->route('admin.panel');
        }

        if ($user->hasRole('tutor')) {
            return redirect()->route('tutor.dashboard');
        }

        // Get enrolled courses from the enrollment table
        $enrolledCourses = Course::whereHas('enrollments', function($query) use ($user) {
            $query->where('user_id', $user->id)
                  ->whereIn('status', ['enrolled', 'in-progress', 'completed']);
        })
        ->with(['lessons' => function($query) {
            $query->orderBy('order_index')->with(['topics' => function($topicQuery) {
                $topicQuery->orderBy('order_index');
            }]);
        }, 'enrollments' => function($query) use ($user) {
            $query->where('user_id', $user->id);
        }])
        ->get();

        // Calculate progress for each course
        $progress = [];
        $completedCourses = 0;
        
        foreach ($enrolledCourses as $course) {
            $totalTopics = $course->lessons->sum(function($lesson) {
                return $lesson->topics->count();
            });

            $completedTopics = 0;
            foreach ($course->lessons as $lesson) {
                foreach ($lesson->topics as $topic) {
                    $topicProgress = $topic->userProgress()->where('user_id', $user->id)->first();
                    if ($topicProgress && $topicProgress->completed) {
                        $completedTopics++;
                    }
                }
            }

            $courseProgress = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
            $progress[$course->id] = $courseProgress;
            
            // Update enrollment progress
            if ($course->enrollments->first()) {
                $course->enrollments->first()->updateProgress($courseProgress);
            }
            
            if ($courseProgress >= 100) {
                $completedCourses++;
            }
        }

        // Get recent activity
        $recentActivity = UserProgress::where('user_id', $user->id)
            ->with(['topic.lesson.course'])
            ->orderBy('updated_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function($activity) {
                $topic = $activity->topic;
                $lesson = $topic ? $topic->lesson : null;
                $course = $lesson ? $lesson->course : null;
                
                return [
                    'title' => $topic 
                        ? ($activity->completed ? "Completed: {$topic->title}" : "Started: {$topic->title}")
                        : "Activity",
                    'icon' => $activity->completed ? 'check-circle' : 'play-circle',
                    'time' => $activity->updated_at->diffForHumans(),
                    'course' => $course ? $course->name : 'Unknown'
                ];
            });

        // Calculate streak days
        $streak = $this->calculateStreak($user);

        // Prepare in-progress courses
        $inProgressCourses = [];

        // Get all active courses for Browse section
        $allCourses = Course::where('is_active', true)
            ->with(['lessons.topics'])
            ->get()
            ->map(function($course) use ($user) {
                // Check enrollment from course_enrollments table
                $enrollment = \App\Models\CourseEnrollment::where('user_id', $user->id)
                    ->where('course_id', $course->id)
                    ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
                    ->first();
                
                $course->is_enrolled = $enrollment !== null;
                
                if ($enrollment) {
                    $course->progress_percentage = $enrollment->progress;
                } else {
                    $course->progress_percentage = 0;
                }
                
                return $course;
            });

        // Get schedule data (next topics)
        $nextTopics = [];
        foreach($enrolledCourses as $course) {
            foreach($course->lessons as $lesson) {
                foreach($lesson->topics as $topic) {
                    $topicProgress = UserProgress::where('user_id', $user->id)
                        ->where('topic_id', $topic->id)
                        ->first();
                    
                    if (!$topicProgress || !$topicProgress->completed) {
                        $nextTopics[] = [
                            'topic' => $topic,
                            'lesson' => $lesson,
                            'course' => $course,
                            'status' => $topicProgress ? 'in-progress' : 'not-started'
                        ];
                        break 2; // Only get first incomplete topic per course
                    }
                }
            }
        }

        // Get certificates
        $certificates = IssuedCertificate::where('user_id', $user->id)
            ->with('course')
            ->orderBy('issued_at', 'desc')
            ->get();

        // Get leaderboard data
        $leaderboardUsers = User::role('user')
            ->get()
            ->map(function($u) {
                $completedTopics = UserProgress::where('user_id', $u->id)
                    ->where('completed', true)
                    ->count();
                $avgScore = UserProgress::where('user_id', $u->id)
                    ->whereNotNull('quiz_score')
                    ->avg('quiz_score') ?? 0;
                
                $u->total_points = ($completedTopics * 10) + round($avgScore);
                $u->completed_topics = $completedTopics;
                $u->avg_score = round($avgScore);
                
                return $u;
            })
            ->sortByDesc('total_points')
            ->values();

        // Add rank to each user
        $leaderboardUsers->each(function($u, $index) {
            $u->rank = $index + 1;
        });

        // Find current user rank
        $currentUserRank = $leaderboardUsers->firstWhere('id', $user->id);

        // Get user's active penalty cards
        $penaltyCards = UserPenaltyCard::where('user_id', $user->id)
            ->where('is_active', true)
            ->with(['penaltyRule', 'issuer'])
            ->orderBy('issued_at', 'desc')
            ->get();

        return view('dashboard.student-new', [
            'enrolledCourses' => $enrolledCourses,
            'progress' => $progress,
            'completedCourses' => $completedCourses,
            'inProgressCourses' => $inProgressCourses,
            'recentActivity' => UserProgress::where('user_id', $user->id)
                ->with(['topic.lesson.course'])
                ->orderBy('updated_at', 'desc')
                ->limit(5)
                ->get(),
            'streak' => $streak,
            'allCourses' => $allCourses,
            'nextTopics' => $nextTopics,
            'certificates' => $certificates,
            'leaderboardUsers' => $leaderboardUsers->take(10),
            'currentUserRank' => $currentUserRank,
            'penaltyCards' => $penaltyCards
        ]);
    }

    private function calculateStreak($user)
    {
        // Get unique activity dates
        $activityDates = UserProgress::where('user_id', $user->id)
            ->selectRaw('DATE(updated_at) as activity_date')
            ->distinct()
            ->orderBy('activity_date', 'desc')
            ->pluck('activity_date')
            ->toArray();

        if (empty($activityDates)) {
            return 0;
        }

        $streak = 1;
        $currentDate = Carbon::parse($activityDates[0]);

        for ($i = 1; $i < count($activityDates); $i++) {
            $previousDate = Carbon::parse($activityDates[$i]);
            
            if ($currentDate->diffInDays($previousDate) === 1) {
                $streak++;
                $currentDate = $previousDate;
            } else {
                break;
            }
        }

        return $streak;
    }
}