<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use App\Models\Course;
use App\Models\Lesson;
use App\Models\Topic;
use App\Models\User;
use App\Models\UserProgress;
use App\Models\UserLessonProgress;
use App\Models\QuizQuestion;
use App\Models\IssuedCertificate;
use App\Models\CertificateTemplate;

class TutorDashboardController extends Controller
{
    public function index()
    {
        $tutor = Auth::user();

        // Get only tutor's courses
        $tutorCourses = Course::with(['lessons.topics'])
            ->where('created_by', Auth::id())
            ->orderBy('created_at', 'desc')
            ->get();

        // Get lessons from tutor's courses
        $tutorLessonIds = Lesson::whereIn('course_id', $tutorCourses->pluck('id'))->pluck('id');
        
        // Get topics from tutor's lessons
        $tutorTopicIds = Topic::whereIn('lesson_id', $tutorLessonIds)->pluck('id');

        // Get total unique students who have progress in tutor's topics
        $totalStudents = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->distinct('user_id')
            ->count('user_id');

        // Get recent enrollments (last 7 days) - students who started learning tutor's courses
        $recentEnrollments = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->where('created_at', '>=', now()->subDays(7))
            ->distinct('user_id')
            ->count('user_id');

        // Get total quizzes in tutor's lessons
        $totalQuizzes = QuizQuestion::whereIn('lesson_id', $tutorLessonIds)->count();

        // Get completed lessons count in tutor's courses
        $completedLessons = UserLessonProgress::whereIn('lesson_id', $tutorLessonIds)
            ->where('completed', true)
            ->count();

        // Get average lesson progress in tutor's courses
        $totalLessons = UserLessonProgress::whereIn('lesson_id', $tutorLessonIds)->count();
        $averageProgress = $totalLessons > 0 ? round(($completedLessons / $totalLessons) * 100, 1) : 0;

        // Get recent student activities in tutor's courses (last 10)
        $recentActivities = UserLessonProgress::whereIn('lesson_id', $tutorLessonIds)
            ->with(['user', 'lesson.course'])
            ->orderBy('updated_at', 'desc')
            ->limit(10)
            ->get();

        // Get students needing attention in tutor's courses
        $studentsNeedingAttention = UserLessonProgress::whereIn('lesson_id', $tutorLessonIds)
            ->where('completed', false)
            ->where('updated_at', '<', now()->subDays(7))
            ->with(['user', 'lesson.course'])
            ->groupBy('user_id', 'lesson_id')
            ->limit(5)
            ->get();

        // Get certificates issued for tutor's courses (using polymorphic relationship)
        $certificatesIssued = IssuedCertificate::where('assignable_type', 'App\Models\Course')
            ->whereIn('assignable_id', $tutorCourses->pluck('id'))
            ->count();

        // Get certificate templates (all templates available to tutors)
        $certificateTemplates = CertificateTemplate::count();

        // Monthly enrollment trend (last 6 months) - based on tutor's courses
        $monthlyEnrollments = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->where('created_at', '>=', now()->subMonths(6))
            ->select(
                DB::raw('DATE_FORMAT(created_at, "%Y-%m") as month'),
                DB::raw('COUNT(DISTINCT user_id) as count')
            )
            ->groupBy('month')
            ->orderBy('month', 'asc')
            ->get();

        // Get active students in tutor's courses (accessed topics in last 7 days)
        $activeStudents = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->where('updated_at', '>=', now()->subDays(7))
            ->distinct('user_id')
            ->count('user_id');

        // Build students aggregate for initial server-render (page 1)
        $page = 1;
        $limit = 50;

        $totalUsers = UserProgress::whereIn('topic_id', $tutorTopicIds)
            ->distinct('user_id')
            ->count('user_id');

        $aggregates = DB::table('user_progress as up')
            ->join('topics as t', 't.id', '=', 'up.topic_id')
            ->join('lessons as l', 'l.id', '=', 't.lesson_id')
            ->join('courses as c', 'c.id', '=', 'l.course_id')
            ->whereIn('l.id', $tutorLessonIds)
            ->groupBy('up.user_id')
            ->select([
                'up.user_id',
                DB::raw('COUNT(*) as total_topics'),
                DB::raw('SUM(CASE WHEN up.completed = 1 THEN 1 ELSE 0 END) as completed_topics'),
                DB::raw('AVG(NULLIF(up.quiz_score, NULL)) as avg_quiz_score'),
                DB::raw('MAX(up.updated_at) as last_activity'),
                DB::raw('COUNT(DISTINCT c.id) as enrolled_courses')
            ])
            ->orderBy('up.user_id')
            ->skip(($page - 1) * $limit)
            ->take($limit)
            ->get()
            ->keyBy('user_id');

        $userIds = $aggregates->keys();
        $users = User::role('user')->whereIn('id', $userIds)->get()->keyBy('id');

        $students = $aggregates->map(function ($row) use ($users) {
            $user = $users->get($row->user_id);
            $totalTopics = (int) ($row->total_topics ?? 0);
            $completedTopics = (int) ($row->completed_topics ?? 0);
            $avgScore = (float) ($row->avg_quiz_score ?? 0);
            $progressPercent = $totalTopics > 0 ? ($completedTopics / $totalTopics) * 100 : 0;
            return (object) [
                'id' => $row->user_id,
                'name' => $user?->name,
                'email' => $user?->email,
                'total_progress' => round($progressPercent, 1),
                'average_quiz_score' => round($avgScore, 1),
                'last_activity' => $row->last_activity,
                'enrolled_courses' => (int) ($row->enrolled_courses ?? 0)
            ];
        })->values();

        $studentsMeta = [
            'total' => $totalUsers,
            'page' => $page,
            'limit' => $limit,
            'has_more' => $limit < $totalUsers,
        ];

        return view('dashboard.tutor', [
            'user' => $tutor,
            'tutorCourses' => $tutorCourses,
            'totalCourses' => $tutorCourses->count(),
            'totalStudents' => $totalStudents,
            'activeStudents' => $activeStudents,
            'recentEnrollments' => $recentEnrollments,
            'totalQuizzes' => $totalQuizzes,
            'completedCourses' => $completedLessons,
            'averageProgress' => $averageProgress,
            'recentActivities' => $recentActivities,
            'studentsNeedingAttention' => $studentsNeedingAttention,
            'certificatesIssued' => $certificatesIssued,
            'certificateTemplates' => $certificateTemplates,
            'monthlyEnrollments' => $monthlyEnrollments,
            'students' => $students,
            'studentsMeta' => (object) $studentsMeta
        ]);
    }
}