<?php

namespace App\Http\Controllers;

use App\Models\Topic;
use App\Models\Lesson;
use App\Models\SiteSetting;
use App\Services\TopicSplitterService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class TopicController extends Controller
{
    protected $topicSplitter;

    public function __construct(TopicSplitterService $topicSplitter)
    {
        $this->topicSplitter = $topicSplitter;
    }
    public function index(Request $request)
    {
        $lessonId = $request->query('lesson_id');

        if (!$lessonId) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid lesson_id'
            ], 400);
        }

        try {
            // Note: Removed is_published filter to show all topics in admin view
            // Frontend should filter published topics for student view
            $topics = Topic::where('lesson_id', $lessonId)
                ->orderBy('order_index')
                ->orderBy('id')
                ->get();

            return response()->json([
                'success' => true,
                'topics' => $topics
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading topics: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $topic = Topic::find($id);

            if (!$topic) {
                return response()->json([
                    'success' => false,
                    'message' => 'Topic not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'topic' => $topic
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lesson_id' => 'required|exists:lessons,id',
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string|max:1024',
            'image_url' => 'nullable|string|max:1024',
            'order_index' => 'integer',
            'is_published' => 'boolean',
            'auto_split' => 'boolean', // New: enable/disable auto-splitting
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            // Check if auto-split is enabled (default: true)
            $autoSplit = $request->input('auto_split', true);
            
            // Get lesson and course chunk_words
            $lesson = Lesson::with('course')->find($request->lesson_id);
            if (!$lesson || !$lesson->course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Lesson or course not found'
                ], 404);
            }

            $chunkWords = $lesson->course->chunk_words ?? SiteSetting::getChunkWordsDefault();
            $body = $request->input('body', '');

            // Check if splitting should occur
            if ($autoSplit && $this->topicSplitter->shouldSplit($body, $chunkWords)) {
                // Calculate how many topics will be created
                $topicCount = $this->topicSplitter->calculateTopicCount($body, $chunkWords);
                $wordCount = $this->topicSplitter->getWordCount($body);

                // Split and create multiple topics
                $topicIds = $this->topicSplitter->splitAndCreateTopics(
                    $request->lesson_id,
                    $request->title,
                    $body,
                    $chunkWords,
                    [
                        'video_url' => $request->video_url,
                        'image_url' => $request->image_url,
                        'is_published' => $request->input('is_published', true),
                        'order_index' => $request->input('order_index', 0),
                    ]
                );

                return response()->json([
                    'success' => true,
                    'auto_split' => true,
                    'topics_created' => count($topicIds),
                    'topic_ids' => $topicIds,
                    'word_count' => $wordCount,
                    'chunk_words' => $chunkWords,
                    'message' => "Content was automatically split into " . count($topicIds) . " topics based on {$chunkWords} words per topic."
                ]);
            }

            // Create single topic (no splitting needed or disabled)
            $topic = Topic::create($request->all());

            return response()->json([
                'success' => true,
                'auto_split' => false,
                'id' => $topic->id,
                'topics_created' => 1
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating topic: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $topic = Topic::find($id);

        if (!$topic) {
            return response()->json([
                'success' => false,
                'message' => 'Topic not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'lesson_id' => 'required|exists:lessons,id',
            'title' => 'required|string|max:255',
            'body' => 'nullable|string',
            'video_url' => 'nullable|string|max:1024',
            'image_url' => 'nullable|string|max:1024',
            'order_index' => 'integer',
            'is_published' => 'boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $topic->update($request->all());

            return response()->json([
                'success' => true,
                'id' => $topic->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating topic: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $topic = Topic::find($id);

        if (!$topic) {
            return response()->json([
                'success' => false,
                'message' => 'Topic not found'
            ], 404);
        }

        try {
            $topic->delete();

            return response()->json([
                'success' => true,
                'message' => 'Topic deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting topic: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Split an existing topic into multiple topics
     */
    public function split(Request $request, $id)
    {
        $topic = Topic::with('lesson.course')->find($id);

        if (!$topic) {
            return response()->json([
                'success' => false,
                'message' => 'Topic not found'
            ], 404);
        }

        try {
            $chunkWords = $request->input('chunk_words') 
                ?? $topic->lesson->course->chunk_words 
                ?? SiteSetting::getChunkWordsDefault();

            // Check if splitting is possible
            if (!$this->topicSplitter->shouldSplit($topic->body, $chunkWords)) {
                $wordCount = $this->topicSplitter->getWordCount($topic->body);
                return response()->json([
                    'success' => false,
                    'message' => "Topic content is too short to split. Word count: {$wordCount}, Minimum required: " . ceil($chunkWords * 1.5)
                ], 400);
            }

            // Perform the split
            $topicIds = $this->topicSplitter->splitExistingTopic($topic, $chunkWords);

            return response()->json([
                'success' => true,
                'message' => 'Topic split successfully',
                'topics_created' => count($topicIds),
                'topic_ids' => $topicIds
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error splitting topic: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Preview topic split without actually creating topics
     */
    public function previewSplit(Request $request, $id)
    {
        $topic = Topic::with('lesson.course')->find($id);

        if (!$topic) {
            return response()->json([
                'success' => false,
                'message' => 'Topic not found'
            ], 404);
        }

        try {
            $chunkWords = $request->input('chunk_words') 
                ?? $topic->lesson->course->chunk_words 
                ?? SiteSetting::getChunkWordsDefault();

            $wordCount = $this->topicSplitter->getWordCount($topic->body);
            $canSplit = $this->topicSplitter->shouldSplit($topic->body, $chunkWords);
            $topicCount = $this->topicSplitter->calculateTopicCount($topic->body, $chunkWords);

            return response()->json([
                'success' => true,
                'can_split' => $canSplit,
                'word_count' => $wordCount,
                'chunk_words' => $chunkWords,
                'topics_would_create' => $topicCount,
                'message' => $canSplit 
                    ? "This topic would be split into {$topicCount} topics" 
                    : "Topic content is too short to split"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error previewing split: ' . $e->getMessage()
            ], 500);
        }
    }
}