<?php

namespace App\Http\Controllers;

use App\Models\CertificateTemplate;
use App\Models\User;
use App\Models\Course;
use App\Services\CertificateGeneratorService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class TestCertificateController extends Controller
{
    protected $certificateGenerator;

    public function __construct(CertificateGeneratorService $certificateGenerator)
    {
        $this->certificateGenerator = $certificateGenerator;
    }

    /**
     * Generate a test certificate
     */
    public function generateTestCertificate()
    {
        try {
            // Get first active certificate template
            $template = CertificateTemplate::where('is_active', true)->first();
            
            if (!$template) {
                return response()->json([
                    'error' => 'No active certificate template found. Please create one first.'
                ], 404);
            }

            // Get first user (or current auth user)
            $user = Auth::user() ?? User::first();
            
            if (!$user) {
                return response()->json([
                    'error' => 'No users found in the system.'
                ], 404);
            }

            // Get first course or create a mock assignable
            $course = Course::first();
            
            if (!$course) {
                // Create a mock course object for testing
                $course = new \stdClass();
                $course->id = 1;
                $course->title = 'Sample Course - Test Certificate';
            }

            // Generate the certificate
            $issuedCertificate = $this->certificateGenerator->generateCertificate($user, $template, $course);

            return response()->json([
                'success' => true,
                'message' => 'Test certificate generated successfully!',
                'certificate' => [
                    'id' => $issuedCertificate->id,
                    'code' => $issuedCertificate->certificate_code,
                    'student_name' => $user->name,
                    'course_name' => is_object($course) ? ($course->title ?? 'Test Course') : $course,
                    'issued_at' => $issuedCertificate->issued_at->format('F j, Y'),
                    'verification_url' => $issuedCertificate->verification_url,
                    'qr_code_url' => $issuedCertificate->qr_code_path 
                        ? asset('storage/' . $issuedCertificate->qr_code_path) 
                        : null,
                    'certificate_url' => $issuedCertificate->pdf_path 
                        ? asset('storage/' . $issuedCertificate->pdf_path) 
                        : null,
                ],
                'links' => [
                    'view_certificate' => route('test.view-certificate', $issuedCertificate->id),
                    'verify_certificate' => route('certificate.verify', $issuedCertificate->certificate_code),
                    'qr_code_image' => $issuedCertificate->qr_code_path 
                        ? asset('storage/' . $issuedCertificate->qr_code_path) 
                        : null,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Failed to generate certificate: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * View a test certificate
     */
    public function viewCertificate($id)
    {
        $certificate = \App\Models\IssuedCertificate::with(['user', 'template'])->findOrFail($id);

        if ($certificate->pdf_path && Storage::disk('public')->exists($certificate->pdf_path)) {
            $filePath = storage_path('app/public/' . $certificate->pdf_path);
            return response()->file($filePath);
        }

        return response()->json([
            'error' => 'Certificate file not found'
        ], 404);
    }
}
