<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\CourseSurvey;
use App\Models\CourseSurveyResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SurveyController extends Controller
{
    /**
     * Get survey questions for a course
     */
    public function getCourseSurvey($courseId)
    {
        $course = Course::findOrFail($courseId);
        
        // Use correct field for enable flag
        if (!$course->enable_course_survey) {
            return response()->json(['message' => 'Survey not enabled'], 404);
        }
        
        // Get survey from course_surveys table which stores questions in JSON
        $survey = DB::table('course_surveys')->where('course_id', $courseId)->first();
        
        $questions = [];
        if ($survey && $survey->questions) {
            $questions = json_decode($survey->questions, true) ?? [];
        }
        
        // If feedback mode and no custom questions, provide sensible defaults
        if (empty($questions) && ($course->survey_mode ?? 'feedback') === 'feedback') {
            $questions = [
                [
                    'id' => 'default_1',
                    'question' => 'How would you rate this course overall?',
                    'question_type' => 'rating',
                    'options' => null,
                    'is_required' => true,
                    'order_index' => 0,
                ],
                [
                    'id' => 'default_2',
                    'question' => 'Any comments or suggestions?',
                    'question_type' => 'text',
                    'options' => null,
                    'is_required' => false,
                    'order_index' => 1,
                ]
            ];
        }

        return response()->json([
            'title' => $course->survey_title ?? ($course->survey_mode === 'survey' ? 'Course Survey' : 'Course Feedback'),
            'description' => $course->survey_description ?? null,
            'mode' => $course->survey_mode ?? 'feedback',
            'questions' => $questions
        ]);
    }

    /**
     * Submit survey response
     */
    public function submitSurvey(Request $request, $courseId)
    {
        $request->validate([
            'responses' => 'required|array'
        ]);
        
        $userId = Auth::id();
        
        // Check if already submitted
        $existing = CourseSurveyResponse::where('user_id', $userId)
            ->where('course_id', $courseId)
            ->first();
        
        if ($existing) {
            return response()->json(['message' => 'Already submitted'], 400);
        }
        
        // Process each response
        foreach ($request->responses as $questionId => $responseData) {
            $rating = $responseData['type'] === 'rating' ? (int)$responseData['value'] : null;
            
            CourseSurveyResponse::create([
                'survey_id' => $questionId,
                'user_id' => $userId,
                'course_id' => $courseId,
                'response' => $responseData,
                'rating' => $rating,
                'completed_at' => now()
            ]);
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Survey submitted successfully'
        ]);
    }

    /**
     * Get survey statistics (admin/tutor)
     */
    public function getStats($courseId)
    {
        // Authorization: only admins or owning tutor can view stats
        $course = Course::findOrFail($courseId);
        $user = Auth::user();
        $isAdmin = $user && $user->roles && $user->roles->contains('name', 'admin');
        if (!$user || (!$isAdmin && $course->created_by !== $user->id)) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Get survey questions from course_surveys table
        $survey = DB::table('course_surveys')->where('course_id', $courseId)->first();
        $questions = [];
        if ($survey && $survey->questions) {
            $questions = json_decode($survey->questions, true) ?? [];
        }
        
        // Get responses from CourseSurveyResponse
        $responses = CourseSurveyResponse::where('course_id', $courseId)->get();
        
        $totalResponses = $responses->groupBy('user_id')->count();
        
        $questionStats = [];
        foreach ($questions as $question) {
            $questionId = $question['id'] ?? null;
            $questionResponses = $responses->where('survey_id', $questionId);
            
            $stats = [
                'question' => $question['question'] ?? '',
                'question_type' => $question['question_type'] ?? 'text',
                'response_count' => $questionResponses->count(),
                'average_value' => null,
                'option_counts' => [],
                'text_responses' => []
            ];
            
            if ($question['question_type'] === 'rating') {
                $ratings = $questionResponses->pluck('rating')->filter();
                $stats['average_value'] = $ratings->count() > 0 ? $ratings->avg() : 0;
            } elseif (in_array($question['question_type'], ['multiple_choice', 'dropdown', 'checkboxes'])) {
                foreach ($questionResponses as $resp) {
                    $value = $resp->response['value'] ?? null;
                    if ($value) {
                        $stats['option_counts'][$value] = ($stats['option_counts'][$value] ?? 0) + 1;
                    }
                }
            } elseif (in_array($question['question_type'], ['text', 'short_text', 'long_text'])) {
                $stats['text_responses'] = $questionResponses->pluck('response')->filter(function($r) {
                    return $r && isset($r['value']);
                })->map(function($r) { return $r['value'] ?? ''; })->toArray();
            }
            
            $questionStats[] = $stats;
        }
        
        return response()->json([
            'total_responses' => $totalResponses,
            'question_stats' => $questionStats,
            'responses' => $responses->isEmpty() ? [] : $responses->toArray()
        ]);
    }

    /**
     * Create/update survey questions (admin/tutor)
     */
    public function manageQuestions(Request $request, $courseId)
    {
        $request->validate([
            'questions' => 'required|array',
            'questions.*.question' => 'required|string',
            'questions.*.question_type' => 'required|in:rating,text,multiple_choice,dropdown,checkboxes,short_text,long_text',
            'questions.*.is_required' => 'boolean',
            'questions.*.options' => 'nullable|array'
        ]);
        
        $course = Course::findOrFail($courseId);
        $user = Auth::user();
        $isAdmin = $user && $user->roles && $user->roles->contains('name', 'admin');
        if (!$user || (!$isAdmin && $course->created_by !== $user->id)) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }
        
        // Update or create survey record with questions in JSON
        $questionsData = [];
        foreach ($request->questions as $index => $questionData) {
            $questionsData[] = [
                'id' => $questionData['id'] ?? 'q_' . uniqid(),
                'question' => $questionData['question'],
                'question_type' => $questionData['question_type'],
                'options' => $questionData['options'] ?? null,
                'is_required' => $questionData['is_required'] ?? false,
                'order_index' => $index
            ];
        }
        
        // Store in course_surveys table as JSON
        DB::table('course_surveys')
            ->updateOrInsert(
                ['course_id' => $courseId],
                [
                    'title' => $course->survey_title ?? 'Survey',
                    'description' => $course->survey_description ?? null,
                    'questions' => json_encode($questionsData),
                    'is_active' => 1,
                    'updated_at' => now()
                ]
            );
        
        return response()->json([
            'success' => true,
            'message' => 'Survey questions updated successfully'
        ]);
    }
}

