<?php

namespace App\Http\Controllers;

use App\Services\StudyTimeService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class StudyTimeController extends Controller
{
    protected $studyTimeService;

    public function __construct(StudyTimeService $studyTimeService)
    {
        $this->studyTimeService = $studyTimeService;
    }

    /**
     * Get study time status for a course
     */
    public function getStatus(Request $request, $courseId)
    {
        $userId = Auth::id();
        
        return response()->json([
            'status' => $this->studyTimeService->hasReachedDailyLimit($userId, $courseId)
        ]);
    }

    /**
     * Start a study session
     */
    public function startSession(Request $request, $courseId)
    {
        $userId = Auth::id();
        
        $result = $this->studyTimeService->startSession($userId, $courseId);
        
        if (!$result['can_start']) {
            return response()->json([
                'success' => false,
                'message' => $result['reason']
            ], 403);
        }

        return response()->json([
            'success' => true,
            'session_start' => $result['session_start']->toIso8601String(),
            'remaining_minutes' => $result['remaining_minutes']
        ]);
    }

    /**
     * End a study session and record time
     */
    public function endSession(Request $request, $courseId)
    {
        $request->validate([
            'session_start' => 'required|date'
        ]);

        $userId = Auth::id();
        $sessionStart = Carbon::parse($request->session_start);
        
        $result = $this->studyTimeService->endSession($userId, $courseId, $sessionStart);
        
        return response()->json([
            'success' => true,
            'minutes_added' => $result['minutes_added'],
            'total_today' => $result['total_today'],
            'limit_status' => $result['limit_status']
        ]);
    }

    /**
     * Add study time (for auto-save/tracking)
     */
    public function addTime(Request $request, $courseId)
    {
        $request->validate([
            'minutes' => 'required|integer|min:1|max:1440'
        ]);

        $userId = Auth::id();
        $record = $this->studyTimeService->addStudyTime($userId, $courseId, $request->minutes);
        
        return response()->json([
            'success' => true,
            'total_today' => $record->total_minutes_studied,
            'limit_status' => $this->studyTimeService->hasReachedDailyLimit($userId, $courseId)
        ]);
    }

    /**
     * Get study statistics
     */
    public function getStats(Request $request, $courseId)
    {
        $userId = Auth::id();
        $days = $request->query('days', 7);
        
        $stats = $this->studyTimeService->getStudyStats($userId, $courseId, $days);
        
        return response()->json($stats);
    }

    /**
     * Check minimum requirement status
     */
    public function checkMinimum(Request $request, $courseId)
    {
        $userId = Auth::id();
        
        return response()->json(
            $this->studyTimeService->hasMetMinimumRequirement($userId, $courseId)
        );
    }
}

