<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\UserProgress;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class StudentEnrollmentController extends Controller
{
    /**
     * Enroll a student in a course
     */
    public function enroll(Request $request, Course $course)
    {
        $user = Auth::user();

        // Check if already enrolled
        $existingEnrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->first();

        if ($existingEnrollment && $existingEnrollment->isActive()) {
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You are already enrolled in this course',
                ], 422);
            }
            return back()->with('error', 'You are already enrolled in this course');
        }

        // Check if course is active
        if (!$course->is_active) {
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'This course is not currently available',
                ], 422);
            }
            return back()->with('error', 'This course is not currently available');
        }

        // Check prerequisite completion
        if ($course->has_prerequisite && $course->prerequisite_course_id) {
            $prerequisiteCompleted = $this->hasCompletedPrerequisite($user->id, $course->prerequisite_course_id);
            
            if (!$prerequisiteCompleted) {
                $prerequisiteCourse = Course::find($course->prerequisite_course_id);
                $message = "You must complete '{$prerequisiteCourse->name}' before enrolling in this course";
                
                if ($request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => $message,
                        'prerequisite' => [
                            'id' => $prerequisiteCourse->id,
                            'name' => $prerequisiteCourse->name,
                        ],
                    ], 422);
                }
                return back()->with('error', $message);
            }
        }

        // Check if course is free or requires payment
        if (!$course->is_free && $course->price > 0) {
            // Check if user has completed payment for this course
            $payment = Payment::where('user_id', $user->id)
                ->where('course_id', $course->id)
                ->where('status', 'completed')
                ->first();

            if (!$payment) {
                if ($request->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'This course requires payment. Please complete the payment to enroll.',
                        'course_price' => $course->price,
                        'currency' => $course->currency,
                        'redirect_url' => route('payment.checkout', $course->id),
                    ], 402);
                }
                return redirect()->route('payment.checkout', $course->id)
                    ->with('info', 'Please complete the payment to enroll in this course.');
            }
        }

        // Create or reactivate enrollment
        if ($existingEnrollment) {
            $existingEnrollment->update([
                'status' => 'enrolled',
                'enrolled_at' => now(),
            ]);
            $enrollment = $existingEnrollment;
        } else {
            $enrollment = CourseEnrollment::create([
                'user_id' => $user->id,
                'course_id' => $course->id,
                'status' => 'enrolled',
                'enrolled_at' => now(),
            ]);
        }

        $message = "Successfully enrolled in '{$course->name}'!";

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
                'enrollment' => $enrollment,
                'redirect' => route('learn.course', $course),
            ]);
        }

        return redirect()->route('learn.course', $course)->with('success', $message);
    }

    /**
     * Drop a course
     */
    public function drop(Request $request, Course $course)
    {
        $user = Auth::user();

        $enrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->first();

        if (!$enrollment) {
            if ($request->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'You are not enrolled in this course',
                ], 404);
            }
            return back()->with('error', 'You are not enrolled in this course');
        }

        $enrollment->update(['status' => 'dropped']);

        $message = "Successfully dropped '{$course->name}'";

        if ($request->wantsJson()) {
            return response()->json([
                'success' => true,
                'message' => $message,
            ]);
        }

        return back()->with('success', $message);
    }

    /**
     * Get enrollment status for a course
     */
    public function status(Request $request, Course $course)
    {
        $user = Auth::user();

        $enrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->first();

        $isEnrolled = $enrollment && $enrollment->isActive();
        $prerequisiteCompleted = true;

        if ($course->has_prerequisite && $course->prerequisite_course_id) {
            $prerequisiteCompleted = $this->hasCompletedPrerequisite($user->id, $course->prerequisite_course_id);
        }

        return response()->json([
            'enrolled' => $isEnrolled,
            'status' => $enrollment?->status,
            'progress' => $enrollment?->progress ?? 0,
            'can_enroll' => !$isEnrolled && $prerequisiteCompleted,
            'prerequisite_completed' => $prerequisiteCompleted,
            'is_free' => $course->is_free,
            'price' => $course->price,
            'currency' => $course->currency,
        ]);
    }

    /**
     * Get user's enrolled courses
     */
    public function myCourses()
    {
        $user = Auth::user();

        $enrollments = CourseEnrollment::where('user_id', $user->id)
            ->whereIn('status', ['enrolled', 'in-progress', 'completed'])
            ->with('course.lessons.topics')
            ->get();

        return response()->json([
            'enrollments' => $enrollments,
        ]);
    }

    /**
     * Mark a course as completed
     */
    public function markCompleted(Request $request, Course $course)
    {
        $user = Auth::user();

        $enrollment = CourseEnrollment::where('user_id', $user->id)
            ->where('course_id', $course->id)
            ->first();

        if (!$enrollment) {
            return response()->json([
                'success' => false,
                'message' => 'You are not enrolled in this course',
            ], 404);
        }

        // Check if all topics are completed
        $totalTopics = $course->lessons->sum(function($lesson) {
            return $lesson->topics->count();
        });

        $completedTopics = UserProgress::where('user_id', $user->id)
            ->whereHas('topic.lesson', function($query) use ($course) {
                $query->where('course_id', $course->id);
            })
            ->where('completed', true)
            ->count();

        if ($completedTopics < $totalTopics) {
            return response()->json([
                'success' => false,
                'message' => 'Complete all topics before marking the course as complete',
                'progress' => [
                    'completed' => $completedTopics,
                    'total' => $totalTopics,
                ],
            ], 422);
        }

        $enrollment->markAsCompleted();

        return response()->json([
            'success' => true,
            'message' => "Congratulations! You've completed '{$course->name}'",
            'enrollment' => $enrollment,
        ]);
    }

    /**
     * Check if user has completed a prerequisite course
     */
    private function hasCompletedPrerequisite(int $userId, int $prerequisiteCourseId): bool
    {
        // Check if enrolled and completed the prerequisite
        $enrollment = CourseEnrollment::where('user_id', $userId)
            ->where('course_id', $prerequisiteCourseId)
            ->where('status', 'completed')
            ->exists();

        if ($enrollment) {
            return true;
        }

        // Alternative: Check if user completed all topics in prerequisite course
        $prerequisiteCourse = Course::with('lessons.topics')->find($prerequisiteCourseId);
        
        if (!$prerequisiteCourse) {
            return false;
        }

        $totalTopics = $prerequisiteCourse->lessons->sum(function($lesson) {
            return $lesson->topics->count();
        });

        if ($totalTopics === 0) {
            return true; // If no topics, consider it completed
        }

        $completedTopics = UserProgress::where('user_id', $userId)
            ->whereHas('topic.lesson', function($query) use ($prerequisiteCourseId) {
                $query->where('course_id', $prerequisiteCourseId);
            })
            ->where('completed', true)
            ->count();

        return $completedTopics >= $totalTopics;
    }
}
