<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\IssuedCertificate;
use App\Models\Course;
use App\Models\Lesson;
use App\Services\CertificateGeneratorService;
use Illuminate\Support\Facades\Auth;

class StudentCertificateController extends Controller
{
    protected $certificateGenerator;

    public function __construct(CertificateGeneratorService $certificateGenerator)
    {
        $this->certificateGenerator = $certificateGenerator;
    }

    /**
     * Display student's certificates
     */
    public function index()
    {
        $user = Auth::user();
        
        $certificates = IssuedCertificate::where('user_id', $user->id)
            ->with(['template', 'assignable'])
            ->orderBy('issued_at', 'desc')
            ->get();

        return view('student.certificates', compact('certificates'));
    }

    /**
     * Generate certificate for a course (manual trigger if needed)
     */
    public function generate(Course $course)
    {
        $user = Auth::user();
        
        // Check if student has completed the course
        $enrollment = $user->enrollments()->where('course_id', $course->id)->first();
        
        if (!$enrollment || $enrollment->progress < 100) {
            return redirect()->back()->with('error', 'You must complete the course before generating a certificate.');
        }

        // Check if certificate assignment exists
        $assignment = \App\Models\CertificateAssignment::where([
            'assignable_type' => Course::class,
            'assignable_id' => $course->id,
        ])->first();

        if (!$assignment) {
            return redirect()->back()->with('error', 'No certificate is configured for this course.');
        }

        // Check if certificate already issued
        $existing = IssuedCertificate::where([
            'user_id' => $user->id,
            'assignable_type' => Course::class,
            'assignable_id' => $course->id,
        ])->first();

        if ($existing) {
            return redirect()->route('student.certificate.view', $existing->id);
        }

        // Generate certificate
        try {
            $certificate = $this->certificateGenerator->generateCertificate($user, $assignment->template, $course);
            
            return redirect()->route('student.certificate.view', $certificate->id)
                ->with('success', 'Certificate generated successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to generate certificate: ' . $e->getMessage());
        }
    }

    /**
     * Generate certificate for a lesson (manual trigger if needed)
     */
    public function generateLesson(Lesson $lesson)
    {
        $user = Auth::user();
        
        // Check if student has completed all topics in the lesson
        $allCompleted = true;
        foreach ($lesson->topics as $topic) {
            $progress = \App\Models\UserProgress::where([
                'user_id' => $user->id,
                'topic_id' => $topic->id,
            ])->first();
            
            if (!$progress || !$progress->completed) {
                $allCompleted = false;
                break;
            }
        }

        if (!$allCompleted) {
            return redirect()->back()->with('error', 'You must complete all topics in this lesson before generating a certificate.');
        }

        // Check if certificate assignment exists
        $assignment = \App\Models\CertificateAssignment::where([
            'assignable_type' => Lesson::class,
            'assignable_id' => $lesson->id,
        ])->first();

        if (!$assignment) {
            return redirect()->back()->with('error', 'No certificate is configured for this lesson.');
        }

        // Check if certificate already issued
        $existing = IssuedCertificate::where([
            'user_id' => $user->id,
            'assignable_type' => Lesson::class,
            'assignable_id' => $lesson->id,
        ])->first();

        if ($existing) {
            return redirect()->route('student.certificate.view', $existing->id);
        }

        // Generate certificate
        try {
            $certificate = $this->certificateGenerator->generateCertificate($user, $assignment->template, $lesson);
            
            return redirect()->route('student.certificate.view', $certificate->id)
                ->with('success', 'Certificate generated successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to generate certificate: ' . $e->getMessage());
        }
    }

    /**
     * View certificate
     */
    public function view(IssuedCertificate $certificate)
    {
        // Ensure the certificate belongs to the authenticated user
        if ($certificate->user_id !== Auth::id()) {
            abort(403, 'Unauthorized access to certificate.');
        }

        return view('student.certificate-view', compact('certificate'));
    }

    /**
     * Download certificate PDF
     */
    public function download(IssuedCertificate $certificate)
    {
        // Ensure the certificate belongs to the authenticated user
        if ($certificate->user_id !== Auth::id()) {
            abort(403, 'Unauthorized access to certificate.');
        }

        if (!$certificate->pdf_path || !\Storage::disk('public')->exists($certificate->pdf_path)) {
            return redirect()->back()->with('error', 'Certificate file not found.');
        }

        return response()->download(storage_path('app/public/' . $certificate->pdf_path));
    }
}
