<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\User;
use App\Models\UserProgress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class StudentBrowseController extends Controller
{
    /**
     * Browse all available courses
     */
    public function browse(Request $request)
    {
        $user = Auth::user();
        $search = $request->get('search', '');
        $category = $request->get('category', '');
        
        // Get all active courses
        $query = Course::where('is_active', true);
        
        if ($search) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        $courses = $query->with(['lessons.topics', 'creator'])
            ->get()
            ->map(function($course) use ($user) {
                // Calculate total topics
                $totalTopics = $course->lessons->sum(function($lesson) {
                    return $lesson->topics->count();
                });
                
                // Check if enrolled (has any progress)
                $hasProgress = UserProgress::where('user_id', $user->id)
                    ->whereIn('topic_id', $course->lessons->flatMap->topics->pluck('id'))
                    ->exists();
                
                // Calculate progress if enrolled
                $progressPercentage = 0;
                if ($hasProgress) {
                    $completedTopics = UserProgress::where('user_id', $user->id)
                        ->whereIn('topic_id', $course->lessons->flatMap->topics->pluck('id'))
                        ->where('completed', true)
                        ->count();
                    $progressPercentage = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
                }
                
                $course->total_topics = $totalTopics;
                $course->is_enrolled = $hasProgress;
                $course->progress_percentage = $progressPercentage;
                
                return $course;
            });
        
        return view('student.browse', compact('courses', 'search', 'category'));
    }
    
    /**
     * My Schedule - Show enrolled courses with upcoming topics
     */
    public function schedule()
    {
        $user = Auth::user();
        
        // Get enrolled courses with progress
        $enrolledCourses = Course::whereHas('lessons.topics.userProgress', function($query) use ($user) {
            $query->where('user_id', $user->id);
        })
        ->with(['lessons' => function($query) {
            $query->orderBy('order_index')->with(['topics' => function($topicQuery) {
                $topicQuery->where('is_published', true)->orderBy('order_index');
            }]);
        }])
        ->get();
        
        // Calculate next topics to study
        $schedule = [];
        $today = now()->format('Y-m-d');
        
        foreach ($enrolledCourses as $course) {
            $nextTopic = null;
            
            foreach ($course->lessons as $lesson) {
                foreach ($lesson->topics as $topic) {
                    $progress = UserProgress::where('user_id', $user->id)
                        ->where('topic_id', $topic->id)
                        ->first();
                    
                    // Find first incomplete topic
                    if (!$progress || !$progress->completed) {
                        $nextTopic = [
                            'course' => $course,
                            'lesson' => $lesson,
                            'topic' => $topic,
                            'status' => $progress ? 'in-progress' : 'not-started'
                        ];
                        break 2;
                    }
                }
            }
            
            if ($nextTopic) {
                $schedule[] = $nextTopic;
            }
        }
        
        // Get recent activity
        $recentActivity = UserProgress::where('user_id', $user->id)
            ->with(['topic.lesson.course'])
            ->orderBy('updated_at', 'desc')
            ->limit(10)
            ->get();
        
        return view('student.schedule', compact('schedule', 'recentActivity'));
    }
    
    /**
     * Progress & Achievements
     */
    public function progress()
    {
        $user = Auth::user();
        
        // Get all enrolled courses with detailed progress
        $courses = Course::whereHas('lessons.topics.userProgress', function($query) use ($user) {
            $query->where('user_id', $user->id);
        })
        ->with(['lessons.topics'])
        ->get()
        ->map(function($course) use ($user) {
            $totalTopics = 0;
            $completedTopics = 0;
            $totalQuizScore = 0;
            $quizCount = 0;
            
            foreach ($course->lessons as $lesson) {
                foreach ($lesson->topics as $topic) {
                    $totalTopics++;
                    $progress = UserProgress::where('user_id', $user->id)
                        ->where('topic_id', $topic->id)
                        ->first();
                    
                    if ($progress) {
                        if ($progress->completed) {
                            $completedTopics++;
                        }
                        if ($progress->quiz_score !== null) {
                            $totalQuizScore += $progress->quiz_score;
                            $quizCount++;
                        }
                    }
                }
            }
            
            $course->total_topics = $totalTopics;
            $course->completed_topics = $completedTopics;
            $course->progress_percentage = $totalTopics > 0 ? round(($completedTopics / $totalTopics) * 100) : 0;
            $course->average_quiz_score = $quizCount > 0 ? round($totalQuizScore / $quizCount) : 0;
            
            return $course;
        });
        
        // Calculate achievements
        $achievements = $this->calculateAchievements($user);
        
        // Get overall stats
        $stats = [
            'total_courses' => $courses->count(),
            'completed_courses' => $courses->where('progress_percentage', 100)->count(),
            'in_progress' => $courses->where('progress_percentage', '>', 0)->where('progress_percentage', '<', 100)->count(),
            'total_topics_completed' => $courses->sum('completed_topics'),
            'average_score' => $courses->avg('average_quiz_score'),
            'streak' => $this->calculateStreak($user),
        ];
        
        return view('student.progress', compact('courses', 'achievements', 'stats'));
    }
    
    /**
     * Leaderboard
     */
    public function leaderboard()
    {
        // Get all users with their stats
        $users = User::role('user')
            ->get()
            ->map(function($user) {
                // Get enrolled courses count
                $enrolledCourses = Course::whereHas('lessons.topics.userProgress', function($query) use ($user) {
                    $query->where('user_id', $user->id);
                })->count();
                
                // Get completed topics
                $completedTopics = UserProgress::where('user_id', $user->id)
                    ->where('completed', true)
                    ->count();
                
                // Get average quiz score
                $avgScore = UserProgress::where('user_id', $user->id)
                    ->whereNotNull('quiz_score')
                    ->avg('quiz_score');
                
                // Calculate points (completed topics * 10 + avg score)
                $points = ($completedTopics * 10) + round($avgScore ?? 0);
                
                $user->enrolled_courses = $enrolledCourses;
                $user->completed_topics = $completedTopics;
                $user->average_score = round($avgScore ?? 0);
                $user->total_points = $points;
                
                return $user;
            })
            ->sortByDesc('total_points')
            ->values();
        
        // Add rank
        $users->each(function($user, $index) {
            $user->rank = $index + 1;
        });
        
        $currentUser = Auth::user();
        $currentUserRank = $users->firstWhere('id', $currentUser->id);
        
        return view('student.leaderboard', compact('users', 'currentUserRank'));
    }
    
    /**
     * Calculate user achievements
     */
    private function calculateAchievements($user)
    {
        $completedTopics = UserProgress::where('user_id', $user->id)
            ->where('completed', true)
            ->count();
        
        $completedCourses = Course::whereHas('lessons.topics.userProgress', function($query) use ($user) {
            $query->where('user_id', $user->id);
        })
        ->get()
        ->filter(function($course) use ($user) {
            $totalTopics = 0;
            $completedTopics = 0;
            
            foreach ($course->lessons as $lesson) {
                foreach ($lesson->topics as $topic) {
                    $totalTopics++;
                    $progress = UserProgress::where('user_id', $user->id)
                        ->where('topic_id', $topic->id)
                        ->where('completed', true)
                        ->first();
                    if ($progress) {
                        $completedTopics++;
                    }
                }
            }
            
            return $totalTopics > 0 && $completedTopics == $totalTopics;
        })->count();
        
        $streak = $this->calculateStreak($user);
        
        $avgScore = UserProgress::where('user_id', $user->id)
            ->whereNotNull('quiz_score')
            ->avg('quiz_score');
        
        return [
            [
                'name' => 'First Steps',
                'description' => 'Complete your first topic',
                'icon' => '🎯',
                'earned' => $completedTopics >= 1
            ],
            [
                'name' => 'Bookworm',
                'description' => 'Complete 5 topics',
                'icon' => '📚',
                'earned' => $completedTopics >= 5
            ],
            [
                'name' => 'Rising Star',
                'description' => 'Complete your first course',
                'icon' => '⭐',
                'earned' => $completedCourses >= 1
            ],
            [
                'name' => 'Hot Streak',
                'description' => 'Learn for 7 consecutive days',
                'icon' => '🔥',
                'earned' => $streak >= 7
            ],
            [
                'name' => 'Champion',
                'description' => 'Complete 3 courses',
                'icon' => '🏆',
                'earned' => $completedCourses >= 3
            ],
            [
                'name' => 'Diamond',
                'description' => 'Maintain 90% average quiz score',
                'icon' => '💎',
                'earned' => $avgScore >= 90
            ],
            [
                'name' => 'Rocket',
                'description' => 'Complete 20 topics',
                'icon' => '🚀',
                'earned' => $completedTopics >= 20
            ],
            [
                'name' => 'Master',
                'description' => 'Complete 5 courses',
                'icon' => '👑',
                'earned' => $completedCourses >= 5
            ],
        ];
    }
    
    /**
     * Calculate learning streak
     */
    private function calculateStreak($user)
    {
        $activityDates = UserProgress::where('user_id', $user->id)
            ->selectRaw('DATE(updated_at) as activity_date')
            ->distinct()
            ->orderBy('activity_date', 'desc')
            ->pluck('activity_date')
            ->toArray();

        if (empty($activityDates)) {
            return 0;
        }

        $streak = 1;
        $currentDate = new \DateTime($activityDates[0]);

        for ($i = 1; $i < count($activityDates); $i++) {
            $previousDate = new \DateTime($activityDates[$i]);
            $diff = $currentDate->diff($previousDate)->days;
            
            if ($diff === 1) {
                $streak++;
                $currentDate = $previousDate;
            } else {
                break;
            }
        }

        return $streak;
    }
}
