<?php

namespace App\Http\Controllers;

use App\Models\QuizQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class QuizQuestionController extends Controller
{
    public function index(Request $request, $lesson = null)
    {
        // Support both route parameter and query parameter
        $lessonId = $lesson ?? $request->query('lesson_id');

        if (!$lessonId) {
            return response()->json([
                'success' => false,
                'message' => 'Missing lesson_id'
            ], 400);
        }

        $questions = QuizQuestion::where('lesson_id', $lessonId)
            ->orderBy('order_index')
            ->orderBy('id')
            ->get()
            ->map(function ($question) {
                $options = json_decode($question->options_json, true);
                if (!is_array($options)) {
                    $options = [];
                }

                // Use correct_index if correct_answer is empty
                $correctAnswer = !empty($question->correct_answer) ? $question->correct_answer : $question->correct_index;

                return [
                    'id' => $question->id,
                    'lesson_id' => $question->lesson_id,
                    'question' => $question->question,
                    'options' => $options,
                    'correct_answer' => $correctAnswer,
                    'correct_index' => $question->correct_index,
                    'explanation' => $question->explanation,
                    'order_index' => $question->order_index,
                    'is_active' => $question->is_active,
                    'created_at' => $question->created_at,
                    'updated_at' => $question->updated_at,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $questions
        ]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'lesson_id' => 'required|exists:lessons,id',
            'question' => 'required|string|max:1000',
            'options' => 'required|array|min:2|max:6',
            'options.*' => 'required|string|max:500',
            'correct_answer' => 'required|integer|min:0|max:5',
            'explanation' => 'nullable|string|max:1000',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();
        $data['options_json'] = json_encode($data['options']);
        
        // Store in both correct_index and correct_answer for compatibility
        if (isset($data['correct_answer'])) {
            $data['correct_index'] = $data['correct_answer'];
        }
        
        unset($data['options']);

        // Set default order_index if not provided
        if (!isset($data['order_index'])) {
            $maxOrder = QuizQuestion::where('lesson_id', $data['lesson_id'])->max('order_index') ?? 0;
            $data['order_index'] = $maxOrder + 1;
        }

        $question = QuizQuestion::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Quiz question created successfully',
            'data' => $question
        ], 201);
    }

    public function show(QuizQuestion $quizQuestion)
    {
        $options = json_decode($quizQuestion->options_json, true);
        if (!is_array($options)) {
            $options = [];
        }

        // Use correct_index if correct_answer is empty
        $correctAnswer = !empty($quizQuestion->correct_answer) ? $quizQuestion->correct_answer : $quizQuestion->correct_index;

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $quizQuestion->id,
                'lesson_id' => $quizQuestion->lesson_id,
                'question' => $quizQuestion->question,
                'options' => $options,
                'correct_answer' => $correctAnswer,
                'correct_index' => $quizQuestion->correct_index,
                'explanation' => $quizQuestion->explanation,
                'order_index' => $quizQuestion->order_index,
                'is_active' => $quizQuestion->is_active,
                'created_at' => $quizQuestion->created_at,
                'updated_at' => $quizQuestion->updated_at,
            ]
        ]);
    }

    public function update(Request $request, QuizQuestion $quizQuestion)
    {
        $validator = Validator::make($request->all(), [
            'question' => 'required|string|max:1000',
            'options' => 'required|array|min:2|max:6',
            'options.*' => 'required|string|max:500',
            'correct_answer' => 'required|integer|min:0|max:5',
            'explanation' => 'nullable|string|max:1000',
            'order_index' => 'nullable|integer|min:0',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $data = $request->all();
        $data['options_json'] = json_encode($data['options']);
        
        // Store in both correct_index and correct_answer for compatibility
        if (isset($data['correct_answer'])) {
            $data['correct_index'] = $data['correct_answer'];
        }
        
        unset($data['options']);

        $quizQuestion->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Quiz question updated successfully',
            'data' => $quizQuestion
        ]);
    }

    public function destroy(QuizQuestion $quizQuestion)
    {
        $quizQuestion->delete();

        return response()->json([
            'success' => true,
            'message' => 'Quiz question deleted successfully'
        ]);
    }
}