<?php

namespace App\Http\Controllers;

use App\Models\UserProgress;
use App\Models\UserLessonProgress;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class ProgressController extends Controller
{
    public function updateProgress(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'topic_id' => 'required|exists:topics,id',
            'video_watch_seconds' => 'integer',
            'completed' => 'boolean',
            'quiz_score' => 'nullable|integer|min:0|max:100',
            'attempts' => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $progress = UserProgress::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'topic_id' => $request->topic_id,
                ],
                [
                    'video_watch_seconds' => max(
                        $request->video_watch_seconds ?? 0,
                        UserProgress::where('user_id', $user->id)
                            ->where('topic_id', $request->topic_id)
                            ->value('video_watch_seconds') ?? 0
                    ),
                    'completed' => $request->completed ?? false,
                    'quiz_score' => $request->quiz_score,
                    'attempts' => $request->attempts ?? 0,
                    'last_attempt_at' => now(),
                    'completed_at' => ($request->completed ?? false) ? now() : null,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Progress updated'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating progress: ' . $e->getMessage()
            ], 500);
        }
    }

    public function updateLessonProgress(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $validator = Validator::make($request->all(), [
            'lesson_id' => 'required|exists:lessons,id',
            'completed' => 'boolean',
            'quiz_score' => 'nullable|integer|min:0|max:100',
            'attempts' => 'integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $progress = UserLessonProgress::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'lesson_id' => $request->lesson_id,
                ],
                [
                    'completed' => $request->completed ?? false,
                    'quiz_score' => $request->quiz_score,
                    'attempts' => $request->attempts ?? 0,
                    'last_attempt_at' => now(),
                    'completed_at' => ($request->completed ?? false) ? now() : null,
                ]
            );

            return response()->json([
                'success' => true,
                'message' => 'Lesson progress updated'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating lesson progress: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getProgress(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $topicId = $request->query('topic_id');

        if (!$topicId) {
            return response()->json([
                'success' => false,
                'message' => 'Topic ID required'
            ], 400);
        }

        try {
            $progress = UserProgress::where('user_id', $user->id)
                ->where('topic_id', $topicId)
                ->first();

            return response()->json([
                'success' => true,
                'progress' => $progress
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving progress: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getLessonProgress(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $lessonId = $request->query('lesson_id');

        if (!$lessonId) {
            return response()->json([
                'success' => false,
                'message' => 'Lesson ID required'
            ], 400);
        }

        try {
            $progress = UserLessonProgress::where('user_id', $user->id)
                ->where('lesson_id', $lessonId)
                ->first();

            return response()->json([
                'success' => true,
                'progress' => $progress
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving lesson progress: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getUserProgress(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        try {
            $progress = UserProgress::where('user_id', $user->id)
                ->with(['topic.lesson.course'])
                ->get();

            return response()->json([
                'success' => true,
                'progress' => $progress
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving user progress: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getResumePoint(Request $request)
    {
        $user = Auth::user();

        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Not authenticated'
            ], 401);
        }

        $courseId = $request->query('course_id');

        if (!$courseId) {
            return response()->json([
                'success' => false,
                'message' => 'Course ID required'
            ], 400);
        }

        try {
            // Find the last incomplete topic in the course
            $resumePoint = UserProgress::where('user_id', $user->id)
                ->whereHas('topic.lesson', function ($query) use ($courseId) {
                    $query->where('course_id', $courseId);
                })
                ->where('completed', false)
                ->with(['topic.lesson'])
                ->orderBy('last_attempt_at', 'desc')
                ->first();

            if (!$resumePoint) {
                // If no incomplete topics, find the last completed topic
                $resumePoint = UserProgress::where('user_id', $user->id)
                    ->whereHas('topic.lesson', function ($query) use ($courseId) {
                        $query->where('course_id', $courseId);
                    })
                    ->with(['topic.lesson'])
                    ->orderBy('completed_at', 'desc')
                    ->first();
            }

            return response()->json([
                'success' => true,
                'resume_point' => $resumePoint
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving resume point: ' . $e->getMessage()
            ], 500);
        }
    }
}