<?php

namespace App\Http\Controllers;

use App\Models\CustomPage;
use Illuminate\Http\Request;
use Illuminate\View\View;
use Illuminate\Http\Response;

class PageController extends Controller
{
    /**
     * Display the homepage.
     *
     * @return View
     */
    public function home(): View
    {
        // Try to load home page from CustomPage model
        $page = CustomPage::where('slug', 'home')
                          ->published()
                          ->first();

        // If custom home page exists, use it
        if ($page) {
            // Determine which template to use
            $template = $page->template ?? 'default';
            
            // Check if a custom template view exists
            $viewName = "pages.templates.{$template}";
            if (!view()->exists($viewName)) {
                $viewName = 'pages.templates.default';
            }

            return view($viewName, compact('page'));
        }

        // Fallback to static home view if no custom page found
        // Get featured/popular courses (limit to 3 for homepage)
        $popularCourses = \App\Models\Course::where('is_active', true)
            ->with('lessons')
            ->withCount('lessons')
            ->orderBy('created_at', 'desc')
            ->limit(3)
            ->get();

        return view('pages.home', compact('popularCourses'));
    }

    /**
     * Display the about page.
     *
     * @return View
     */
    public function about(): View
    {
        return view('pages.about');
    }

    /**
     * Display the contact page.
     *
     * @return View
     */
    public function contact(): View
    {
        return view('pages.contact');
    }

    /**
     * Display the terms and conditions page.
     *
     * @return View
     */
    public function terms(): View
    {
        return view('pages.terms');
    }

    /**
     * Display the privacy policy page.
     *
     * @return View
     */
    public function privacy(): View
    {
        return view('pages.privacy');
    }

    /**
     * Display the courses page.
     *
     * @return View
     */
    public function courses(): View
    {
        return view('pages.courses');
    }

    /**
     * Display a custom page by slug.
     *
     * @param string $slug
     * @return View|Response
     */
    public function show(string $slug)
    {
        // Find the page by slug
        $page = CustomPage::where('slug', $slug)
                          ->published()
                          ->firstOrFail();

        // Determine which template to use
        $template = $page->template ?? 'default';
        
        // Check if a custom template view exists
        $viewName = "pages.templates.{$template}";
        if (!view()->exists($viewName)) {
            $viewName = 'pages.templates.default';
        }

        return view($viewName, compact('page'));
    }

    /**
     * Handle contact form submission.
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function submitContact(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'subject' => 'required|string|max:255',
            'message' => 'required|string|max:5000',
        ]);

        // TODO: Implement contact form handling
        // - Send email to admin
        // - Store in database
        // - Send confirmation email to user

        return back()->with('success', 'Thank you for your message! We will get back to you soon.');
    }
}
