<?php

namespace App\Http\Controllers;

use App\Models\Lesson;
use App\Models\QuizQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LessonController extends Controller
{
    public function index(Request $request)
    {
        $courseId = $request->query('course_id');

        if (!$courseId) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid course_id',
                'lessons' => []
            ], 400);
        }

        try {
            $lessons = Lesson::where('course_id', $courseId)
                ->orderBy('order_index')
                ->orderBy('id')
                ->get();

            // Add has_quiz flag to each lesson
            foreach ($lessons as $lesson) {
                $lesson->has_quiz = QuizQuestion::where('lesson_id', $lesson->id)->exists();
            }

            return response()->json([
                'success' => true,
                'message' => 'OK',
                'lessons' => $lessons
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'No lessons available',
                'lessons' => []
            ]);
        }
    }

    public function show($id)
    {
        try {
            $lesson = Lesson::with(['topics' => function($query) {
                $query->orderBy('order_index');
            }, 'course'])->find($id);

            if (!$lesson) {
                return response()->json([
                    'success' => false,
                    'message' => 'Lesson not found'
                ], 404);
            }

            // Load quiz questions for this lesson
            $quizQuestions = QuizQuestion::where('lesson_id', $id)
                ->orderBy('order_index')
                ->orderBy('id')
                ->get()
                ->map(function ($question) {
                    $options = json_decode($question->options_json, true);
                    if (!is_array($options)) {
                        $options = [];
                    }

                    return [
                        'id' => $question->id,
                        'lesson_id' => $question->lesson_id,
                        'question' => $question->question,
                        'options' => $options,
                        'correct_answer' => $question->correct_answer,
                        'explanation' => $question->explanation,
                        'order_index' => $question->order_index,
                        'is_active' => $question->is_active,
                        'created_at' => $question->created_at,
                        'updated_at' => $question->updated_at,
                    ];
                });

            return response()->json([
                'success' => true,
                'lesson' => array_merge($lesson->toArray(), ['quiz_questions' => $quizQuestions]),
                'course' => $lesson->course
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'course_id' => 'required|exists:courses,id',
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'integer',
            'share_enabled' => 'boolean',
            'share_type' => 'nullable|in:upload,link',
            'share_url' => 'nullable|string|max:1024',
            'share_filename' => 'nullable|string',
            'share_size' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $lesson = Lesson::create($request->all());

            return response()->json([
                'success' => true,
                'id' => $lesson->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating lesson: ' . $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request, $id)
    {
        $lesson = Lesson::find($id);

        if (!$lesson) {
            return response()->json([
                'success' => false,
                'message' => 'Lesson not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'course_id' => 'required|exists:courses,id',
            'name' => 'required|string|max:255',
            'intro' => 'nullable|string',
            'order_index' => 'integer',
            'share_enabled' => 'boolean',
            'share_type' => 'nullable|in:upload,link',
            'share_url' => 'nullable|string|max:1024',
            'share_filename' => 'nullable|string',
            'share_size' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            $lesson->update($request->all());

            return response()->json([
                'success' => true,
                'id' => $lesson->id
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating lesson: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $lesson = Lesson::find($id);

        if (!$lesson) {
            return response()->json([
                'success' => false,
                'message' => 'Lesson not found'
            ], 404);
        }

        try {
            $lesson->delete();

            return response()->json([
                'success' => true,
                'message' => 'Lesson deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting lesson: ' . $e->getMessage()
            ], 500);
        }
    }
}