<?php

namespace App\Http\Controllers;

use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class CourseController extends Controller
{
    /**
     * Get all courses for public viewing (no authentication required)
     */
    public function publicIndex(Request $request)
    {
        try {
            $query = Course::where('is_active', true)
                ->with('language')
                ->withCount('lessons');

            // Filter by language if provided
            if ($request->has('language_id') && $request->language_id) {
                $query->where('language_id', $request->language_id);
            }

            $courses = $query->orderBy('created_at', 'desc')->get();

            return response()->json([
                'success' => true,
                'courses' => $courses
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading courses: ' . $e->getMessage()
            ], 500);
        }
    }

    public function index(Request $request)
    {
        try {
            $query = Course::with(['language', 'creator:id,name,email'])
                ->withCount('lessons')
                ->select([
                    'id',
                    'name',
                    'slug',
                    'short_description',
                    'category',
                    'is_active',
                    'is_free',
                    'price',
                    'discount',
                    'discount_type',
                    'currency',
                    'thumbnail_url',
                    'language_id',
                    'created_by',
                    'has_prerequisite',
                    'prerequisite_course_id',
                    'enable_course_survey',
                    'survey_mode',
                    'survey_title',
                    'survey_description',
                    'chunk_words',
                    'created_at',
                    'updated_at',
                ]);

            // Filter by language if provided
            if ($request->has('language_id') && $request->language_id) {
                $query->where('language_id', $request->language_id);
            }

            // Filter by tutor (creator) if provided
            if ($request->has('tutor_id') && $request->tutor_id) {
                $query->where('created_by', $request->tutor_id);
            }

            $courses = $query->orderBy('created_at', 'desc')->get();

            // Add tutor details for frontend convenience
            $courses->transform(function ($course) {
                $course->tutor_name = $course->creator->name ?? null;
                $course->tutor_email = $course->creator->email ?? null;
                return $course;
            });

            // Collect tutors with courses
            $tutorIds = $courses->pluck('created_by')->filter()->unique()->values();
            $tutors = [];
            if ($tutorIds->count()) {
                $tutors = \App\Models\User::whereIn('id', $tutorIds)->get(['id', 'name', 'email']);
            }

            return response()->json([
                'success' => true,
                'courses' => $courses,
                'tutors' => $tutors,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading courses: ' . $e->getMessage()
            ], 500);
        }
    }

    public function create()
    {
        return response()->json(['message' => 'Create course form']);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'short_description' => 'nullable|string',
            'category' => 'nullable|string|max:100',
            'has_prerequisite' => 'boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'certificate_template' => 'nullable|string',
            'completion_instructions' => 'nullable|string',
            'is_active' => 'boolean',
            'chunk_words' => 'nullable|integer',
            'language_id' => 'nullable|exists:languages,id',
            'language' => 'nullable|string|max:10',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            // Handle language code conversion to language_id
            if ($request->has('language') && !$request->has('language_id')) {
                $language = \App\Models\Language::where('code', $request->language)->first();
                if ($language) {
                    $request->merge(['language_id' => $language->id]);
                }
            }
            
            // If no language specified, use default
            if (!$request->language_id) {
                $defaultLanguage = \App\Models\Language::getDefault();
                $request->merge(['language_id' => $defaultLanguage ? $defaultLanguage->id : null]);
            }

            // Get all request data except 'language'
            $data = $request->all();
            unset($data['language']);
            $data['currency'] = isset($data['currency']) ? strtoupper(substr($data['currency'], 0, 3)) : 'USD';
            $data['is_free'] = array_key_exists('is_free', $data) ? filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) !== false : true;
            if (!isset($data['created_by']) && auth()->check()) {
                $data['created_by'] = auth()->id();
            }
            if ($data['is_free']) {
                $data['price'] = 0;
                $data['discount'] = null;
                $data['discount_type'] = 'amount';
            } else {
                $data['is_free'] = false;
                $data['price'] = isset($data['price']) ? (float) $data['price'] : 0;
                $data['discount'] = isset($data['discount']) ? (float) $data['discount'] : null;
                $data['discount_type'] = $data['discount_type'] ?? 'amount';
            }

            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('course-images', 'public');
                $data['image_url'] = Storage::url($path);
            }

            if ($request->hasFile('thumbnail')) {
                $path = $request->file('thumbnail')->store('course-thumbnails', 'public');
                $data['thumbnail_url'] = Storage::url($path);
            }
            
            // Map frontend field names to database field names
            if (isset($data['enable_survey'])) {
                $data['enable_course_survey'] = $data['enable_survey'];
                unset($data['enable_survey']);
            }
            if (isset($data['min_study_minutes_per_day'])) {
                $data['min_study_time_per_day'] = $data['min_study_minutes_per_day'];
                unset($data['min_study_minutes_per_day']);
            }
            if (isset($data['max_study_minutes_per_day'])) {
                $data['max_study_time_per_day'] = $data['max_study_minutes_per_day'];
                unset($data['max_study_minutes_per_day']);
            }
            
            $course = Course::create($data);
            $course->load('language');
            
            // Add language code for frontend compatibility
            $courseArray = $course->toArray();
            if ($course->language) {
                $courseArray['language'] = $course->language->code;
            }

            return response()->json([
                'success' => true,
                'id' => $course->id,
                'course' => $courseArray,
                'message' => 'Course created successfully'
            ]);
        } catch (\Exception $e) {
            \Log::error('Course creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error creating course: ' . $e->getMessage()
            ], 500);
        }
    }

    public function show($id)
    {
        try {
            $course = Course::with(['lessons' => function($query) {
                $query->withCount('topics')->orderBy('order_index');
            }, 'language'])->find($id);

            if (!$course) {
                return response()->json([
                    'success' => false,
                    'message' => 'Course not found'
                ], 404);
            }
            
            // Add language code for frontend compatibility
            $courseArray = $course->toArray();
            if ($course->language) {
                $courseArray['language'] = $course->language->code;
            }

            return response()->json([
                'success' => true,
                'course' => $courseArray
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Server error: ' . $e->getMessage()
            ], 500);
        }
    }

    public function edit($id)
    {
        $course = Course::find($id);

        if (!$course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'course' => $course
        ]);
    }

    public function update(Request $request, $id)
    {
        $course = Course::find($id);

        if (!$course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'short_description' => 'nullable|string',
            'category' => 'nullable|string|max:100',
            'has_prerequisite' => 'boolean',
            'prerequisite_course_id' => 'nullable|exists:courses,id',
            'certificate_template' => 'nullable|string',
            'completion_instructions' => 'nullable|string',
            'is_active' => 'boolean',
            'chunk_words' => 'nullable|integer',
            'language_id' => 'nullable|exists:languages,id',
            'language' => 'nullable|string|max:10',
            'is_free' => 'nullable|boolean',
            'price' => 'nullable|numeric|min:0',
            'discount' => 'nullable|numeric|min:0',
            'discount_type' => 'nullable|in:amount,percent',
            'currency' => 'nullable|string|size:3',
            'image' => 'nullable|image|max:5120',
            'thumbnail' => 'nullable|image|max:5120',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            // Handle language code conversion to language_id
            if ($request->has('language') && !$request->has('language_id')) {
                $language = \App\Models\Language::where('code', $request->language)->first();
                if ($language) {
                    $request->merge(['language_id' => $language->id]);
                }
            }
            
            // Get all request data except 'language'
            $data = $request->all();
            unset($data['language']);
            $data['currency'] = isset($data['currency']) ? strtoupper(substr($data['currency'], 0, 3)) : ($course->currency ?? 'USD');

            if (array_key_exists('is_free', $data)) {
                $isFree = filter_var($data['is_free'], FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE);
                $data['is_free'] = $isFree === null ? true : $isFree;
            } else {
                $data['is_free'] = $course->is_free ?? true;
            }

            if ($data['is_free']) {
                $data['price'] = 0;
                $data['discount'] = null;
                $data['discount_type'] = null;
            } else {
                $data['is_free'] = false;
                $data['price'] = isset($data['price']) ? (float) $data['price'] : ($course->price ?? 0);
                $data['discount'] = isset($data['discount']) ? (float) $data['discount'] : ($course->discount ?? null);
                $data['discount_type'] = $data['discount_type'] ?? ($course->discount_type ?? 'amount');
            }

            if ($request->hasFile('image')) {
                $this->deleteStoredFile($course->image_url);
                $path = $request->file('image')->store('course-images', 'public');
                $data['image_url'] = Storage::url($path);
            }

            if ($request->hasFile('thumbnail')) {
                $this->deleteStoredFile($course->thumbnail_url);
                $path = $request->file('thumbnail')->store('course-thumbnails', 'public');
                $data['thumbnail_url'] = Storage::url($path);
            }
            
            // Map frontend field names to database field names
            if (isset($data['enable_survey'])) {
                $data['enable_course_survey'] = $data['enable_survey'];
                unset($data['enable_survey']);
            }
            if (isset($data['min_study_minutes_per_day'])) {
                $data['min_study_time_per_day'] = $data['min_study_minutes_per_day'];
                unset($data['min_study_minutes_per_day']);
            }
            if (isset($data['max_study_minutes_per_day'])) {
                $data['max_study_time_per_day'] = $data['max_study_minutes_per_day'];
                unset($data['max_study_minutes_per_day']);
            }
            
            $course->update($data);
            $course->load('language');
            
            // Add language code for frontend compatibility
            $courseArray = $course->toArray();
            if ($course->language) {
                $courseArray['language'] = $course->language->code;
            }

            return response()->json([
                'success' => true,
                'id' => $course->id,
                'course' => $courseArray,
                'message' => 'Course updated successfully'
            ]);
        } catch (\Exception $e) {
            \Log::error('Course update failed', [
                'course_id' => $id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);
            
            return response()->json([
                'success' => false,
                'message' => 'Error updating course: ' . $e->getMessage()
            ], 500);
        }
    }

    public function destroy($id)
    {
        $course = Course::find($id);

        if (!$course) {
            return response()->json([
                'success' => false,
                'message' => 'Course not found'
            ], 404);
        }

        try {
            $course->delete();

            return response()->json([
                'success' => true,
                'message' => 'Course deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting course: ' . $e->getMessage()
            ], 500);
        }
    }

    protected function deleteStoredFile(?string $url): void
    {
        if (!$url) {
            return;
        }

        $prefix = '/storage/';
        if (str_starts_with($url, $prefix)) {
            $relativePath = substr($url, strlen($prefix));
            Storage::disk('public')->delete($relativePath);
        }
    }
}
