<?php

namespace App\Http\Controllers;

use App\Models\CertificateTemplate;
use App\Models\CertificateAssignment;
use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use App\Services\CertificateGeneratorService;

class CertificateTemplateController extends Controller
{
    /**
     * Display a listing of certificate templates
     */
    public function index()
    {
        $templates = CertificateTemplate::with('creator')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'templates' => $templates
        ]);
    }

    /**
     * Store a newly created certificate template
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'orientation' => 'required|in:landscape,portrait',
            'layout_config' => 'required|array',
            'text_fields' => 'required|array',
            'qr_code_enabled' => 'sometimes|boolean',
            'qr_code_config' => 'nullable|array',
            'background_image' => 'nullable|image|max:2048',
        ]);

        $data = $request->only(['name', 'description', 'orientation', 'layout_config', 'text_fields', 'qr_code_enabled', 'qr_code_config']);
        $data['created_by'] = Auth::id();
        $data['is_active'] = true;

        // Handle background image upload
        if ($request->hasFile('background_image')) {
            $path = $request->file('background_image')->store('certificates/backgrounds', 'public');
            $data['background_image'] = $path;
        }

        $template = CertificateTemplate::create($data);

        return response()->json([
            'success' => true,
            'message' => 'Certificate template created successfully',
            'template' => $template->load('creator')
        ], 201);
    }

    /**
     * Display the specified certificate template
     */
    public function show($id)
    {
        $template = CertificateTemplate::with(['creator', 'assignments'])->findOrFail($id);

        return response()->json([
            'success' => true,
            'template' => $template
        ]);
    }

    /**
     * Update the specified certificate template
     */
    public function update(Request $request, $id)
    {
        $template = CertificateTemplate::findOrFail($id);

        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'description' => 'nullable|string',
            'orientation' => 'sometimes|required|in:landscape,portrait',
            'layout_config' => 'sometimes|required|array',
            'text_fields' => 'sometimes|required|array',
            'qr_code_enabled' => 'sometimes|boolean',
            'qr_code_config' => 'nullable|array',
            'background_image' => 'nullable|image|max:2048',
            'is_active' => 'sometimes|boolean',
        ]);

        $data = $request->only(['name', 'description', 'orientation', 'layout_config', 'text_fields', 'qr_code_enabled', 'qr_code_config', 'is_active']);

        // Handle background image upload
        if ($request->hasFile('background_image')) {
            // Delete old image if exists
            if ($template->background_image) {
                Storage::disk('public')->delete($template->background_image);
            }
            
            $path = $request->file('background_image')->store('certificates/backgrounds', 'public');
            $data['background_image'] = $path;
        }

        $template->update($data);

        return response()->json([
            'success' => true,
            'message' => 'Certificate template updated successfully',
            'template' => $template->load('creator')
        ]);
    }

    /**
     * Remove the specified certificate template
     */
    public function destroy($id)
    {
        $template = CertificateTemplate::findOrFail($id);

        // Delete background image if exists
        if ($template->background_image) {
            Storage::disk('public')->delete($template->background_image);
        }

        $template->delete();

        return response()->json([
            'success' => true,
            'message' => 'Certificate template deleted successfully'
        ]);
    }

    /**
     * Assign certificate template to a course or lesson
     */
    public function assign(Request $request)
    {
        $request->validate([
            'certificate_template_id' => 'required|exists:certificate_templates,id',
            'assignable_type' => 'required|in:App\Models\Course,App\Models\Lesson',
            'assignable_id' => 'required|integer',
        ]);

        // Check if assignment already exists
        $existing = CertificateAssignment::where([
            'assignable_type' => $request->assignable_type,
            'assignable_id' => $request->assignable_id,
        ])->first();

        if ($existing) {
            // Update existing assignment
            $existing->update([
                'certificate_template_id' => $request->certificate_template_id
            ]);
            $assignment = $existing;
        } else {
            // Create new assignment
            $assignment = CertificateAssignment::create($request->all());
        }

        return response()->json([
            'success' => true,
            'message' => 'Certificate template assigned successfully',
            'assignment' => $assignment->load('template')
        ]);
    }

    /**
     * Remove certificate assignment
     */
    public function unassign($id)
    {
        $assignment = CertificateAssignment::findOrFail($id);
        $assignment->delete();

        return response()->json([
            'success' => true,
            'message' => 'Certificate template unassigned successfully'
        ]);
    }

    /**
     * Get certificate assignment for a course or lesson
     */
    public function getAssignment(Request $request)
    {
        $request->validate([
            'assignable_type' => 'required|in:App\Models\Course,App\Models\Lesson',
            'assignable_id' => 'required|integer',
        ]);

        $assignment = CertificateAssignment::with('template')
            ->where([
                'assignable_type' => $request->assignable_type,
                'assignable_id' => $request->assignable_id,
            ])
            ->first();

        return response()->json([
            'success' => true,
            'assignment' => $assignment
        ]);
    }

    /**
     * Duplicate a certificate template
     */
    public function duplicate($id)
    {
        $original = CertificateTemplate::findOrFail($id);

        $duplicate = $original->replicate();
        $duplicate->name = $original->name . ' (Copy)';
        $duplicate->created_by = Auth::id();
        $duplicate->save();

        return response()->json([
            'success' => true,
            'message' => 'Certificate template duplicated successfully',
            'template' => $duplicate->load('creator')
        ]);
    }

    /**
     * Preview the specified certificate template as HTML
     */
    public function preview($id, CertificateGeneratorService $generator)
    {
        $template = CertificateTemplate::findOrFail($id);
        $html = $generator->buildPreviewHTML($template);
        return response($html)->header('Content-Type', 'text/html');
    }
}

