<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\AccessCode;
use App\Providers\RouteServiceProvider;
use Illuminate\Auth\Events\Registered;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rules;
// use Inertia\Inertia;
// use Inertia\Response;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): \Illuminate\View\View
    {
        $accessCodeRequired = false;

        try {
            $accessCodeRequired = DB::table('settings')
                ->where('access_code_required', 1)
                ->exists();
        } catch (\Exception $e) {
            $accessCodeRequired = false;
        }

        return view('auth.combined', [
            'accessCodeRequired' => $accessCodeRequired,
        ]);
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        // Check if access codes are required
        $accessCodeRequired = false;
        try {
            $setting = DB::table('settings')
                ->where('access_code_required', 1)
                ->exists();
            $accessCodeRequired = $setting;
        } catch (\Exception $e) {
            // If settings table doesn't exist or other error, assume not required
            $accessCodeRequired = false;
        }

        // Base validation rules
        $rules = [
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:'.User::class],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
        ];

        // Add access code validation if required
        if ($accessCodeRequired) {
            $rules['access_code'] = ['required', 'string'];
        }

        $request->validate($rules);

        // Validate access code if required
        if ($accessCodeRequired) {
            $accessCode = AccessCode::where('code', strtoupper($request->access_code))
                ->where('is_used', false)
                ->where(function ($query) {
                    $query->whereNull('expires_at')
                        ->orWhere('expires_at', '>', now());
                })
                ->whereNull('used_by')
                ->first();

            if (!$accessCode) {
                return back()->withErrors([
                    'access_code' => 'Invalid or expired access code'
                ])->withInput();
            }
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        // Update access code with user info if it was used
        if ($accessCodeRequired && isset($accessCode)) {
            $accessCode->update([
                'is_used' => true,
                'used_by' => $user->id
            ]);
        }

        // Assign default 'user' role
        $user->assignRole('user');

        event(new Registered($user));

        Auth::login($user);

        return redirect(RouteServiceProvider::HOME);
    }
}