<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\Auth;

class RoleController extends Controller
{
    public function index()
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $roles = Role::with('permissions')->withCount('users')->get()->map(function ($role) {
            return [
                'id' => $role->id,
                'name' => $role->name,
                'guard_name' => $role->guard_name,
                'permissions' => $role->permissions,
                'users_count' => $role->users_count ?? 0,
                'created_at' => $role->created_at,
                'updated_at' => $role->updated_at
            ];
        });
        
        return response()->json(['success' => true, 'roles' => $roles]);
    }

    public function show($id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $role = Role::with('permissions')->find($id);
        if (!$role) return response()->json(['success' => false, 'message' => 'Role not found'], 404);

        return response()->json(['success' => true, 'role' => $role]);
    }

    public function store(Request $request)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $request->validate(['name' => 'required|string|max:50|unique:roles,name']);

        $role = Role::create(['name' => $request->input('name')]);

        // Optionally sync permissions by ID
        if ($request->filled('permissions')) {
            $permissionIds = $request->input('permissions');
            $permissionNames = Permission::whereIn('id', $permissionIds)->pluck('name')->toArray();
            $role->syncPermissions($permissionNames);
        }

        return response()->json(['success' => true, 'role' => $role->load('permissions')], 201);
    }

    public function update(Request $request, $id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $role = Role::find($id);
        if (!$role) return response()->json(['success' => false, 'message' => 'Role not found'], 404);

        $request->validate(['name' => 'required|string|max:50|unique:roles,name,' . $id]);
        $role->name = $request->input('name');
        $role->save();

        if ($request->filled('permissions')) {
            $permissionIds = $request->input('permissions');
            $permissionNames = Permission::whereIn('id', $permissionIds)->pluck('name')->toArray();
            $role->syncPermissions($permissionNames);
        }

        return response()->json(['success' => true, 'role' => $role->load('permissions')]);
    }

    public function destroy($id)
    {
        if (!Auth::user() || !Auth::user()->is_admin) {
            return response()->json(['success' => false, 'message' => 'Admin access required'], 403);
        }

        $role = Role::find($id);
        if (!$role) return response()->json(['success' => false, 'message' => 'Role not found'], 404);

        // Prevent deleting core roles
        if (in_array($role->name, ['admin', 'user', 'tutor'])) {
            return response()->json(['success' => false, 'message' => 'Cannot delete system role'], 400);
        }

        $role->delete();
        return response()->json(['success' => true]);
    }
}
