<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Badge;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

class AdminBadgeController extends Controller
{
    /**
     * Get all badges with statistics
     */
    public function index()
    {
        $badges = Badge::withCount('users')->orderBy('order')->get();

        $stats = [
            'total_badges' => $badges->count(),
            'active_badges' => $badges->where('is_active', true)->count(),
            'total_points' => $badges->sum('points'),
            'total_awarded' => DB::table('user_badges')->count(),
        ];

        return response()->json([
            'success' => true,
            'badges' => $badges,
            'stats' => $stats
        ]);
    }

    /**
     * Create a new badge
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:badges,name',
            'description' => 'required|string',
            'icon' => 'required|string|max:10',
            'color' => 'required|string|max:20',
            'category' => 'required|string|in:course,lesson,quiz,speed,special,ee_specific',
            'points' => 'required|integer|min:0',
            'criteria' => 'required|array',
            'criteria.type' => 'required|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        // Get the next order number
        $maxOrder = Badge::max('order') ?? 0;

        $badge = Badge::create([
            'name' => $request->name,
            'description' => $request->description,
            'icon' => $request->icon,
            'type' => 'achievement', // default type
            'category' => $request->category,
            'criteria' => $request->criteria,
            'points' => $request->points,
            'color' => $request->color,
            'is_active' => $request->is_active ?? true,
            'order' => $maxOrder + 1
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Badge created successfully!',
            'badge' => $badge
        ], 201);
    }

    /**
     * Update a badge
     */
    public function update(Request $request, $id)
    {
        $badge = Badge::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255|unique:badges,name,' . $id,
            'description' => 'required|string',
            'icon' => 'required|string|max:10',
            'color' => 'required|string|max:20',
            'category' => 'required|string|in:course,lesson,quiz,speed,special,ee_specific',
            'points' => 'required|integer|min:0',
            'criteria' => 'required|array',
            'criteria.type' => 'required|string',
            'is_active' => 'boolean'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $badge->update($request->all());

        return response()->json([
            'success' => true,
            'message' => 'Badge updated successfully!',
            'badge' => $badge
        ]);
    }

    /**
     * Toggle badge active status
     */
    public function toggleStatus(Request $request, $id)
    {
        $badge = Badge::findOrFail($id);

        $badge->update([
            'is_active' => $request->is_active
        ]);

        return response()->json([
            'success' => true,
            'message' => $badge->is_active ? 'Badge activated!' : 'Badge deactivated!',
            'badge' => $badge
        ]);
    }

    /**
     * Delete a badge
     */
    public function destroy($id)
    {
        $badge = Badge::findOrFail($id);

        // Detach from all users
        $badge->users()->detach();

        // Delete the badge
        $badge->delete();

        return response()->json([
            'success' => true,
            'message' => 'Badge deleted successfully!'
        ]);
    }

    /**
     * Get leaderboard - top users by badge points
     */
    public function leaderboard()
    {
        $leaderboard = User::select('users.id', 'users.name', 'users.email')
            ->join('user_badges', 'users.id', '=', 'user_badges.user_id')
            ->join('badges', 'user_badges.badge_id', '=', 'badges.id')
            ->groupBy('users.id', 'users.name', 'users.email')
            ->selectRaw('COUNT(DISTINCT badges.id) as badge_count')
            ->selectRaw('SUM(badges.points) as total_points')
            ->orderByDesc('total_points')
            ->limit(50)
            ->get();

        return response()->json([
            'success' => true,
            'leaderboard' => $leaderboard
        ]);
    }

    /**
     * Get badge statistics
     */
    public function stats()
    {
        $totalBadges = Badge::count();
        $activeBadges = Badge::where('is_active', true)->count();
        $totalPoints = Badge::sum('points');
        $totalAwarded = DB::table('user_badges')->count();
        $uniqueRecipients = DB::table('user_badges')->distinct('user_id')->count('user_id');

        return response()->json([
            'success' => true,
            'stats' => [
                'total_badges' => $totalBadges,
                'active_badges' => $activeBadges,
                'inactive_badges' => $totalBadges - $activeBadges,
                'total_points' => $totalPoints,
                'total_awarded' => $totalAwarded,
                'unique_recipients' => $uniqueRecipients,
                'average_per_user' => $uniqueRecipients > 0 ? round($totalAwarded / $uniqueRecipients, 2) : 0
            ]
        ]);
    }

    /**
     * Award badge to specific user (manual)
     */
    public function awardBadge(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'badge_id' => 'required|exists:badges,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($request->user_id);
        $badge = Badge::findOrFail($request->badge_id);

        if ($user->hasBadge($badge->id)) {
            return response()->json([
                'success' => false,
                'message' => 'User already has this badge'
            ], 400);
        }

        $user->awardBadge($badge->id, ['awarded_by' => 'admin']);

        return response()->json([
            'success' => true,
            'message' => "Badge '{$badge->name}' awarded to {$user->name}!"
        ]);
    }

    /**
     * Remove badge from user
     */
    public function removeBadge(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_id' => 'required|exists:users,id',
            'badge_id' => 'required|exists:badges,id'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        $user = User::findOrFail($request->user_id);
        $badge = Badge::findOrFail($request->badge_id);

        $user->badges()->detach($badge->id);

        return response()->json([
            'success' => true,
            'message' => "Badge '{$badge->name}' removed from {$user->name}!"
        ]);
    }
}
