<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SiteSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Artisan;

class SiteSettingsController extends Controller
{
    /**
     * Get all settings grouped by category
     */
    public function index()
    {
        try {
            $settings = SiteSetting::getGrouped();
            
            return response()->json([
                'success' => true,
                'settings' => $settings
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a single setting
     */
    public function update(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'key' => 'required|string',
            'value' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $setting = SiteSetting::where('key', $request->key)->first();
            
            if (!$setting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Setting not found'
                ], 404);
            }

            $setting->value = $request->value;
            $setting->save();

            // Clear cache
            SiteSetting::clearCache();

            return response()->json([
                'success' => true,
                'message' => 'Setting updated successfully',
                'setting' => $setting
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update setting: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update multiple settings at once
     */
    public function updateBulk(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'settings' => 'required|array',
            'settings.*.key' => 'required|string',
            'settings.*.value' => 'nullable',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            foreach ($request->settings as $settingData) {
                $setting = SiteSetting::where('key', $settingData['key'])->first();
                if ($setting) {
                    $setting->value = $settingData['value'] ?? '';
                    $setting->save();
                }
            }

            // Clear cache
            SiteSetting::clearCache();

            return response()->json([
                'success' => true,
                'message' => 'Settings updated successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Upload logo/favicon/image
     */
    public function uploadImage(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'key' => 'required|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg,ico|max:2048'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            $setting = SiteSetting::where('key', $request->key)->first();
            
            if (!$setting) {
                return response()->json([
                    'success' => false,
                    'message' => 'Setting not found'
                ], 404);
            }

            // Delete old image if exists
            if ($setting->value && Storage::disk('public')->exists($setting->value)) {
                Storage::disk('public')->delete($setting->value);
            }

            // Store new image
            $image = $request->file('image');
            $filename = $request->key . '_' . time() . '.' . $image->getClientOriginalExtension();
            $path = $image->storeAs('site-images', $filename, 'public');

            // Update setting
            $setting->value = '/storage/' . $path;
            $setting->save();

            // Clear cache
            SiteSetting::clearCache();

            return response()->json([
                'success' => true,
                'message' => 'Image uploaded successfully',
                'path' => $setting->value
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to upload image: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reset settings to default
     */
    public function reset(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'group' => 'nullable|string'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            // Run seeder to reset settings
            Artisan::call('db:seed', ['--class' => 'SiteSettingsSeeder', '--force' => true]);

            // Clear cache
            SiteSetting::clearCache();

            return response()->json([
                'success' => true,
                'message' => 'Settings reset to default values'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to reset settings: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get public settings (for frontend)
     */
    public function getPublic()
    {
        try {
            $publicKeys = [
                'site_name',
                'site_description',
                'copyright_text',
                'logo_path',
                'favicon_path',
                'primary_color',
                'secondary_color',
                'contact_email',
                'contact_phone',
                'facebook_url',
                'twitter_url',
                'linkedin_url',
                'instagram_url',
                'youtube_url',
            ];

            $settings = SiteSetting::whereIn('key', $publicKeys)->pluck('value', 'key');

            // Add current year for copyright
            $settings['current_year'] = date('Y');

            return response()->json([
                'success' => true,
                'settings' => $settings
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to load settings'
            ], 500);
        }
    }
}
