<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Log;

class PaymentSettingsController extends Controller
{
    /**
     * Display payment settings page
     */
    public function index(Request $request)
    {
        $gateways = [
            'stripe' => 'Stripe',
            'paypal' => 'PayPal',
            'paystack' => 'Paystack',
            'crypto' => 'Cryptocurrency'
        ];

        $settings = [];
        foreach ($gateways as $key => $name) {
            $setting = PaymentSetting::where('gateway', $key)->first();
            if (!$setting) {
                $setting = PaymentSetting::create([
                    'gateway' => $key,
                    'is_enabled' => false,
                    'mode' => 'sandbox'
                ]);
            }
            $settings[$key] = $setting;
        }

        // If it's an AJAX request, return just the content
        if ($request->ajax() || $request->wantsJson()) {
            return view('admin.payment-settings-content', compact('settings', 'gateways'));
        }

        return view('admin.payment-settings', compact('settings', 'gateways'));
    }

    /**
     * Update payment gateway settings
     */
    public function update(Request $request, $gateway)
    {
        $validated = $request->validate([
            'is_enabled' => 'required|boolean',
            'public_key' => 'nullable|string',
            'secret_key' => 'nullable|string',
            'webhook_secret' => 'nullable|string',
            'mode' => 'required|in:sandbox,live',
            'additional_config' => 'nullable|array'
        ]);

        $setting = PaymentSetting::where('gateway', $gateway)->firstOrFail();

        $data = [
            'is_enabled' => $validated['is_enabled'],
            'mode' => $validated['mode'],
            'additional_config' => $validated['additional_config'] ?? null
        ];

        // Update keys - encrypt if provided, set to null if empty
        if (array_key_exists('public_key', $validated)) {
            $data['public_key'] = $validated['public_key'] ? Crypt::encryptString($validated['public_key']) : null;
        }
        if (array_key_exists('secret_key', $validated)) {
            $data['secret_key'] = $validated['secret_key'] ? Crypt::encryptString($validated['secret_key']) : null;
        }
        if (array_key_exists('webhook_secret', $validated)) {
            $data['webhook_secret'] = $validated['webhook_secret'] ? Crypt::encryptString($validated['webhook_secret']) : null;
        }

        $setting->update($data);

        return response()->json([
            'success' => true,
            'message' => ucfirst($gateway) . ' settings updated successfully!',
            'data' => [
                'is_enabled' => $setting->is_enabled,
                'mode' => $setting->mode
            ]
        ]);
    }

    /**
     * Toggle gateway status
     */
    public function toggle(Request $request, $gateway)
    {
        $setting = PaymentSetting::where('gateway', $gateway)->firstOrFail();
        $newStatus = !$setting->is_enabled;
        $setting->update(['is_enabled' => $newStatus]);

        Log::info("Gateway {$gateway} toggled to: " . ($newStatus ? 'enabled' : 'disabled'));

        return response()->json([
            'success' => true,
            'is_enabled' => $newStatus,
            'message' => ucfirst($gateway) . ' ' . ($newStatus ? 'enabled' : 'disabled') . ' successfully!'
        ]);
    }

    /**
     * Get decrypted settings for a gateway
     */
    public function show($gateway)
    {
        $setting = PaymentSetting::where('gateway', $gateway)->firstOrFail();
        
        $data = $setting->toArray();
        
        // Decrypt sensitive fields for display (masked)
        if ($setting->public_key) {
            $decrypted = Crypt::decryptString($setting->public_key);
            $data['public_key_masked'] = $this->maskKey($decrypted);
        }
        if ($setting->secret_key) {
            $decrypted = Crypt::decryptString($setting->secret_key);
            $data['secret_key_masked'] = $this->maskKey($decrypted);
        }
        if ($setting->webhook_secret) {
            $decrypted = Crypt::decryptString($setting->webhook_secret);
            $data['webhook_secret_masked'] = $this->maskKey($decrypted);
        }

        return response()->json($data);
    }

    /**
     * Mask API key for display
     */
    private function maskKey($key)
    {
        if (strlen($key) <= 8) {
            return str_repeat('*', strlen($key));
        }
        return substr($key, 0, 4) . str_repeat('*', strlen($key) - 8) . substr($key, -4);
    }
}
