<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LanguageController extends Controller
{
    /**
     * Get all languages
     */
    public function index()
    {
        try {
            $languages = Language::withCount([
                'courses',
                'courses as active_courses_count' => function ($query) {
                    $query->where('is_active', true);
                }
            ])->ordered()->get();

            return response()->json([
                'success' => true,
                'languages' => $languages
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading languages: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get active languages (public endpoint)
     */
    public function getActive()
    {
        try {
            $languages = Language::active()->ordered()->get();

            return response()->json([
                'success' => true,
                'languages' => $languages
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading languages: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Store a new language
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'code' => 'required|string|max:10|unique:languages,code',
            'native_name' => 'nullable|string|max:100',
            'flag_emoji' => 'nullable|string|max:10',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
            'order_index' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            // If this is set as default, unset other defaults
            if ($request->is_default) {
                Language::where('is_default', true)->update(['is_default' => false]);
            }

            $language = Language::create($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Language created successfully',
                'language' => $language
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error creating language: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get a specific language
     */
    public function show($id)
    {
        try {
            $language = Language::withCount('courses')->find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            return response()->json([
                'success' => true,
                'language' => $language
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading language: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a language
     */
    public function update(Request $request, $id)
    {
        $language = Language::find($id);

        if (!$language) {
            return response()->json([
                'success' => false,
                'message' => 'Language not found'
            ], 404);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:100',
            'code' => 'required|string|max:10|unique:languages,code,' . $id,
            'native_name' => 'nullable|string|max:100',
            'flag_emoji' => 'nullable|string|max:10',
            'is_active' => 'boolean',
            'is_default' => 'boolean',
            'order_index' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            // If this is set as default, unset other defaults
            if ($request->is_default) {
                Language::where('id', '!=', $id)
                    ->where('is_default', true)
                    ->update(['is_default' => false]);
            }

            $language->update($request->all());

            return response()->json([
                'success' => true,
                'message' => 'Language updated successfully',
                'language' => $language
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating language: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle language active status
     */
    public function toggle($id)
    {
        try {
            $language = Language::find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            // Prevent disabling default language
            if ($language->is_default && $language->is_active) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot disable the default language'
                ], 400);
            }

            $language->is_active = !$language->is_active;
            $language->save();

            return response()->json([
                'success' => true,
                'message' => 'Language status updated',
                'language' => $language
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error toggling language: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a language
     */
    public function destroy($id)
    {
        try {
            $language = Language::find($id);

            if (!$language) {
                return response()->json([
                    'success' => false,
                    'message' => 'Language not found'
                ], 404);
            }

            // Prevent deleting default language
            if ($language->is_default) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete the default language'
                ], 400);
            }

            // Check if language has courses
            if ($language->courses()->count() > 0) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot delete language with associated courses. Please reassign or delete courses first.'
                ], 400);
            }

            $language->delete();

            return response()->json([
                'success' => true,
                'message' => 'Language deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error deleting language: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder languages
     */
    public function reorder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'languages' => 'required|array',
            'languages.*.id' => 'required|exists:languages,id',
            'languages.*.order_index' => 'required|integer',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        try {
            foreach ($request->languages as $languageData) {
                Language::where('id', $languageData['id'])
                    ->update(['order_index' => $languageData['order_index']]);
            }

            return response()->json([
                'success' => true,
                'message' => 'Languages reordered successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error reordering languages: ' . $e->getMessage()
            ], 500);
        }
    }
}
